<?php

namespace Modules\Chat\Console\Commands;

use Illuminate\Console\Command;
use Modules\Chat\Services\ChatRedisService;

class ChatRedisCommand extends Command
{
    protected $signature = 'chat:redis {action=stats : stats|cleanup|health}';
    protected $description = 'Gerenciar Redis do Chat - integrado ao ConcordCRM';

    protected $chatRedis;

    public function __construct(ChatRedisService $chatRedis)
    {
        parent::__construct();
        $this->chatRedis = $chatRedis;
    }

    public function handle()
    {
        $action = $this->argument('action');

        switch ($action) {
            case 'stats':
                $this->showStats();
                break;
            case 'cleanup':
                $this->runCleanup();
                break;
            case 'health':
                $this->checkHealth();
                break;
            default:
                $this->error('Ação inválida. Use: stats, cleanup ou health');
                return 1;
        }

        return 0;
    }

    protected function showStats()
    {
        $this->info('📊 Estatísticas do Redis Chat');
        $this->line('');

        try {
            $stats = $this->chatRedis->getStats();

            $this->table(
                ['Métrica', 'Valor'],
                [
                    ['Mensagens Processadas', $stats['processed'] ?? 0],
                    ['Locks Ativos', $stats['locks'] ?? 0],
                    ['Usuários Digitando', $stats['typing'] ?? 0],
                    ['Contadores Não Lidas', $stats['unread'] ?? 0],
                    ['Últimas Mensagens Cache', $stats['last_messages'] ?? 0],
                    ['Uso de Memória', $stats['memory_usage'] ?? 'N/A'],
                    ['Clientes Conectados', $stats['connected_clients'] ?? 0],
                    ['Comandos Processados', number_format($stats['total_commands_processed'] ?? 0)]
                ]
            );
        } catch (\Exception $e) {
            $this->error('Erro ao obter estatísticas: ' . $e->getMessage());
        }
    }

    protected function runCleanup()
    {
        $this->info('🧹 Iniciando limpeza do Redis Chat...');

        try {
            $cleaned = $this->chatRedis->cleanup();

            $this->table(
                ['Tipo', 'Itens Limpos'],
                [
                    ['Locks Expirados', $cleaned['locks'] ?? 0],
                    ['Indicadores de Digitação', $cleaned['typing'] ?? 0],
                    ['Dados Processados', $cleaned['processed'] ?? 0],
                    ['Estatísticas', $cleaned['stats'] ?? 0]
                ]
            );

            $this->info('✅ Limpeza concluída!');
        } catch (\Exception $e) {
            $this->error('Erro na limpeza: ' . $e->getMessage());
        }
    }

    protected function checkHealth()
    {
        $this->info('🏥 Verificando saúde do Redis Chat...');

        try {
            $health = $this->chatRedis->healthCheck();

            if ($health['status'] === 'healthy') {
                $this->info('✅ Redis Chat está saudável');
                $this->line("Ping: {$health['ping']}");
                $this->line("Latência: {$health['latency_ms']}ms");
            } else {
                $this->error('❌ Redis Chat com problemas');
                $this->line("Erro: {$health['error']}");
            }
        } catch (\Exception $e) {
            $this->error('Erro no health check: ' . $e->getMessage());
        }
    }
}
