<?php

namespace Modules\Chat\Console\Commands;

use Illuminate\Console\Command;
use Modules\Chat\Events\MessageSent;
use Modules\Chat\Models\Message;
use Modules\Chat\Models\Conversation;

class TestWebSocket extends Command
{
    protected $signature = 'chat:test-websocket {conversation_id}';
    protected $description = 'Testar WebSocket com Pusher';

    public function handle()
    {
        $conversationId = $this->argument('conversation_id');
        
        $conversation = Conversation::find($conversationId);
        if (!$conversation) {
            $this->error("Conversa {$conversationId} não encontrada");
            return 1;
        }

        $this->info('🔧 Verificando configuração...');
        
        $pusherConfig = config('broadcasting.connections.pusher');
        $this->info('Pusher App ID: ' . $pusherConfig['app_id']);
        $this->info('Pusher Key: ' . substr($pusherConfig['key'], 0, 10) . '...');
        $this->info('Pusher Cluster: ' . $pusherConfig['options']['cluster']);
        
        $this->info('📝 Criando mensagem de teste...');
        
        try {
            $message = Message::create([
                'conversation_id' => $conversationId,
                'sender_type' => 'system',
                'sender_id' => 1,
                'content' => 'Mensagem de teste WebSocket - ' . now(),
                'message_type' => 'text',
                'sent_at' => now(),
                'is_from_contact' => true,
                'status' => 'sent',
                'metadata' => [
                    'sender_name' => 'Sistema de Teste'
                ]
            ]);

            $this->info("✅ Mensagem criada no banco (ID: {$message->id})");
            $this->info('📡 Enviando via WebSocket...');
            
            broadcast(new MessageSent($message, $conversation))->toOthers();
            
            $this->info('🎉 Mensagem enviada com sucesso!');
            $this->info('🔍 Verifique:');
            $this->info('   1. Console do navegador');
            $this->info('   2. Dashboard Pusher: https://dashboard.pusher.com/');
            $this->info('   3. Logs: tail -f storage/logs/laravel.log');
            
            return 0;
            
        } catch (\Exception $e) {
            $this->error('❌ Erro: ' . $e->getMessage());
            
            if (str_contains($e->getMessage(), 'cURL error')) {
                $this->error('💡 Verifique conectividade com Pusher');
            }
            if (str_contains($e->getMessage(), 'authentication')) {
                $this->error('💡 Verifique credenciais no .env');
            }
            
            return 1;
        }
    }
}
