<?php

namespace Modules\Chat\Events;

use Illuminate\Broadcasting\Channel;
use Illuminate\Broadcasting\InteractsWithSockets;
use Illuminate\Broadcasting\PresenceChannel;
use Illuminate\Broadcasting\PrivateChannel;
use Illuminate\Contracts\Broadcasting\ShouldBroadcast;
use Illuminate\Foundation\Events\Dispatchable;
use Illuminate\Queue\SerializesModels;
use Modules\Chat\Models\Message;

class MessageSent implements ShouldBroadcast
{
    use Dispatchable, InteractsWithSockets, SerializesModels;

    public $message;

    public function __construct(Message $message)
    {
        $this->message = $message;
        
        // Log para debug
        \Log::info('🚀 MessageSent event criado', [
            'message_id' => $message->id,
            'conversation_id' => $message->conversation_id,
            'content' => $message->content,
            'class' => self::class
        ]);
    }

    public function broadcastOn()
    {
        $channels = [
            new Channel('chat'), // Canal global
            new Channel('chat.conversation.' . $this->message->conversation_id) // Canal específico
        ];
        
        \Log::info('📡 Broadcasting nos canais:', [
            'channels' => ['chat', 'chat.conversation.' . $this->message->conversation_id],
            'message_id' => $this->message->id,
            'event_class' => self::class
        ]);
        
        return $channels;
    }

    public function broadcastWith()
    {
        $data = [
            'message' => [
                'id' => $this->message->id,
                'conversation_id' => $this->message->conversation_id,
                'content' => $this->message->content,
                'content_type' => $this->message->content_type ?? 'text',
                'is_from_contact' => $this->message->is_from_contact,
                'sender_name' => $this->message->sender_name ?? ($this->message->is_from_contact ? 'Contato' : 'Você'),
                'created_at' => $this->message->created_at->toISOString(),
                'status' => $this->message->status ?? 'sent',
                'metadata' => $this->message->metadata ?? []
            ],
            'conversation_id' => $this->message->conversation_id,
            'event_info' => [
                'class' => self::class,
                'timestamp' => now()->toISOString()
            ]
        ];
        
        \Log::info('📤 Dados sendo enviados via broadcast:', $data);
        
        return $data;
    }

    public function broadcastAs()
    {
        $eventName = 'MessageSent';
        \Log::info('�� Nome do evento broadcast:', [
            'event_name' => $eventName,
            'full_class' => self::class
        ]);
        return $eventName;
    }
}
