<?php

namespace Modules\Chat\Http\Controllers\Api;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Modules\Chat\Models\ChatChannel;
use Modules\Chat\Models\EvolutionInstance;
use Modules\Chat\Services\EvolutionSyncService;
use Modules\Core\Http\Controllers\ApiController;
use Modules\Chat\Services\EvolutionApiService;
use Exception;

class ChatChannelController extends ApiController
{
    protected EvolutionSyncService $evolutionSyncService;

    public function __construct(EvolutionSyncService $evolutionSyncService)
    {
        $this->evolutionSyncService = $evolutionSyncService;
    }

    /**
     * 📋 LISTAR TODOS OS CANAIS
     */
    public function index(Request $request): JsonResponse
    {
        try {
            Log::info('📋 Listando canais de chat', [
                'user_id' => auth()->id(),
                'filters' => $request->all()
            ]);

            $query = ChatChannel::query();

            // ✅ FILTROS
            if ($request->filled('type')) {
                $query->where('type', $request->input('type'));
            }

            if ($request->filled('is_active')) {
                $query->where('is_active', $request->boolean('is_active'));
            }

            if ($request->filled('search')) {
                $search = $request->input('search');
                $query->where(function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                        ->orWhere('description', 'like', "%{$search}%");
                });
            }

            // ✅ ORDENAÇÃO
            $sortBy = $request->input('sort_by', 'priority');
            $sortDirection = $request->input('sort_direction', 'asc');

            if (in_array($sortBy, ['name', 'type', 'priority', 'created_at', 'updated_at'])) {
                $query->orderBy($sortBy, $sortDirection);
            }

            // ✅ PAGINAÇÃO OU TODOS
            if ($request->boolean('paginate', true)) {
                $perPage = $request->input('per_page', 15);
                $channels = $query->paginate($perPage);
            } else {
                $channels = $query->get();
            }

            // ✅ INCLUIR ESTATÍSTICAS SE SOLICITADO
            if ($request->boolean('include_stats')) {
                if ($channels instanceof \Illuminate\Pagination\LengthAwarePaginator) {
                    $channels->getCollection()->transform(function ($channel) {
                        return $this->addChannelStats($channel);
                    });
                } else {
                    $channels = $channels->map(function ($channel) {
                        return $this->addChannelStats($channel);
                    });
                }
            }

            Log::info('✅ Canais listados com sucesso', [
                'total' => $channels instanceof \Illuminate\Pagination\LengthAwarePaginator
                    ? $channels->total()
                    : $channels->count()
            ]);

            return $this->response($channels);
        } catch (\Exception $e) {
            Log::error('❌ Erro ao listar canais', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'user_id' => auth()->id()
            ]);

            return $this->response([
                'error' => 'Erro interno do servidor'
            ], 500);
        }
    }

    /**
     * 📄 EXIBIR CANAL ESPECÍFICO
     */
    public function show(Request $request, $channelId): JsonResponse
    {
        try {
            Log::info('📄 Buscando canal específico', [
                'channel_id' => $channelId,
                'user_id' => auth()->id()
            ]);

            $channel = ChatChannel::findOrFail($channelId);

            // ✅ INCLUIR ESTATÍSTICAS SE SOLICITADO
            if ($request->boolean('include_stats')) {
                $channel = $this->addChannelStats($channel);
            }

            // ✅ INCLUIR CONVERSAS RECENTES SE SOLICITADO
            if ($request->boolean('include_conversations')) {
                $limit = $request->input('conversations_limit', 10);
                $channel->load(['conversations' => function ($query) use ($limit) {
                    $query->latest('last_message_at')
                        ->limit($limit)
                        ->with(['contact', 'assignedUser']);
                }]);
            }

            Log::info('✅ Canal encontrado', [
                'channel_id' => $channel->id,
                'channel_name' => $channel->name,
                'channel_type' => $channel->type
            ]);

            return $this->response($channel);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            Log::warning('⚠️ Canal não encontrado', [
                'channel_id' => $channelId,
                'user_id' => auth()->id()
            ]);

            return $this->response([
                'error' => 'Canal não encontrado'
            ], 404);
        } catch (\Exception $e) {
            Log::error('❌ Erro ao buscar canal', [
                'channel_id' => $channelId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'user_id' => auth()->id()
            ]);

            return $this->response([
                'error' => 'Erro interno do servidor'
            ], 500);
        }
    }

    /**
     * ➕ CRIAR NOVO CANAL - VERSÃO CORRIGIDA COM VALIDAÇÃO EVOLUTION
     */
    public function store(Request $request): JsonResponse
    {
        // ✅ REGRAS DE VALIDAÇÃO BÁSICAS
        $rules = [
            'name' => 'required|string|max:255',
            'type' => 'required|string|in:internal_chat,evolution_api,whatsapp_business,telegram,instagram,facebook',
            'description' => 'nullable|string|max:1000',
            'config' => 'nullable|array',
            'is_active' => 'boolean',
            'priority' => 'integer|min:1|max:10',
            'webhook_token' => 'nullable|string|max:255',
        ];

        // ✅ VALIDAÇÃO ESPECÍFICA PARA EVOLUTION API
        if ($request->type === 'evolution_api') {
            $rules['config.instance_name'] = [
                'required',
                'string',
                function ($attribute, $value, $fail) {
                    // ✅ VERIFICAR SE A INSTÂNCIA EXISTE
                    $instance = EvolutionInstance::where('instance_name', $value)->first();
                    if (!$instance) {
                        $fail('A instância Evolution selecionada não existe.');
                        return;
                    }

                    // ✅ VERIFICAR SE JÁ ESTÁ SENDO USADA POR OUTRO CANAL
                    $existingChannel = ChatChannel::where('type', 'evolution_api')
                        ->whereJsonContains('config->instance_name', $value)
                        ->first();
                    
                    if ($existingChannel) {
                        $fail("A instância '{$value}' já está sendo usada pelo canal '{$existingChannel->name}'.");
                    }
                }
            ];
        }

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return $this->response(['errors' => $validator->errors()], 422);
        }

        try {
            DB::beginTransaction();

            $data = $request->only(['name', 'type', 'description', 'is_active', 'priority', 'webhook_token']);
            $data['config'] = $request->config ?? [];

            // ✅ PARA EVOLUTION API, VALIDAR E CONFIGURAR
            if ($request->type === 'evolution_api' && !empty($data['config']['instance_name'])) {
                $evolutionInstance = EvolutionInstance::where('instance_name', $data['config']['instance_name'])->first();

                if ($evolutionInstance) {
                    // ✅ TESTAR CONEXÃO ANTES DE CRIAR O CANAL
                    try {
                        $apiService = new EvolutionApiService();
                        $apiService->setInstance($evolutionInstance);
                        $connectionTest = $apiService->testConnection();
                        
                        if (!$connectionTest['success']) {
                            return $this->response([
                                'error' => 'Falha ao conectar com a instância Evolution: ' . $connectionTest['message']
                            ], 422);
                        }
                    } catch (Exception $e) {
                        Log::warning('⚠️ Não foi possível testar conexão Evolution durante criação do canal', [
                            'instance_name' => $evolutionInstance->instance_name,
                            'error' => $e->getMessage()
                        ]);
                        // Continua mesmo se o teste falhar (pode estar offline temporariamente)
                    }

                    // ✅ CONFIGURAR CANAL COM DADOS DA INSTÂNCIA
                    $data['config'] = [
                        'instance_name' => $evolutionInstance->instance_name,
                        'api_url' => $evolutionInstance->api_url,
                        'webhook_url' => $evolutionInstance->webhook_url
                    ];

                    Log::info('✅ Canal Evolution vinculado à instância', [
                        'channel_name' => $data['name'],
                        'instance_name' => $evolutionInstance->instance_name,
                        'instance_id' => $evolutionInstance->id
                    ]);
                }
            }

            $channel = ChatChannel::create($data);

            DB::commit();

            Log::info('✅ Canal criado com sucesso', [
                'channel_id' => $channel->id,
                'name' => $channel->name,
                'type' => $channel->type,
                'user_id' => auth()->id()
            ]);

            return $this->response($channel, 201);

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('❌ Erro ao criar canal: ' . $e->getMessage(), [
                'data' => $request->all(),
                'trace' => $e->getTraceAsString()
            ]);
            return $this->response(['error' => 'Erro ao criar canal: ' . $e->getMessage()], 500);
        }
    }

    /**
     * ✏️ ATUALIZAR CANAL
     */
    public function update(Request $request, $channelId): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|required|string|max:255',
            'type' => 'sometimes|required|in:evolution_api,whatsapp_business,internal_chat,telegram,instagram,facebook',
            'description' => 'nullable|string|max:500',
            'config' => 'nullable|array',
            'is_active' => 'boolean',
            'priority' => 'integer|min:1|max:100',
            'webhook_url' => 'nullable|url|max:500',
            'webhook_token' => 'nullable|string|max:255'
        ]);

        if ($validator->fails()) {
            return $this->response([
                'errors' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();

        try {
            $channel = ChatChannel::findOrFail($channelId);

            Log::info('✏️ Atualizando canal', [
                'channel_id' => $channel->id,
                'changes' => $request->except(['webhook_token']),
                'user_id' => auth()->id()
            ]);

            // ✅ VERIFICAR CONFLITO DE NOME/TIPO SE ALTERADO
            if ($request->filled('name') || $request->filled('type')) {
                $name = $request->input('name', $channel->name);
                $type = $request->input('type', $channel->type);

                $existingChannel = ChatChannel::where('name', $name)
                    ->where('type', $type)
                    ->where('id', '!=', $channel->id)
                    ->first();

                if ($existingChannel) {
                    return $this->response([
                        'error' => 'Já existe outro canal com este nome e tipo'
                    ], 422);
                }
            }

            // ✅ VALIDAR NOVA CONFIGURAÇÃO SE FORNECIDA
            if ($request->filled('config')) {
                $type = $request->input('type', $channel->type);
                $config = $request->input('config');

                $validationResult = $this->validateChannelConfig($type, $config);

                if (!$validationResult['valid']) {
                    return $this->response([
                        'error' => 'Configuração inválida: ' . $validationResult['message']
                    ], 422);
                }
            }

            // ✅ ATUALIZAR CAMPOS
            $updateData = $request->only([
                'name',
                'type',
                'description',
                'config',
                'is_active',
                'priority',
                'webhook_url',
                'webhook_token'
            ]);

            $channel->update($updateData);

            // ✅ RECONFIGURAR WEBHOOK SE NECESSÁRIO
            if ($request->filled('webhook_url') && $channel->isExternal()) {
                $this->setupWebhook($channel);
            }

            DB::commit();

            Log::info('✅ Canal atualizado com sucesso', [
                'channel_id' => $channel->id,
                'channel_name' => $channel->name
            ]);

            return $this->response($channel);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->response([
                'error' => 'Canal não encontrado'
            ], 404);
        } catch (\Exception $e) {
            DB::rollBack();

            Log::error('❌ Erro ao atualizar canal', [
                'channel_id' => $channelId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'user_id' => auth()->id()
            ]);

            return $this->response([
                'error' => 'Erro interno do servidor'
            ], 500);
        }
    }

    /**
     * 🗑️ DELETAR CANAL - VERSÃO MELHORADA
     */
    public function destroy($channelId): JsonResponse
    {
        try {
            $channel = ChatChannel::findOrFail($channelId);

            Log::info('🗑️ Iniciando exclusão do canal', [
                'channel_id' => $channel->id,
                'channel_name' => $channel->name,
                'channel_type' => $channel->type,
                'user_id' => auth()->id()
            ]);

            // ✅ VERIFICAÇÕES DE SEGURANÇA
            
            // 1. Verificar se há conversas ativas
            $activeConversations = $channel->conversations()
                ->whereIn('status', ['active', 'waiting'])
                ->count();

            if ($activeConversations > 0) {
                return $this->response([
                    'error' => "Não é possível excluir o canal. Existem {$activeConversations} conversas ativas.",
                    'details' => [
                        'active_conversations' => $activeConversations,
                        'suggestion' => 'Finalize ou arquive as conversas ativas antes de excluir o canal.'
                    ]
                ], 422);
            }

            // 2. Verificar se é o último canal ativo
            $activeChannelsCount = ChatChannel::where('is_active', true)
                ->where('id', '!=', $channel->id)
                ->count();

            if ($activeChannelsCount === 0) {
                return $this->response([
                    'error' => 'Não é possível excluir o último canal ativo do sistema.',
                    'details' => [
                        'suggestion' => 'Crie outro canal antes de excluir este.'
                    ]
                ], 422);
            }

            // 3. Para canais Evolution, verificar dependências
            if ($channel->type === 'evolution_api') {
                $instanceName = $channel->getConfig('instance_name');
                
                if ($instanceName) {
                    // Verificar se há outras dependências da instância
                    $otherChannels = ChatChannel::where('type', 'evolution_api')
                        ->where('id', '!=', $channel->id)
                        ->whereJsonContains('config->instance_name', $instanceName)
                        ->count();

                    Log::info('📊 Verificação de dependências Evolution', [
                        'instance_name' => $instanceName,
                        'other_channels_using' => $otherChannels
                    ]);
                }
            }

            DB::beginTransaction();

            // ✅ COLETAR ESTATÍSTICAS ANTES DA EXCLUSÃO
            $stats = [
                'total_conversations' => $channel->conversations()->count(),
                'total_messages' => $channel->conversations()->withCount('messages')->get()->sum('messages_count'),
                'archived_conversations' => $channel->conversations()->where('status', 'archived')->count()
            ];

            // ✅ SOFT DELETE DO CANAL
            $channel->delete();

            DB::commit();

            Log::info('✅ Canal excluído com sucesso', [
                'channel_id' => $channel->id,
                'channel_name' => $channel->name,
                'stats' => $stats,
                'user_id' => auth()->id()
            ]);

            return $this->response([
                'message' => 'Canal excluído com sucesso',
                'channel' => [
                    'id' => $channel->id,
                    'name' => $channel->name,
                    'type' => $channel->type
                ],
                'stats' => $stats
            ]);

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->response([
                'error' => 'Canal não encontrado'
            ], 404);

        } catch (Exception $e) {
            DB::rollBack();

            Log::error('❌ Erro ao excluir canal', [
                'channel_id' => $channelId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'user_id' => auth()->id()
            ]);

            return $this->response([
                'error' => 'Erro interno do servidor ao excluir canal'
            ], 500);
        }
    }

    /**
     * 🔄 FORÇAR EXCLUSÃO (PARA CASOS ESPECIAIS)
     */
    public function forceDestroy($channelId): JsonResponse
    {
        try {
            $channel = ChatChannel::findOrFail($channelId);

            Log::warning('⚠️ Exclusão forçada do canal solicitada', [
                'channel_id' => $channel->id,
                'channel_name' => $channel->name,
                'user_id' => auth()->id()
            ]);

            DB::beginTransaction();

            // ✅ ARQUIVAR CONVERSAS ATIVAS ANTES DE EXCLUIR
            $archivedCount = $channel->conversations()
                ->whereIn('status', ['active', 'waiting'])
                ->update([
                    'status' => 'archived',
                    'closed_at' => now(),
                    'close_reason' => 'Canal excluído pelo administrador'
                ]);

            // ✅ SOFT DELETE DO CANAL
            $channel->delete();

            DB::commit();

            Log::warning('⚠️ Canal excluído forçadamente', [
                'channel_id' => $channel->id,
                'conversations_archived' => $archivedCount,
                'user_id' => auth()->id()
            ]);

            return $this->response([
                'message' => 'Canal excluído forçadamente',
                'conversations_archived' => $archivedCount,
                'warning' => 'Conversas ativas foram arquivadas automaticamente'
            ]);

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->response(['error' => 'Canal não encontrado'], 404);

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('❌ Erro na exclusão forçada', [
                'channel_id' => $channelId,
                'error' => $e->getMessage()
            ]);
            return $this->response(['error' => 'Erro interno do servidor'], 500);
        }
    }

    /**
     * 🔄 ATIVAR/DESATIVAR CANAL
     */
    public function toggleStatus($channelId): JsonResponse
    {
        try {
            $channel = ChatChannel::findOrFail($channelId);

            Log::info('🔄 Alternando status do canal', [
                'channel_id' => $channel->id,
                'current_status' => $channel->is_active,
                'user_id' => auth()->id()
            ]);

            $channel->update([
                'is_active' => !$channel->is_active
            ]);

            $newStatus = $channel->is_active ? 'ativado' : 'desativado';

            Log::info('✅ Status do canal alterado', [
                'channel_id' => $channel->id,
                'new_status' => $channel->is_active
            ]);

            return $this->response([
                'message' => "Canal {$newStatus} com sucesso",
                'channel' => $channel
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->response([
                'error' => 'Canal não encontrado'
            ], 404);
        } catch (\Exception $e) {
            Log::error('❌ Erro ao alterar status do canal', [
                'channel_id' => $channelId,
                'error' => $e->getMessage(),
                'user_id' => auth()->id()
            ]);

            return $this->response([
                'error' => 'Erro interno do servidor'
            ], 500);
        }
    }

    /**
     * 🧪 TESTAR CONEXÃO DO CANAL
     */
    public function testConnection($channelId): JsonResponse
    {
        try {
            $channel = ChatChannel::findOrFail($channelId);

            Log::info('🧪 Testando conexão do canal', [
                'channel_id' => $channel->id,
                'channel_type' => $channel->type,
                'user_id' => auth()->id()
            ]);

            $result = $this->performConnectionTest($channel);

            Log::info('✅ Teste de conexão concluído', [
                'channel_id' => $channel->id,
                'success' => $result['success'],
                'response_time' => $result['response_time'] ?? null
            ]);

            return $this->response($result);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->response([
                'error' => 'Canal não encontrado'
            ], 404);
        } catch (\Exception $e) {
            Log::error('❌ Erro ao testar conexão', [
                'channel_id' => $channelId,
                'error' => $e->getMessage(),
                'user_id' => auth()->id()
            ]);

            return $this->response([
                'error' => 'Erro interno do servidor'
            ], 500);
        }
    }

    /**
     * 📊 ESTATÍSTICAS DO CANAL
     */
    public function stats($channelId): JsonResponse
    {
        try {
            $channel = ChatChannel::findOrFail($channelId);

            Log::info('📊 Gerando estatísticas do canal', [
                'channel_id' => $channel->id,
                'user_id' => auth()->id()
            ]);

            $stats = $this->generateChannelStats($channel);

            return $this->response($stats);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->response([
                'error' => 'Canal não encontrado'
            ], 404);
        } catch (\Exception $e) {
            Log::error('❌ Erro ao gerar estatísticas', [
                'channel_id' => $channelId,
                'error' => $e->getMessage(),
                'user_id' => auth()->id()
            ]);

            return $this->response([
                'error' => 'Erro interno do servidor'
            ], 500);
        }
    }

    /**
     * 🧪 TESTAR CANAL EVOLUTION
     */
    public function testEvolutionChannel(string $channelId): JsonResponse
    {
        try {
            $channel = ChatChannel::findOrFail($channelId);

            if ($channel->type !== 'evolution_api') {
                return $this->response([
                    'error' => 'Canal não é do tipo Evolution API'
                ], 400);
            }

            // ✅ USAR O MÉTODO testChannel DO SERVICE
            $result = $this->evolutionSyncService->testChannel($channel);

            return $this->response($result);
        } catch (Exception $e) {
            Log::error('❌ Erro ao testar canal Evolution: ' . $e->getMessage(), [
                'channel_id' => $channelId,
                'trace' => $e->getTraceAsString()
            ]);

            return $this->response([
                'error' => 'Erro ao testar canal: ' . $e->getMessage(),
                'success' => false,
                'connected' => false
            ], 500);
        }
    }

    /**
     * 🔄 SINCRONIZAR CANAL EVOLUTION
     */
    public function syncEvolutionChannel(string $channelId): JsonResponse
    {
        try {
            $channel = ChatChannel::findOrFail($channelId);

            if ($channel->type !== 'evolution_api') {
                return $this->response([
                    'error' => 'Canal não é do tipo Evolution API'
                ], 400);
            }

            // ✅ USAR O MÉTODO syncChannel DO SERVICE
            $result = $this->evolutionSyncService->syncChannel($channel);

            return $this->response($result);
        } catch (Exception $e) {
            Log::error('❌ Erro ao sincronizar canal Evolution: ' . $e->getMessage(), [
                'channel_id' => $channelId,
                'trace' => $e->getTraceAsString()
            ]);

            return $this->response([
                'error' => 'Erro ao sincronizar canal: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * 📋 LISTAR INSTÂNCIAS EVOLUTION - CORRIGIDO
     */
    public function getEvolutionInstances(): JsonResponse
    {
        try {
            Log::info('📋 Listando instâncias da Evolution API', [
                'user_id' => auth()->id()
            ]);

            // ✅ BUSCAR INSTÂNCIAS LOCAIS
            $localInstances = EvolutionInstance::select([
                'id', 'instance_name', 'display_name', 'status', 'connection_state',
                'whatsapp_name', 'whatsapp_number', 'is_default', 'created_at'
            ])->get()->map(function($instance) {
                // ✅ VERIFICAR SE ESTÁ SENDO USADA POR ALGUM CANAL
                $isInUse = ChatChannel::where('type', 'evolution_api')
                    ->whereJsonContains('config->instance_name', $instance->instance_name)
                    ->exists();

                return [
                    'id' => $instance->id,
                    'instance_name' => $instance->instance_name,
                    'display_name' => $instance->display_name,
                    'status' => $instance->status,
                    'connection_state' => $instance->connection_state,
                    'whatsapp_name' => $instance->whatsapp_name,
                    'whatsapp_number' => $instance->whatsapp_number,
                    'is_default' => $instance->is_default,
                    'source' => 'crm',
                    'exists_locally' => true,
                    'is_in_use' => $isInUse,
                    'can_be_selected' => !$isInUse // Só pode selecionar se não estiver em uso
                ];
            });

            // ✅ BUSCAR INSTÂNCIAS DA EVOLUTION API
            $evolutionInstances = [];
            $apiError = null;
            
            try {
                $apiInstances = EvolutionApiService::fetchInstancesFromEvolutionApi();
                
                foreach ($apiInstances as $apiInstance) {
                    $instanceName = $apiInstance['name'] ?? null;
                    if (!$instanceName) continue;

                    // ✅ VERIFICAR SE JÁ EXISTE LOCALMENTE
                    $existsLocally = $localInstances->where('instance_name', $instanceName)->first();

                    // ✅ SE NÃO EXISTE LOCALMENTE, ADICIONAR À LISTA
                    if (!$existsLocally) {
                        $evolutionInstances[] = [
                            'instance_name' => $instanceName,
                            'display_name' => $apiInstance['profileName'] ?? $instanceName,
                            'whatsapp_name' => $apiInstance['profileName'] ?? null,
                            'whatsapp_number' => $this->extractNumberFromJid($apiInstance['ownerJid'] ?? null),
                            'status' => $apiInstance['connectionStatus'] ?? 'close',
                            'connection_state' => strtolower($apiInstance['connectionStatus'] ?? 'close'),
                            'is_online' => ($apiInstance['connectionStatus'] ?? 'close') === 'open',
                            'exists_locally' => false,
                            'source' => 'evolution_api',
                            'can_be_selected' => false, // Precisa criar localmente primeiro
                            'api_data' => $apiInstance
                        ];
                    }
                }
            } catch (Exception $e) {
                $apiError = $e->getMessage();
                Log::warning('⚠️ Erro ao buscar instâncias da Evolution API', [
                    'error' => $e->getMessage()
                ]);
            }

            $response = [
                'local_instances' => $localInstances->values()->toArray(),
                'evolution_instances' => $evolutionInstances,
                'summary' => [
                    'local_count' => $localInstances->count(),
                    'evolution_count' => count($evolutionInstances),
                    'available_count' => $localInstances->where('can_be_selected', true)->count()
                ]
            ];

            if ($apiError) {
                $response['api_error'] = $apiError;
            }

            Log::info('✅ Instâncias listadas com sucesso', [
                'local_count' => $localInstances->count(),
                'evolution_count' => count($evolutionInstances),
                'available_count' => $localInstances->where('can_be_selected', true)->count()
            ]);

            return $this->response($response);

        } catch (Exception $e) {
            Log::error('❌ Erro ao listar instâncias da Evolution API: ' . $e->getMessage());

            return $this->response([
                'error' => 'Erro ao listar instâncias: ' . $e->getMessage(),
                'local_instances' => [],
                'evolution_instances' => [],
                'summary' => ['local_count' => 0, 'evolution_count' => 0, 'available_count' => 0]
            ], 500);
        }
    }

    /**
     * 📥 CRIAR INSTÂNCIA LOCAL DA EVOLUTION API - VERSÃO CORRIGIDA
     */
/**
 * 📥 CRIAR INSTÂNCIA LOCAL DA EVOLUTION API - NOME CORRIGIDO
 */
public function createInstanceFromEvolution(Request $request): JsonResponse
{
    $request->validate([
        'instance_name' => 'required|string|unique:evolution_instances,instance_name',
        'api_data' => 'required|array'
    ]);

    try {
        DB::beginTransaction();

        $instanceName = $request->instance_name;
        $apiData = $request->api_data;

        Log::info('📥 Criando instância local baseada na Evolution API', [
            'instance_name' => $instanceName,
            'user_id' => auth()->id()
        ]);

        // ✅ OBTER CONFIGURAÇÕES PADRÃO
        $apiUrl = env('EVOLUTION_API_URL');
        $apiKey = env('EVOLUTION_API_KEY');
        $apiVersion = env('EVOLUTION_API_VERSION', 'v1');

        if (!$apiUrl || !$apiKey) {
            return $this->response([
                'error' => 'Configurações da Evolution API não encontradas no .env'
            ], 422);
        }

        // ✅ PROCESSAR URL
        if (!str_starts_with($apiUrl, 'http')) {
            $apiUrl = 'https://' . $apiUrl;
        }

        // ✅ MAPEAR STATUS
        $connectionStatus = strtolower($apiData['connectionStatus'] ?? 'close');
        $status = match($connectionStatus) {
            'open' => 'active',
            'connecting' => 'connecting',
            'close', 'closed' => 'disconnected',
            default => 'disconnected'
        };

        // ✅ EXTRAIR NÚMERO DO WHATSAPP
        $whatsappNumber = null;
        if (!empty($apiData['ownerJid'])) {
            $parts = explode('@', $apiData['ownerJid']);
            $number = $parts[0] ?? null;
            if ($number && is_numeric($number) && strlen($number) >= 10) {
                $whatsappNumber = $number;
            }
        }

        // ✅ PROCESSAR URL DA FOTO DE PERFIL (TRUNCAR SE MUITO LONGA)
        $profilePicture = $apiData['profilePicUrl'] ?? null;
        if ($profilePicture && strlen($profilePicture) > 191) {
            $profilePicture = substr($profilePicture, 0, 191);
            Log::warning('⚠️ URL da foto de perfil truncada por ser muito longa');
        }

        // ✅ GERAR WEBHOOK URL
        $webhookUrl = route('evolution.webhook.receive', ['instanceName' => $instanceName]);

        // ✅ CRIAR INSTÂNCIA LOCAL
        $instance = EvolutionInstance::create([
            'instance_name' => $instanceName,
            'display_name' => $apiData['profileName'] ?? $instanceName,
            'description' => 'Criada baseada na Evolution API',
            'api_url' => rtrim($apiUrl, '/'),
            'api_key' => $apiKey,
            'api_version' => $apiVersion,
            'status' => $status,
            'connection_state' => $connectionStatus,
            'whatsapp_name' => $apiData['profileName'] ?? null,
            'whatsapp_number' => $whatsappNumber,
            'whatsapp_profile_picture' => $profilePicture,
            'instance_data' => $apiData,
            'webhook_url' => $webhookUrl,
            'webhook_active' => true,
            'webhook_events' => ['messages', 'message_status', 'connection_status'],
            'is_default' => false,
            'priority' => 2,
            'created_by' => auth()->id(),
            'last_ping' => now(),
            'connected_at' => ($status === 'active') ? now() : null
        ]);

        DB::commit();

        Log::info('✅ Instância local criada com sucesso', [
            'instance_id' => $instance->id,
            'instance_name' => $instance->instance_name
        ]);

        return $this->response([
            'success' => true,
            'message' => 'Instância criada com sucesso',
            'instance' => $instance->makeHidden(['api_key'])
        ], 201);

    } catch (Exception $e) {
        DB::rollBack();
        Log::error('❌ Erro ao criar instância: ' . $e->getMessage());

        return $this->response([
            'error' => 'Erro ao criar instância: ' . $e->getMessage()
        ], 500);
    }
}

    // ===============================================
    // 🔧 MÉTODOS AUXILIARES PRIVADOS
    // ===============================================

    /**
     * ✅ VALIDAR CONFIGURAÇÃO ESPECÍFICA POR TIPO DE CANAL
     */
    private function validateChannelConfig(string $type, array $config): array
    {
        switch ($type) {
            case 'evolution_api':
                return $this->validateEvolutionApiConfig($config);

            case 'whatsapp_business':
                return $this->validateWhatsAppBusinessConfig($config);

            case 'internal_chat':
                return $this->validateInternalChatConfig($config);

            case 'telegram':
                return $this->validateTelegramConfig($config);

            default:
                return ['valid' => true, 'message' => 'Configuração válida'];
        }
    }

    private function validateEvolutionApiConfig(array $config): array
    {
        $required = ['instance_name'];

        foreach ($required as $field) {
            if (empty($config[$field])) {
                return [
                    'valid' => false,
                    'message' => "Campo obrigatório: {$field}"
                ];
            }
        }

        return ['valid' => true, 'message' => 'Configuração válida'];
    }

    private function validateWhatsAppBusinessConfig(array $config): array
    {
        $required = ['phone_number_id', 'access_token'];

        foreach ($required as $field) {
            if (empty($config[$field])) {
                return [
                    'valid' => false,
                    'message' => "Campo obrigatório: {$field}"
                ];
            }
        }

        return ['valid' => true, 'message' => 'Configuração válida'];
    }

    private function validateInternalChatConfig(array $config): array
    {
        // Configurações opcionais para chat interno
        if (isset($config['max_participants']) && $config['max_participants'] < 2) {
            return [
                'valid' => false,
                'message' => 'Máximo de participantes deve ser pelo menos 2'
            ];
        }

        return ['valid' => true, 'message' => 'Configuração válida'];
    }

    private function validateTelegramConfig(array $config): array
    {
        if (empty($config['bot_token'])) {
            return [
                'valid' => false,
                'message' => 'Token do bot é obrigatório'
            ];
        }

        return ['valid' => true, 'message' => 'Configuração válida'];
    }

    /**
     * 🔗 CONFIGURAR WEBHOOK
     */
    private function setupWebhook(ChatChannel $channel): void
    {
        try {
            Log::info('🔗 Configurando webhook', [
                'channel_id' => $channel->id,
                'webhook_url' => $channel->webhook_url
            ]);

            // Implementar configuração específica por tipo
            switch ($channel->type) {
                case 'evolution_api':
                    $this->setupEvolutionApiWebhook($channel);
                    break;

                case 'whatsapp_business':
                    $this->setupWhatsAppBusinessWebhook($channel);
                    break;
            }
        } catch (\Exception $e) {
            Log::error('❌ Erro ao configurar webhook', [
                'channel_id' => $channel->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    private function setupEvolutionApiWebhook(ChatChannel $channel): void
    {
        // Implementar configuração específica do Evolution API
        // Fazer requisição para API externa para configurar webhook
    }

    private function setupWhatsAppBusinessWebhook(ChatChannel $channel): void
    {
        // Implementar configuração específica do WhatsApp Business
    }

    /**
     * 🧪 REALIZAR TESTE DE CONEXÃO
     */
    private function performConnectionTest(ChatChannel $channel): array
    {
        $startTime = microtime(true);

        try {
            switch ($channel->type) {
                case 'evolution_api':
                    return $this->testEvolutionApiConnection($channel, $startTime);

                case 'whatsapp_business':
                    return $this->testWhatsAppBusinessConnection($channel, $startTime);

                case 'internal_chat':
                    return $this->testInternalChatConnection($channel, $startTime);

                default:
                    return [
                        'success' => false,
                        'message' => 'Tipo de canal não suportado para teste',
                        'response_time' => round((microtime(true) - $startTime) * 1000, 2)
                    ];
            }
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Erro ao testar conexão: ' . $e->getMessage(),
                'response_time' => round((microtime(true) - $startTime) * 1000, 2)
            ];
        }
    }

    private function testEvolutionApiConnection(ChatChannel $channel, float $startTime): array
    {
        // Implementar teste específico para Evolution API
        // Fazer requisição para verificar status da instância

        return [
            'success' => true,
            'message' => 'Conexão com Evolution API estabelecida',
            'response_time' => round((microtime(true) - $startTime) * 1000, 2),
            'details' => [
                'api_url' => $channel->getConfig('api_url'),
                'instance_name' => $channel->getConfig('instance_name'),
                'status' => 'connected'
            ]
        ];
    }

    private function testWhatsAppBusinessConnection(ChatChannel $channel, float $startTime): array
    {
        // Implementar teste específico para WhatsApp Business
        return [
            'success' => true,
            'message' => 'Conexão com WhatsApp Business estabelecida',
            'response_time' => round((microtime(true) - $startTime) * 1000, 2)
        ];
    }

    private function testInternalChatConnection(ChatChannel $channel, float $startTime): array
    {
        // Chat interno sempre funciona se o canal estiver ativo
        return [
            'success' => $channel->is_active,
            'message' => $channel->is_active ? 'Chat interno funcionando' : 'Chat interno desativado',
            'response_time' => round((microtime(true) - $startTime) * 1000, 2)
        ];
    }

    /**
     * 📊 GERAR ESTATÍSTICAS COMPLETAS
     */
    private function generateChannelStats(ChatChannel $channel): array
    {
        $now = now();
        $today = $now->startOfDay();
        $thisWeek = $now->copy()->startOfWeek();
        $thisMonth = $now->copy()->startOfMonth();

        return [
            'channel_info' => [
                'id' => $channel->id,
                'name' => $channel->name,
                'type' => $channel->type,
                'is_active' => $channel->is_active,
                'created_at' => $channel->created_at
            ],
            'conversations' => [
                'total' => $channel->conversations()->count(),
                'active' => $channel->conversations()->where('status', 'active')->count(),
                'closed' => $channel->conversations()->where('status', 'closed')->count(),
                'today' => $channel->conversations()->where('created_at', '>=', $today)->count(),
                'this_week' => $channel->conversations()->where('created_at', '>=', $thisWeek)->count(),
                'this_month' => $channel->conversations()->where('created_at', '>=', $thisMonth)->count()
            ],
            'messages' => [
                'total' => $channel->conversations()
                    ->withCount('messages')
                    ->get()
                    ->sum('messages_count'),
                'today' => $this->getMessageCountForPeriod($channel, $today),
                'this_week' => $this->getMessageCountForPeriod($channel, $thisWeek),
                'this_month' => $this->getMessageCountForPeriod($channel, $thisMonth)
            ],
            'performance' => [
                'avg_response_time' => $this->calculateAverageResponseTime($channel),
                'resolution_rate' => $this->calculateResolutionRate($channel)
            ]
        ];
    }

    /**
     * ➕ ADICIONAR ESTATÍSTICAS BÁSICAS AO CANAL
     */
    private function addChannelStats(ChatChannel $channel): ChatChannel
    {
        $channel->conversations_count = $channel->conversations()->count();
        $channel->active_conversations_count = $channel->conversations()
            ->where('status', 'active')
            ->count();

        return $channel;
    }

    private function getMessageCountForPeriod(ChatChannel $channel, $since): int
    {
        return DB::table('messages')
            ->join('conversations', 'messages.conversation_id', '=', 'conversations.id')
            ->where('conversations.chat_channel_id', $channel->id)
            ->where('messages.created_at', '>=', $since)
            ->count();
    }

    private function calculateAverageResponseTime(ChatChannel $channel): ?float
    {
        // Implementar cálculo de tempo médio de resposta
        return null; // Placeholder
    }

    private function calculateResolutionRate(ChatChannel $channel): float
    {
        $total = $channel->conversations()->count();
        if ($total === 0) return 0;

        $resolved = $channel->conversations()->where('status', 'closed')->count();
        return round(($resolved / $total) * 100, 2);
    }

    /**
     * 🔧 EXTRAIR NÚMERO DO JID
     */
    private function extractNumberFromJid(?string $ownerJid): ?string
    {
        if (!$ownerJid) return null;

        $parts = explode('@', $ownerJid);
        $number = $parts[0] ?? null;

        if ($number && is_numeric($number) && strlen($number) >= 10) {
            return $number;
        }

        return null;
    }
}