<?php
// ContactController.php - CORRIGIDO COM CAMPOS REAIS DO CONCORD

namespace Modules\Chat\Http\Controllers\Api;

use Modules\Core\Http\Controllers\ApiController;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Modules\Contacts\Models\Contact;
use Modules\Contacts\Enums\PhoneType;

class ContactController extends ApiController
{
     /**
     * ✅ BUSCAR CONTATOS - CORRIGIDO COM CAMPOS REAIS
     */
    public function search(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'q' => 'required|string|min:1|max:100',
            'limit' => 'nullable|integer|min:1|max:50'
        ]);

        if ($validator->fails()) {
            return $this->response(['errors' => $validator->errors()], 422);
        }

        try {
            $query = $request->input('q');
            $limit = $request->input('limit', 10);

            Log::info('🔍 Buscando contatos', [
                'query' => $query,
                'limit' => $limit,
                'user_id' => auth()->id()
            ]);

            // ✅ BUSCAR NOS CONTATOS DO CONCORDCRM COM CAMPOS REAIS
            $contacts = Contact::where(function ($q) use ($query) {
                $q->where('first_name', 'LIKE', "%{$query}%")
                  ->orWhere('last_name', 'LIKE', "%{$query}%")
                  ->orWhere('email', 'LIKE', "%{$query}%")
                  ->orWhereHas('phones', function ($phoneQuery) use ($query) {
                      $phoneQuery->where('number', 'LIKE', "%{$query}%");
                  });
            })
            ->with(['phones'])
            ->limit($limit)
            ->get()
            ->map(function ($contact) {
                $phone = $contact->phones->first();
                return [
                    'id' => $contact->id,
                    'name' => $contact->full_name, // ✅ USAR full_name
                    'first_name' => $contact->first_name,
                    'last_name' => $contact->last_name,
                    'phone' => $phone ? $this->formatPhone($phone->number) : null,
                    'email' => $contact->email,
                    'avatar' => $contact->avatar_url ?? null, // ✅ USAR avatar_url
                    'source' => 'concordcrm'
                ];
            })
            ->filter(function ($contact) {
                return !empty($contact['phone']); // Só contatos com telefone
            })
            ->values();

            Log::info('🔍 Contatos encontrados', [
                'count' => $contacts->count(),
                'query' => $query
            ]);

            return $this->response($contacts);

        } catch (\Exception $e) {
            Log::error('❌ Erro ao buscar contatos', [
                'query' => $request->input('q'),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return $this->response(['error' => 'Erro interno do servidor'], 500);
        }
    }

 
    /**
     * ✅ CONTATOS RECENTES - CORRIGIDO
     */
    public function recent(Request $request): JsonResponse
    {
        try {
            $limit = $request->input('limit', 10);

            Log::info('📞 Carregando contatos recentes', [
                'limit' => $limit,
                'user_id' => auth()->id()
            ]);

            // ✅ BUSCAR CONTATOS MAIS RECENTES COM CAMPOS REAIS
            $recentContacts = Contact::with(['phones'])
                ->whereHas('phones') // Apenas contatos com telefone
                ->orderBy('updated_at', 'desc')
                ->limit($limit)
                ->get()
                ->map(function ($contact) {
                    $phone = $contact->phones->first();
                    return [
                        'id' => $contact->id,
                        'name' => $contact->full_name, // ✅ USAR full_name
                        'first_name' => $contact->first_name,
                        'last_name' => $contact->last_name,
                        'phone' => $phone ? $this->formatPhone($phone->number) : null,
                        'email' => $contact->email,
                        'avatar' => $contact->avatar_url ?? null, // ✅ USAR avatar_url
                        'source' => 'recent'
                    ];
                })
                ->filter(function ($contact) {
                    return !empty($contact['phone']);
                })
                ->values();

            Log::info('📞 Contatos recentes carregados', [
                'count' => $recentContacts->count()
            ]);

            return $this->response($recentContacts);

        } catch (\Exception $e) {
            Log::error('❌ Erro ao carregar contatos recentes', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return $this->response(['error' => 'Erro interno do servidor'], 500);
        }
    }

    /**
     * ✅ CRIAR NOVO CONTATO - CORRIGIDO
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'email' => 'nullable|email|max:255'
        ]);

        if ($validator->fails()) {
            return $this->response(['errors' => $validator->errors()], 422);
        }

        try {
            $phone = $this->cleanPhone($request->input('phone'));
            
            if (!$this->isValidPhone($phone)) {
                return $this->response(['error' => 'Número de telefone inválido'], 422);
            }

            Log::info('📝 Criando novo contato', [
                'name' => $request->input('name'),
                'phone' => $phone,
                'user_id' => auth()->id()
            ]);

            // ✅ VERIFICAR SE JÁ EXISTE CONTATO COM ESTE TELEFONE
            $existingContact = Contact::whereHas('phones', function ($query) use ($phone) {
                $query->where('number', $phone);
            })->first();

            if ($existingContact) {
                return $this->response([
                    'error' => 'Já existe um contato com este telefone',
                    'existing_contact' => [
                        'id' => $existingContact->id,
                        'name' => $existingContact->full_name, // ✅ USAR full_name
                        'phone' => $phone
                    ]
                ], 409);
            }

            // ✅ CRIAR NOVO CONTATO COM CAMPOS REAIS
            $nameParts = explode(' ', trim($request->input('name')), 2);
            
            $contact = Contact::create([
                'first_name' => $nameParts[0],
                'last_name' => $nameParts[1] ?? '',
                'email' => $request->input('email'),
                'user_id' => auth()->id(), // ✅ OWNER
                'created_by' => auth()->id() // ✅ CRIADOR
            ]);

            // ✅ ADICIONAR TELEFONE
            $contact->phones()->create([
                'number' => $phone,
                'type' => PhoneType::mobile // ✅ USAR ENUM CORRETO
            ]);

            $contactData = [
                'id' => $contact->id,
                'name' => $contact->full_name, // ✅ USAR full_name
                'first_name' => $contact->first_name,
                'last_name' => $contact->last_name,
                'phone' => $this->formatPhone($phone),
                'email' => $contact->email,
                'source' => 'new'
            ];

            Log::info('📝 Contato criado com sucesso', [
                'contact_id' => $contact->id,
                'name' => $contact->full_name,
                'phone' => $phone
            ]);

            return $this->response($contactData, 201);

        } catch (\Exception $e) {
            Log::error('❌ Erro ao criar contato', [
                'name' => $request->input('name'),
                'phone' => $request->input('phone'),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return $this->response(['error' => 'Erro interno do servidor'], 500);
        }
    }

    /**
     * 👁️ DETALHES DO CONTATO
     */
    public function show($contactId): JsonResponse
    {
        try {
            $contact = Contact::with(['phones', 'emails'])->findOrFail($contactId);

            $contactData = [
                'id' => $contact->id,
                'name' => $contact->display_name,
                'first_name' => $contact->first_name,
                'last_name' => $contact->last_name,
                'email' => $contact->email,
                'phones' => $contact->phones->map(function ($phone) {
                    return [
                        'id' => $phone->id,
                        'number' => $this->formatPhone($phone->number),
                        'type' => $phone->type
                    ];
                }),
                'emails' => $contact->emails->map(function ($email) {
                    return [
                        'id' => $email->id,
                        'email' => $email->email,
                        'type' => $email->type
                    ];
                }),
                'avatar' => $contact->avatar_url ?? null,
                'created_at' => $contact->created_at?->toISOString(),
                'updated_at' => $contact->updated_at?->toISOString()
            ];

            return $this->response($contactData);

        } catch (\Exception $e) {
            Log::error('❌ Erro ao carregar contato', [
                'contact_id' => $contactId,
                'error' => $e->getMessage()
            ]);
            return $this->response(['error' => 'Contato não encontrado'], 404);
        }
    }

    /**
     * ✏️ ATUALIZAR CONTATO
     */
    public function update(Request $request, $contactId): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|required|string|max:255',
            'phone' => 'sometimes|required|string|max:20',
            'email' => 'nullable|email|max:255'
        ]);

        if ($validator->fails()) {
            return $this->response(['errors' => $validator->errors()], 422);
        }

        try {
            $contact = Contact::findOrFail($contactId);

            Log::info('✏️ Atualizando contato', [
                'contact_id' => $contactId,
                'updates' => $request->only(['name', 'phone', 'email'])
            ]);

            // Atualizar nome se fornecido
            if ($request->has('name')) {
                $nameParts = explode(' ', trim($request->input('name')), 2);
                $contact->first_name = $nameParts[0];
                $contact->last_name = $nameParts[1] ?? '';
            }

            // Atualizar email se fornecido
            if ($request->has('email')) {
                $contact->email = $request->input('email');
            }

            $contact->save();

            // Atualizar telefone se fornecido
            if ($request->has('phone')) {
                $phone = $this->cleanPhone($request->input('phone'));
                
                if ($this->isValidPhone($phone)) {
                    // Atualizar o primeiro telefone ou criar novo
                    $phoneRecord = $contact->phones()->first();
                    if ($phoneRecord) {
                        $phoneRecord->update(['number' => $phone]);
                    } else {
                        $contact->phones()->create([
                            'number' => $phone,
                            'type' => 'mobile'
                        ]);
                    }
                }
            }

            $contact->load(['phones']);

            $contactData = [
                'id' => $contact->id,
                'name' => $contact->display_name,
                'phone' => $this->formatPhone($contact->phones->first()?->number),
                'email' => $contact->email,
                'updated_at' => $contact->updated_at?->toISOString()
            ];

            Log::info('✏️ Contato atualizado com sucesso', [
                'contact_id' => $contactId,
                'name' => $contact->display_name
            ]);

            return $this->response($contactData);

        } catch (\Exception $e) {
            Log::error('❌ Erro ao atualizar contato', [
                'contact_id' => $contactId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return $this->response(['error' => 'Erro interno do servidor'], 500);
        }
    }

    // 🔧 MÉTODOS AUXILIARES

    private function formatPhone(string $phone): string
    {
        $cleaned = preg_replace('/\D/', '', $phone);
        
        if (strlen($cleaned) === 11) {
            // Celular brasileiro: (XX) 9XXXX-XXXX
            return '(' . substr($cleaned, 0, 2) . ') ' . substr($cleaned, 2, 5) . '-' . substr($cleaned, 7);
        } elseif (strlen($cleaned) === 10) {
            // Fixo brasileiro: (XX) XXXX-XXXX
            return '(' . substr($cleaned, 0, 2) . ') ' . substr($cleaned, 2, 4) . '-' . substr($cleaned, 6);
        }
        
        return $phone; // Retorna como está se não for padrão brasileiro
    }

    private function cleanPhone(string $phone): string
    {
        return preg_replace('/\D/', '', $phone);
    }

    private function isValidPhone(string $phone): bool
    {
        $cleaned = $this->cleanPhone($phone);
        return strlen($cleaned) >= 10 && strlen($cleaned) <= 15;
    }
}
