<?php

namespace Modules\Chat\Http\Controllers\Api;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Modules\Core\Http\Controllers\ApiController;
use Modules\Chat\Models\Conversation;
use Modules\Chat\Models\Message;
use Modules\Chat\Models\ChatChannel;
use Modules\Contacts\Models\Contact;
use Modules\Users\Models\User;
use Modules\Contacts\Enums\PhoneType;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Modules\Chat\Services\ChatRedisService;
use Modules\Chat\Http\Traits\FormatsConversations;
use Modules\Chat\Http\Controllers\Api\MessageController;

class ConversationController extends ApiController
{
    use FormatsConversations;

    public function index(Request $request): JsonResponse
    {
        try {
            $userId = auth()->id();
            $user = auth()->user();
            $isAdmin = $user->super_admin ?? false;

            $perPage = min((int) $request->input('per_page', 20), 100);
            $page = max((int) $request->input('page', 1), 1);

            $includeArchived = $request->boolean('include_archived', false);
            $showAll = $request->boolean('show_all', false);

            $status = $request->input('status');
            $type = $request->input('type');
            $assigned = $request->input('assigned');
            $search = trim((string) $request->input('search', ''));
            $onlyUnread = $request->boolean('only_unread', false);

            $query = Conversation::with([
                'lastMessage',
                'contact.companies:id,name',
                'contact',
                'assignedUser:id,name,email,avatar',
                'participants:id,name,email,avatar',
                'assignedUsers:id,name,email,avatar',
            ]);

            if (!($isAdmin && $showAll)) {
                $query->visibleToUser($userId);
            }

            if (!is_null($status) && $status !== '') {
                if (is_array($status)) {
                    $query->whereIn('status', $status);
                } else {
                    $query->where('status', $status);
                }
            }

            if (!empty($type)) {
                $query->where('type', $type);
            }

            if (!empty($assigned) && $assigned !== 'all') {
                if ($assigned === 'me') {
                    $query->where('assigned_user_id', $userId);
                } elseif ($assigned === 'unassigned') {
                    $query->whereNull('assigned_user_id');
                } elseif (is_numeric($assigned)) {
                    $query->where('assigned_user_id', (int) $assigned);
                }
            }

            if ($onlyUnread) {
                $query->where(function ($q) use ($userId) {
                    $q->where(function ($sub) {
                        $sub->where('type', 'external')->where('unread_count', '>', 0);
                    })->orWhere(function ($sub) use ($userId) {
                        $sub->where('type', 'internal')
                            ->whereHas('messages', function ($msg) use ($userId) {
                                $msg->whereNull('read_at')->where('sender_id', '!=', $userId);
                            });
                    });
                });
            }

            if ($search !== '') {
                $query->where(function ($q) use ($search) {
                    $like = '%' . $search . '%';
                    $q->where('conversations.whatsapp_profile_name', 'LIKE', $like)
                        ->orWhere('conversations.contact_phone', 'LIKE', $like)
                        ->orWhereHas('contact', function ($c) use ($like) {
                            $c->where('first_name', 'LIKE', $like)
                                ->orWhere('last_name', 'LIKE', $like)
                                ->orWhere('email', 'LIKE', $like)
                                ->orWhereHas('phones', function ($p) use ($like) {
                                    $p->where('number', 'LIKE', $like);
                                })
                                ->orWhereHas('companies', function ($comp) use ($like) {
                                    $comp->where('name', 'LIKE', $like);
                                });
                        });
                });
            }

            if (!$includeArchived) {
                $query->where('status', '!=', 'archived');
            }

            $query->orderByDesc('is_pinned')->orderByDesc('last_message_at');

            $conversations = $query->paginate($perPage, ['*'], 'page', $page);

            $formattedConversations = $conversations->getCollection()
                ->map(fn($conv) => $this->formatConversation($conv, $userId));

            return $this->response([
                'data' => $formattedConversations->toArray(),
                'meta' => [
                    'current_page' => $conversations->currentPage(),
                    'last_page' => $conversations->lastPage(),
                    'per_page' => $conversations->perPage(),
                    'total' => $conversations->total(),
                    'from' => $conversations->firstItem(),
                    'to' => $conversations->lastItem(),
                    'has_more' => $conversations->hasMorePages(),
                    'is_admin' => $isAdmin,
                    'show_all' => $showAll,
                    'user_id' => $userId,
                    'filters' => [
                        'status' => $status,
                        'type' => $type,
                        'assigned' => $assigned,
                        'search' => $search,
                        'only_unread' => $onlyUnread,
                        'include_archived' => $includeArchived,
                    ],
                ],
            ]);
        } catch (\Exception $e) {
            Log::error('Erro ao carregar conversas', [
                'user_id' => auth()->id(),
                'error' => $e->getMessage()
            ]);

            return $this->response([
                'error' => 'Erro ao carregar conversas',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao carregar conversas.',
                ],
            ], 500);
        }
    }

    public function archived(Request $request): JsonResponse
    {
        try {
            $userId = auth()->id();
            $isAdmin = auth()->user()->super_admin ?? false;

            $query = Conversation::with([
                'lastMessage',
                'contact.companies:id,name',
                'contact',
                'assignedUser:id,name,email,avatar',
                'participants:id,name,email,avatar'
            ])
                ->where('status', 'archived')
                ->when(!$isAdmin, function ($q) use ($userId) {
                    return $q->where(function ($subQ) use ($userId) {
                        $subQ->whereNull('assigned_user_id')
                            ->orWhere('assigned_user_id', $userId)
                            ->orWhere(function ($internalQ) use ($userId) {
                                $internalQ->where('type', 'internal')
                                    ->whereHas('participants', fn($p) => $p->where('user_id', $userId)->wherePivot('is_active', true));
                            });
                    });
                })
                ->orderByDesc('updated_at');

            $conversations = $query->get()->map(fn($conv) => $this->formatConversation($conv, $userId));

            return $this->response($conversations->toArray());
        } catch (\Exception $e) {
            Log::error('Erro ao carregar conversas arquivadas', ['error' => $e->getMessage()]);
            return $this->response([
                'error' => 'Erro interno do servidor',
                'notification' => ['type' => 'error', 'title' => 'Erro', 'message' => 'Falha ao carregar conversas arquivadas.']
            ], 500);
        }
    }

    public function toggleConversationStatus(Request $request, Conversation $conversation): JsonResponse
    {
        $currentUserId = auth()->id();

        if ($conversation->assigned_user_id !== $currentUserId && !auth()->user()->super_admin) {
            return $this->response([
                'error' => 'Você não tem permissão para alterar o status desta conversa.',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Permissão Negada',
                    'message' => 'Você não tem permissão para alterar o status desta conversa.'
                ]
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'status' => 'required|in:active,closed',
            'reason' => 'nullable|string|max:500',

            // Finalização avançada (todos opcionais)
            'send_farewell_message' => 'boolean',
            'farewell_message'      => 'nullable|string|max:1000',
            'archive_conversation'  => 'boolean',
            'block_contact'         => 'boolean',
            'notify_team'           => 'boolean',
            'request_feedback'      => 'boolean',
            'finalized_at'          => 'nullable|date',
            'finalized_by'          => 'nullable|integer|exists:users,id',
        ]);

        if ($validator->fails()) {
            return $this->response(['errors' => $validator->errors()], 422);
        }

        DB::beginTransaction();
        try {
            $newStatus = $request->input('status');

            $updateData = [
                'status' => $newStatus,
            ];

            if ($newStatus === 'closed') {

                // 1) Mensagem de despedida
                if ($request->boolean('send_farewell_message')) {
                    $this->sendSystemMessage(
                        $conversation,
                        $request->input('farewell_message') ?: 'Atendimento finalizado. Obrigado pelo contato!'
                    );
                }

                // 2) Arquivar conversa
                if ($request->boolean('archive_conversation')) {
                    $conversation->update(['is_archived' => true]);
                }

                // 3) Bloquear contato
                if ($request->boolean('block_contact')) {
                    $this->blockConversationContact($conversation);
                }

                // 4) Notificar time
                if ($request->boolean('notify_team')) {
                    $this->notifyTeamConversationClosed($conversation, auth()->user(), $request->input('reason'));
                }

                // 5) Pedir feedback
                if ($request->boolean('request_feedback')) {
                    $this->sendFeedbackRequest($conversation);
                }
            }

            $conversation->update($updateData);
            $this->sendSystemStatusMessage($conversation, $newStatus, $request->input('reason'), $currentUserId);

            DB::commit();

            $chatRedisService = app(ChatRedisService::class);
            $chatRedisService->publishConversationUpdate($conversation->fresh(), 'status_updated');

            $notificationMessage = $newStatus === 'closed' ? 'Conversa finalizada com sucesso.' : 'Conversa reaberta com sucesso.';
            
            return $this->response([
                'success' => true,
                'conversation' => $this->formatConversation($conversation->fresh(), $currentUserId),
                'message' => $notificationMessage,
                'notification' => [
                    'type' => 'success',
                    'title' => 'Sucesso',
                    'message' => $notificationMessage
                ]
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Erro ao alterar status da conversa', ['error' => $e->getMessage()]);
            return $this->response([
                'error' => 'Erro interno do servidor',
                'details' => config('app.debug') ? $e->getMessage() : null,
                'notification' => ['type' => 'error', 'title' => 'Erro', 'message' => 'Falha ao alterar status da conversa.']
            ], 500);
        }
    }

    private function sendSystemStatusMessage(Conversation $conversation, string $newStatus, ?string $reason, int $userId): void
    {
        try {
            $user = User::find($userId);
            $userName = $user ? $user->name : 'Sistema';
            
            $messageContent = $newStatus === 'closed' 
                ? "{$userName} finalizou este atendimento." . ($reason ? " Motivo: {$reason}" : '')
                : "{$userName} reabriu este atendimento.";

            Message::create([
                'conversation_id' => $conversation->id,
                'sender_type' => 'system',
                'sender_id' => null,
                'content' => $messageContent,
                'message_type' => 'system',
                'is_from_contact' => false,
                'status' => 'sent',
                'sent_at' => now(),
                'metadata' => [
                    'is_system_message' => true,
                    'system_type' => ($newStatus === 'closed' ? 'conversation_closed' : 'conversation_reopened'),
                    'user_id' => $userId
                ]
            ]);

            $conversation->update(['last_message_at' => now()]);
        } catch (\Exception $e) {
            Log::error('Erro ao enviar mensagem de sistema', [
                'conversation_id' => $conversation->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    public function store(Request $request): JsonResponse
    {
        // ✅ VALIDAÇÃO DINÂMICA BASEADA NO TIPO
        $rules = [
            'type' => 'required|in:internal,external',
            'priority' => 'nullable|in:low,normal,high,urgent',
            'tags' => 'nullable|array',
            'initial_message' => 'nullable|string|max:4000',
            'notifications_enabled' => 'boolean',
            'auto_assign' => 'boolean',
        ];

        // ✅ REGRAS ESPECÍFICAS PARA CONVERSA EXTERNA
        if ($request->input('type') === 'external') {
            $rules = array_merge($rules, [
                'channel_id' => 'required|exists:chat_channels,id',
                // Se contact_id não for fornecido, exigir dados do novo contato
                'contact_id' => 'nullable|exists:contacts,id',
                'contact_name' => 'required_without:contact_id|string|max:255',
                'phone' => 'required_without:contact_id|string|max:20',
                'contact_email' => 'nullable|email|max:255',
            ]);
        }

        // ✅ REGRAS ESPECÍFICAS PARA CONVERSA INTERNA
        if ($request->input('type') === 'internal') {
            $rules = array_merge($rules, [
                'subject' => 'required|string|max:255',
                'participants' => 'required|array|min:1',
                'participants.*' => 'exists:users,id',
                'internal_type' => 'nullable|in:discussion,project,support,meeting,announcement,other',
                'description' => 'nullable|string|max:500',
                'channel_id' => 'nullable|exists:chat_channels,id',
            ]);
        }

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return $this->response([
                'errors' => $validator->errors(),
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro de Validação',
                    'message' => 'Dados inválidos para criar conversa.'
                ]
            ], 422);
        }

        DB::beginTransaction();

        try {
            Log::info('📝 Criando nova conversa', [
                'type' => $request->input('type'),
                'user_id' => auth()->id(),
                'data' => $request->except(['contact_email', 'phone']) // Log sem dados sensíveis
            ]);

            if ($request->input('type') === 'external') {
                $result = $this->createExternalConversation($request);
            } else {
                $result = $this->createInternalConversation($request);
            }

            DB::commit();
            return $result;

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('❌ Erro ao criar conversa', [
                'type' => $request->input('type'),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'user_id' => auth()->id()
            ]);

            return $this->response([
                'error' => 'Erro interno do servidor',
                'details' => config('app.debug') ? $e->getMessage() : null,
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao criar conversa.'
                ]
            ], 500);
        }
    }

        /**
     * 👥 CRIAR CONVERSA INTERNA
     */
    private function createInternalConversation(Request $request): JsonResponse
    {
        // ✅ VALIDAR PARTICIPANTES
        $participantIds = $request->input('participants', []);
        $currentUserId = auth()->id();

        // ✅ ADICIONAR USUÁRIO ATUAL SE NÃO ESTIVER NA LISTA
        if (!in_array($currentUserId, $participantIds)) {
            $participantIds[] = $currentUserId;
        }

        // ✅ VERIFICAR SE TODOS OS PARTICIPANTES EXISTEM
        $participants = User::whereIn('id', $participantIds)->get();
        
        if ($participants->count() !== count($participantIds)) {
            return $this->response([
                'error' => 'Um ou mais participantes não foram encontrados',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Participantes inválidos selecionados.'
                ]
            ], 422);
        }

        // ✅ OBTER CANAL INTERNO SE ESPECIFICADO
        $channel = null;
        if ($request->filled('channel_id')) {
            $channel = ChatChannel::where('id', $request->input('channel_id'))
                ->where('type', 'internal_chat')
                ->where('is_active', true)
                ->first();
                
            if (!$channel) {
                return $this->response([
                    'error' => 'Canal interno inválido',
                    'notification' => [
                        'type' => 'error',
                        'title' => 'Erro',
                        'message' => 'Canal interno inválido ou inativo.'
                    ]
                ], 422);
            }
        } else {
            // ✅ USAR CANAL PADRÃO OU CRIAR UM
            $channel = $this->getDefaultInternalChannel();
        }

        // ✅ CRIAR CONVERSA INTERNA
        $conversationData = [
            'chat_channel_id' => $channel->id,
            'type' => 'internal',
            'subject' => $request->input('subject'),
            'description' => $request->input('description'),
            'status' => 'active',
            'assigned_user_id' => $request->boolean('auto_assign') ? $currentUserId : null,
            'priority' => $request->input('priority', 'normal'),
            'internal_type' => $request->input('internal_type', 'discussion'),
            'last_message_at' => now(),
            'unread_count' => 0,
            'tags' => $request->input('tags', []),
            'metadata' => [
                'internal_type' => $request->input('internal_type', 'discussion'),
                'participant_count' => count($participantIds),
                'channel_name' => $channel->name,
                'notifications_enabled' => $request->boolean('notifications_enabled', true),
                'created_by' => $currentUserId
            ]
        ];

        $conversation = Conversation::create($conversationData);

        // ✅ ADICIONAR PARTICIPANTES
        $conversation->participants()->attach($participantIds, [
            'joined_at' => now(),
            'role' => 'participant'
        ]);

        // ✅ MARCAR CRIADOR COMO MODERADOR
        $conversation->participants()->updateExistingPivot($currentUserId, [
            'role' => 'moderator'
        ]);

        // ✅ ENVIAR MENSAGEM INICIAL SE FORNECIDA
        if ($request->filled('initial_message')) {
            $this->sendInitialMessage($conversation, $request->input('initial_message'));
        }

        $conversation->load(['participants', 'assignedUser', 'lastMessage', 'chatChannel']);

        Log::info('✅ Conversa interna criada', [
            'conversation_id' => $conversation->id,
            'subject' => $conversation->subject,
            'participants' => $participantIds,
            'channel_id' => $channel->id
        ]);

        return $this->response([
            'conversation' => $this->formatConversation($conversation, $currentUserId),
            'message' => 'Conversa criada com sucesso',
            'notification' => [
                'type' => 'success',
                'title' => 'Sucesso',
                'message' => 'Conversa interna criada com sucesso!'
            ]
        ], 201);
    }


        /**
     * 📞 CRIAR CONVERSA EXTERNA (WHATSAPP)
     */
    private function createExternalConversation(Request $request): JsonResponse
    {
        // ✅ VALIDAR CANAL
        $channel = ChatChannel::findOrFail($request->input('channel_id'));
        
        if (!$channel->is_active || $channel->type !== 'evolution_api') {
            return $this->response([
                'error' => 'Canal inválido ou inativo',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Canal WhatsApp inválido ou inativo.'
                ]
            ], 422);
        }

        // ✅ PROCESSAR CONTATO
        $contact = null;
        $phone = null;
        $contactName = null;

        if ($request->filled('contact_id')) {
            // ✅ USAR CONTATO EXISTENTE
            $contact = Contact::findOrFail($request->input('contact_id'));
            $phone = $this->getContactPhone($contact);
            $contactName = $contact->display_name ?: $contact->first_name;
        } else {
            // ✅ CRIAR NOVO CONTATO
            $phone = $this->cleanPhone($request->input('phone'));
            $contactName = $request->input('contact_name');

            if (!$this->isValidPhone($phone)) {
                return $this->response([
                    'error' => 'Número de telefone inválido',
                    'notification' => [
                        'type' => 'error',
                        'title' => 'Erro de Validação',
                        'message' => 'Número de telefone inválido.'
                    ]
                ], 422);
            }

            // ✅ VERIFICAR SE JÁ EXISTE CONTATO COM ESTE TELEFONE
            $contact = $this->findContactByPhone($phone);
            
            if (!$contact) {
                $contact = $this->createContact([
                    'first_name' => $contactName,
                    'email' => $request->input('contact_email'),
                    'phone' => $phone
                ]);
            }
        }

        // ✅ VERIFICAR CONVERSA EXISTENTE
        $existingConversation = Conversation::where('type', 'external')
            ->where('contact_id', $contact->id)
            ->where('chat_channel_id', $channel->id)
            ->whereIn('status', ['active', 'waiting', 'pending'])
            ->first();

        if ($existingConversation) {
            return $this->response([
                'conversation' => $this->formatConversation($existingConversation->load(['contact', 'lastMessage']), auth()->id()),
                'message' => 'Conversa já existe',
                'notification' => [
                    'type' => 'info',
                    'title' => 'Conversa Existente',
                    'message' => 'Já existe uma conversa ativa com este contato neste canal.'
                ]
            ]);
        }

        // ✅ CRIAR CONVERSA
        $conversationData = [
            'contact_id' => $contact->id,
            'chat_channel_id' => $channel->id,
            'type' => 'external',
            'contact_phone' => $phone,
            'contact_name' => $contactName,
            'status' => 'active',
            'assigned_user_id' => $request->boolean('auto_assign') ? auth()->id() : null,
            'priority' => $request->input('priority', 'normal'),
            'last_message_at' => now(),
            'unread_count' => 0,
            'tags' => $request->input('tags', []),
            'metadata' => [
                'contact_name' => $contactName,
                'channel_type' => 'whatsapp',
                'channel_name' => $channel->name,
                'instance_name' => $channel->getConfig('instance_name'),
                'notifications_enabled' => $request->boolean('notifications_enabled', true),
                'created_by' => auth()->id()
            ]
        ];

        $conversation = Conversation::create($conversationData);

        // ✅ ENVIAR MENSAGEM INICIAL SE FORNECIDA
        if ($request->filled('initial_message')) {
            $this->sendInitialMessage($conversation, $request->input('initial_message'));
        }

        $conversation->load(['contact', 'assignedUser', 'lastMessage', 'chatChannel']);

        Log::info('✅ Conversa externa criada', [
            'conversation_id' => $conversation->id,
            'contact_id' => $contact->id,
            'channel_id' => $channel->id,
            'phone' => $phone
        ]);

        return $this->response([
            'conversation' => $this->formatConversation($conversation, auth()->id()),
            'message' => 'Conversa criada com sucesso',
            'notification' => [
                'type' => 'success',
                'title' => 'Sucesso',
                'message' => 'Conversa externa criada com sucesso!'
            ]
        ], 201);
    }

        private function getContactPhone(Contact $contact): ?string
    {
        if ($contact->phones && $contact->phones->count() > 0) {
            return $this->cleanPhone($contact->phones->first()->number);
        }
        return null;
    }

    private function findContactByPhone(string $phone): ?Contact
    {
        return Contact::whereHas('phones', function ($query) use ($phone) {
            $query->where('number', 'like', '%' . $phone . '%')
                  ->orWhere('number', 'like', '%' . substr($phone, -9) . '%');
        })->first();
    }

    private function createContact(array $data): Contact
    {
        $contactData = [
            'first_name' => $data['first_name'],
            'created_by' => auth()->id()
        ];

        if (!empty($data['email'])) {
            $contactData['email'] = $data['email'];
        }

        $contact = Contact::create($contactData);

        // ✅ ADICIONAR TELEFONE
        if (!empty($data['phone'])) {
            $contact->phones()->create([
                'number' => $data['phone'],
                'type' => 'mobile'
            ]);
        }

        return $contact;
    }

    private function getDefaultInternalChannel(): ChatChannel
    {
        $channel = ChatChannel::where('type', 'internal_chat')
            ->where('is_active', true)
            ->first();

        if (!$channel) {
            $channel = ChatChannel::create([
                'name' => 'Chat Interno Geral',
                'type' => 'internal_chat',
                'description' => 'Canal padrão para conversas internas',
                'is_active' => true,
                'priority' => 1,
                'config' => [
                    'max_participants' => 100,
                    'allow_file_upload' => true
                ]
            ]);
        }

        return $channel;
    }


    public function createInternalChat(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'subject' => 'required|string|max:255',
            'participants' => 'required|array|min:1',
            'participants.*' => 'integer|exists:users,id',
            'initial_message' => 'nullable|string|max:4000',
            'description' => 'nullable|string|max:500',
            'internal_type' => 'nullable|string|in:discussion,project,support,meeting,announcement,other',
            'priority' => 'nullable|string|in:low,normal,high,urgent'
        ]);

        if ($validator->fails()) {
            return $this->response([
                'errors' => $validator->errors(),
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro de Validação',
                    'message' => 'Dados inválidos para criar chat interno.'
                ]
            ], 422);
        }

        DB::beginTransaction();

        try {
            $currentUserId = auth()->id();
            $participantsIds = array_unique(array_merge($request->input('participants'), [$currentUserId]));
            sort($participantsIds);

            $existingConversation = $this->findSimilarInternalChat(
                $request->input('subject'),
                $participantsIds
            );

            if ($existingConversation) {
                DB::commit();
                return $this->response([
                    'conversation' => $this->formatConversation($existingConversation->load(['contact', 'lastMessage']), $currentUserId),
                    'message' => 'Chat interno similar já existe',
                    'notification' => [
                        'type' => 'info',
                        'title' => 'Chat Existente',
                        'message' => 'Já existe um chat interno similar.'
                    ]
                ]);
            }

            $internalChannel = $this->getOrCreateInternalChannel();

            $conversationData = [
                'chat_channel_id' => $internalChannel->id,
                'type' => 'internal',
                'status' => 'active',
                'assigned_user_id' => $currentUserId,
                'last_message_at' => now(),
                'unread_count' => 0,
                'metadata' => [
                    'subject' => $request->input('subject'),
                    'description' => $request->input('description'),
                    'internal_type' => $request->input('internal_type', 'discussion'),
                    'priority' => $request->input('priority', 'normal'),
                    'created_by' => $currentUserId,
                    'participant_count' => count($participantsIds)
                ]
            ];

            $conversation = Conversation::create($conversationData);

            $participantsData = [];
            foreach ($participantsIds as $userId) {
                $participantsData[$userId] = [
                    'role' => $userId === $currentUserId ? 'admin' : 'participant',
                    'is_active' => true,
                    'joined_at' => now(),
                    'created_at' => now(),
                    'updated_at' => now()
                ];
            }

            $conversation->participants()->attach($participantsData);

            if ($request->filled('initial_message')) {
                $this->sendInitialMessage($conversation, $request->input('initial_message'));
            }

            $this->sendSystemMessage($conversation, $participantsIds);

            $conversation->load(['contact', 'participants', 'lastMessage']);
            DB::commit();

            return $this->response([
                'conversation' => $this->formatConversation($conversation, $currentUserId),
                'message' => 'Chat interno criado com sucesso',
                'notification' => [
                    'type' => 'success',
                    'title' => 'Sucesso',
                    'message' => 'Chat interno criado com sucesso!'
                ]
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Erro ao criar chat interno', [
                'subject' => $request->input('subject'),
                'error' => $e->getMessage()
            ]);

            return $this->response([
                'error' => 'Erro interno do servidor',
                'details' => config('app.debug') ? $e->getMessage() : null,
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao criar chat interno.'
                ]
            ], 500);
        }
    }

    private function findSimilarInternalChat(string $subject, array $participantsIds): ?Conversation
    {
        try {
            $conversations = Conversation::where('type', 'internal')
                ->where('status', 'active')
                ->where('metadata->subject', $subject)
                ->with(['participants' => function ($query) {
                    $query->wherePivot('is_active', true);
                }])
                ->get();

            foreach ($conversations as $conversation) {
                $existingParticipantIds = $conversation->participants->pluck('id')->sort()->values()->toArray();
                $newParticipantIds = collect($participantsIds)->sort()->values()->toArray();

                if ($existingParticipantIds === $newParticipantIds) {
                    return $conversation;
                }
            }

            return null;
        } catch (\Exception $e) {
            Log::error('Erro ao buscar chat similar', [
                'subject' => $subject,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    private function getOrCreateInternalChannel(): ChatChannel
    {
        $channel = ChatChannel::where('type', 'internal_chat')
            ->where('is_active', true)
            ->first();

        if (!$channel) {
            $channel = ChatChannel::create([
                'name' => 'Chat Interno',
                'type' => 'internal_chat',
                'description' => 'Canal para chats internos entre usuários',
                'is_active' => true,
                'priority' => 1,
                'config' => [
                    'max_participants' => 50,
                    'allow_file_upload' => true,
                    'typing_timeout' => 3000,
                    'auto_archive_days' => 30
                ]
            ]);
        }

        return $channel;
    }

   private function sendInitialMessage(Conversation $conversation, string $message): void
    {
        try {
            $conversation->messages()->create([
                'sender_id' => auth()->id(),
                'content' => $message,
                'message_type' => 'text',
                'status' => 'sent',
                'sent_at' => now(),
                'metadata' => [
                    'is_initial_message' => true
                ]
            ]);

            // ✅ ATUALIZAR ÚLTIMA MENSAGEM DA CONVERSA
            $conversation->update([
                'last_message' => $message,
                'last_message_at' => now()
            ]);

        } catch (\Exception $e) {
            Log::warning('⚠️ Falha ao enviar mensagem inicial', [
                'conversation_id' => $conversation->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    private function sendSystemMessage(Conversation $conversation, array $participantsIds): void
    {
        try {
            $creator = auth()->user();
            $participantNames = User::whereIn('id', $participantsIds)
                ->where('id', '!=', auth()->id())
                ->pluck('name')
                ->toArray();

            $systemMessage = sprintf(
                '%s criou este chat interno com: %s',
                $creator->name,
                implode(', ', $participantNames)
            );

            Message::create([
                'conversation_id' => $conversation->id,
                'sender_type' => 'system',
                'sender_id' => null,
                'content' => $systemMessage,
                'message_type' => 'system',
                'is_from_contact' => false,
                'status' => 'sent',
                'sent_at' => now(),
                'metadata' => [
                    'is_system_message' => true,
                    'system_type' => 'conversation_created',
                    'participants' => $participantsIds
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Erro ao enviar mensagem de sistema', ['error' => $e->getMessage()]);
        }
    }

    public function show($conversationId): JsonResponse
    {
        try {
            $conversation = Conversation::with([
                'contact.companies:id,name',
                'contact',
                'assignedUser:id,name,email,avatar',
                'lastMessage',
                'participants:id,name,email,avatar'
            ])->findOrFail($conversationId);

            return $this->response($this->formatConversation($conversation, auth()->id()));
        } catch (\Exception $e) {
            Log::error('Erro ao carregar conversa', ['error' => $e->getMessage()]);
            return $this->response([
                'error' => 'Conversa não encontrada',
                'notification' => ['type' => 'error', 'title' => 'Erro', 'message' => 'Conversa não encontrada.']
            ], 404);
        }
    }

    public function update(Request $request, Conversation $conversation): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'is_pinned' => 'nullable|boolean',
            'is_muted' => 'nullable|boolean',
            'subject' => 'nullable|string|max:255',
            'description' => 'nullable|string|max:500',
            'priority' => 'nullable|string|in:low,normal,high,urgent',
            'assigned_user_id' => 'nullable|integer|exists:users,id',
        ]);

        if ($validator->fails()) {
            return $this->response([
                'errors' => $validator->errors(),
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro de Validação',
                    'message' => 'Dados inválidos para atualizar conversa.'
                ]
            ], 422);
        }

        try {
            $updated = false;
            $dataToUpdate = [];

            if ($request->has('is_pinned')) {
                $dataToUpdate['is_pinned'] = $request->boolean('is_pinned');
                $updated = true;
            }

            if ($request->has('is_muted')) {
                $dataToUpdate['is_muted'] = $request->boolean('is_muted');
                $updated = true;
            }

            if ($request->has('assigned_user_id')) {
                $dataToUpdate['assigned_user_id'] = $request->input('assigned_user_id');
                $updated = true;
            }

            if ($conversation->type === 'internal') {
                $metadata = $conversation->metadata;

                if ($request->has('subject')) {
                    $metadata['subject'] = $request->input('subject');
                    $updated = true;
                }
                if ($request->has('description')) {
                    $metadata['description'] = $request->input('description');
                    $updated = true;
                }
                if ($request->has('priority')) {
                    $metadata['priority'] = $request->input('priority');
                    $updated = true;
                }

                if ($updated && count($metadata) > 0) {
                    $dataToUpdate['metadata'] = $metadata;
                }
            }

            if ($updated) {
                $conversation->update($dataToUpdate);
            }

            $notificationMessage = 'Conversa atualizada com sucesso.';
            if ($request->has('is_pinned')) {
                $notificationMessage = $request->boolean('is_pinned') ? 'Conversa fixada com sucesso.' : 'Conversa desafixada com sucesso.';
            } elseif ($request->has('is_muted')) {
                $notificationMessage = $request->boolean('is_muted') ? 'Conversa silenciada com sucesso.' : 'Notificações reativadas com sucesso.';
            }

            return $this->response([
                'success' => true,
                'conversation' => $this->formatConversation($conversation->fresh(), auth()->id()),
                'message' => $notificationMessage,
                'notification' => [
                    'type' => 'success',
                    'title' => 'Sucesso',
                    'message' => $notificationMessage
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Erro ao atualizar conversa', [
                'conversation_id' => $conversation->id,
                'error' => $e->getMessage()
            ]);
            return $this->response([
                'error' => 'Erro interno do servidor',
                'details' => config('app.debug') ? $e->getMessage() : null,
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao atualizar conversa.'
                ]
            ], 500);
        }
    }

    public function updateStatus(Request $request, $conversationId): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'status' => 'required|in:active,closed,waiting,archived,pending',
            'reason' => 'nullable|string|max:500'
        ]);

        if ($validator->fails()) {
            return $this->response(['errors' => $validator->errors()], 422);
        }

        try {
            $conversation = Conversation::findOrFail($conversationId);
            $oldStatus = $conversation->status;
            $newStatus = $request->input('status');

            $updateData = ['status' => $newStatus];

            if ($newStatus === 'closed') {
                $updateData['closed_at'] = now();
                $updateData['closed_by'] = auth()->id();
                $updateData['close_reason'] = $request->input('reason');
            } else {
                $updateData['closed_at'] = null;
                $updateData['closed_by'] = null;
                $updateData['close_reason'] = null;
            }

            $conversation->update($updateData);
            $this->sendSystemStatusMessage($conversation, $newStatus, $request->input('reason'), auth()->id());

            try {
                if (class_exists('\Modules\Chat\Events\ConversationStatusUpdated')) {
                    broadcast(new \Modules\Chat\Events\ConversationStatusUpdated(
                        $conversation,
                        $oldStatus,
                        $newStatus,
                        auth()->user()
                    ));
                }
            } catch (\Exception $broadcastError) {
                Log::warning('Erro no broadcast de status', [
                    'conversation_id' => $conversationId,
                    'error' => $broadcastError->getMessage()
                ]);
            }

            $notificationMessage = match ($newStatus) {
                'archived' => 'Conversa arquivada com sucesso!',
                'closed' => 'Conversa finalizada com sucesso!',
                default => 'Status da conversa atualizado com sucesso!'
            };

            return $this->response([
                'success' => true,
                'conversation' => $this->formatConversation($conversation->fresh(), auth()->id()),
                'old_status' => $oldStatus,
                'new_status' => $newStatus,
                'message' => $notificationMessage,
                'notification' => [
                    'type' => 'success',
                    'title' => 'Sucesso',
                    'message' => $notificationMessage
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Erro ao atualizar status', [
                'conversation_id' => $conversationId,
                'error' => $e->getMessage()
            ]);
            return $this->response([
                'error' => 'Erro interno do servidor',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao atualizar status da conversa.'
                ]
            ], 500);
        }
    }

    public function assign(Request $request, $conversationId): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'nullable|integer|exists:users,id'
        ]);

        if ($validator->fails()) {
            return $this->response(['errors' => $validator->errors()], 422);
        }

        try {
            $conversation = Conversation::findOrFail($conversationId);
            $conversation->update(['assigned_user_id' => $request->input('user_id')]);
            $conversation->load('assignedUser');

            return $this->response([
                'success' => true,
                'assigned_to' => $conversation->assignedUser ? [
                    'id' => $conversation->assignedUser->id,
                    'name' => $conversation->assignedUser->name
                ] : null,
                'message' => 'Atribuição atualizada com sucesso.',
                'notification' => [
                    'type' => 'success',
                    'title' => 'Sucesso',
                    'message' => 'Atribuição atualizada com sucesso.'
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Erro ao atribuir conversa', ['error' => $e->getMessage()]);
            return $this->response([
                'error' => 'Erro interno do servidor',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao atribuir conversa.'
                ]
            ], 500);
        }
    }

    public function markAsRead(Request $request, $conversationId): JsonResponse
    {
        try {
            $conversation = Conversation::findOrFail($conversationId);
            $userId = auth()->id();

            $updatedCount = Message::where('conversation_id', $conversationId)
                ->where('sender_id', '!=', $userId)
                ->whereNull('read_at')
                ->update(['read_at' => now()]);

            $conversation->update(['unread_count' => 0]);

            $totalUnread = Message::where('conversation_id', $conversationId)
                ->where('sender_id', '!=', $userId)
                ->whereNull('read_at')
                ->count();

            return $this->response([
                'success' => true,
                'messages_marked' => $updatedCount,
                'conversation_id' => $conversationId,
                'total_unread' => $totalUnread,
                'notification' => [
                    'type' => 'success',
                    'title' => 'Sucesso',
                    'message' => 'Conversa marcada como lida.'
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Erro ao marcar como lida', ['error' => $e->getMessage()]);
            return $this->response([
                'error' => 'Erro interno do servidor',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao marcar conversa como lida.'
                ]
            ], 500);
        }
    }

    public function getUsers(): JsonResponse
    {
        try {
            $users = User::select(['id', 'name', 'email', 'avatar'])
                ->orderBy('name')
                ->get()
                ->map(function ($user) {
                    return [
                        'id' => $user->id,
                        'name' => $user->name,
                        'email' => $user->email,
                        'avatar' => $user->avatar ?? null,
                        'is_super_admin' => $user->super_admin ?? false,
                        'has_api_access' => $user->access_api ?? false
                    ];
                });

            return $this->response($users);
        } catch (\Exception $e) {
            Log::error('Erro ao carregar usuários', ['error' => $e->getMessage()]);
            return $this->response([
                'error' => 'Erro interno do servidor',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao carregar usuários.'
                ]
            ], 500);
        }
    }

    public function getStats(): JsonResponse
    {
        try {
            $stats = [
                'total_conversations' => Conversation::count(),
                'active_conversations' => Conversation::where('status', 'active')->count(),
                'pending_conversations' => Conversation::where('status', 'pending')->count(),
                'closed_conversations' => Conversation::where('status', 'closed')->count(),
                'archived_conversations' => Conversation::where('status', 'archived')->count(),
                'total_messages' => Message::count(),
                'messages_today' => Message::whereDate('sent_at', today())->count(),
                'my_active_conversations' => Conversation::where('assigned_user_id', auth()->id())->where('status', 'active')->count(),
            ];

            return $this->response($stats);
        } catch (\Exception $e) {
            Log::error('Erro ao carregar estatísticas', ['error' => $e->getMessage()]);
            return $this->response([
                'error' => 'Erro interno do servidor',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao carregar estatísticas.'
                ]
            ], 500);
        }
    }

    public function summary(Request $request)
    {
        try {
            $userId = auth()->id();

            $query = Conversation::query()->where(function ($q) use ($userId) {
                $q->where('type', 'external')
                    ->orWhere(function ($subQuery) use ($userId) {
                        $subQuery->where('type', 'internal')
                            ->whereHas('participants', fn($p) => $p->where('user_id', $userId));
                    });
            });

            if ($request->has('status')) {
                $query->where('status', $request->input('status'));
            }

            $totalConversations = $query->count();

            $unreadCount = Message::whereHas('conversation', function ($q) use ($userId, $request) {
                $q->where(function ($q_conv) use ($userId) {
                    $q_conv->where('type', 'external')
                       ->orWhere(function ($subQuery) use ($userId) {
                        $subQuery->where('type', 'internal')
                            ->whereHas('participants', fn($p) => $p->where('users.id', $userId));
                    });
                });
                if ($request->has('status')) {
                    $q->where('status', $request->input('status'));
                }
            })
                ->where('sender_id', '!=', $userId)
                ->whereNull('read_at')
                ->count();

            return $this->response([
                'unread_count' => $unreadCount,
                'total_conversations' => $totalConversations,
                'total_archived' => Conversation::where('status', 'archived')->count(),
            ]);
        } catch (\Exception $e) {
            return $this->response([
                'error' => 'Erro ao obter resumo',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao obter resumo das conversas.'
                ]
            ], 500);
        }
    }

    public function export(Request $request, Conversation $conversation): JsonResponse
    {
        try {
            $downloadUrl = '/api/chat/conversations/' . $conversation->id . '/download?format=' . $request->input('format', 'pdf');

            return $this->response([
                'success' => true,
                'download_url' => $downloadUrl,
                'message' => 'Exportação iniciada, o download deve começar em breve.',
                'notification' => [
                    'type' => 'success',
                    'title' => 'Sucesso',
                    'message' => 'Download da conversa iniciado!'
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Erro ao exportar conversa', ['error' => $e->getMessage()]);
            return $this->response([
                'error' => 'Erro interno do servidor',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao exportar conversa.'
                ]
            ], 500);
        }
    }

    public function destroy(Conversation $conversation): JsonResponse
    {
        try {
            DB::beginTransaction();

            Message::where('conversation_id', $conversation->id)->delete();
            $conversation->participants()->detach();
            $conversation->delete();

            DB::commit();

            return $this->response([
                'success' => true,
                'message' => 'Conversa excluída com sucesso',
                'notification' => [
                    'type' => 'success',
                    'title' => 'Sucesso',
                    'message' => 'Conversa excluída permanentemente.'
                ]
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Erro ao excluir conversa', ['error' => $e->getMessage()]);
            return $this->response([
                'error' => 'Erro interno do servidor',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao excluir conversa.'
                ]
            ], 500);
        }
    }

    private function findOrCreateContact($phone, $name = null)
    {
        try {
            $cleanedPhone = $this->cleanPhone($phone);

            $contact = Contact::whereHas('phones', function ($query) use ($cleanedPhone) {
                $query->where('number', $cleanedPhone);
            })->first();

            if (!$contact && $name) {
                $nameParts = explode(' ', trim($name), 2);

                $contact = Contact::create([
                    'first_name' => $nameParts[0],
                    'last_name' => $nameParts[1] ?? '',
                    'user_id' => auth()->id(),
                    'created_by' => auth()->id()
                ]);

                $contact->phones()->create([
                    'number' => $cleanedPhone,
                    'type' => PhoneType::mobile
                ]);
            }

            return $contact;
        } catch (\Exception $e) {
            Log::error('Erro ao criar contato', [
                'phone' => $phone,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    private function findOrCreateChatChannel()
    {
        try {
            $chatChannel = ChatChannel::where('type', 'whatsapp')->first();

            if (!$chatChannel) {
                $chatChannel = ChatChannel::create([
                    'name' => 'WhatsApp Default',
                    'type' => 'whatsapp',
                    'description' => 'Canal WhatsApp padrão para conversas externas',
                    'is_active' => true,
                    'priority' => 1,
                    'config' => ['api_url' => '', 'api_key' => '']
                ]);
            }

            return $chatChannel;
        } catch (\Exception $e) {
            Log::error('Erro ao buscar/criar chat channel', ['error' => $e->getMessage()]);
            return (object) ['id' => 1];
        }
    }

    private function cleanPhone(string $phone): string
    {
        // Remove todos os caracteres não numéricos
        $clean = preg_replace('/[^0-9]/', '', $phone);
        
        // Remove código do país 55 se presente e número tiver mais de 11 dígitos
        if (strlen($clean) > 11 && str_starts_with($clean, '55')) {
            $clean = substr($clean, 2);
        }
        
        return $clean;
    }

 private function isValidPhone(string $phone): bool
    {
        // Telefone brasileiro: 10 ou 11 dígitos
        return strlen($phone) >= 10 && strlen($phone) <= 11 && is_numeric($phone);
    }
 /**
 * Finaliza uma conversa com dados avançados
 */
public function finalize(Request $request, $id): JsonResponse
{
    $validator = Validator::make($request->all(), [
        'status' => 'required|in:closed',
        'reason' => 'required|string|max:255',
        'resolution' => 'nullable|string|max:1000',
        'send_farewell_message' => 'boolean',
        'farewell_message' => 'nullable|string|max:500',
        'archive_conversation' => 'boolean',
        'block_contact' => 'boolean',
        'notify_team' => 'boolean',
        'request_feedback' => 'boolean',
        'closed_at' => 'required|date',
        'finalized_by' => 'nullable|integer|exists:users,id'
    ]);

    if ($validator->fails()) {
        return $this->response([
            'errors' => $validator->errors(),
            'notification' => [
                'type' => 'error',
                'title' => 'Erro de Validação',
                'message' => 'Dados inválidos para finalizar conversa.'
            ]
        ], 422);
    }

    try {
        DB::beginTransaction();

        $conversation = Conversation::findOrFail($id);
        $currentUserId = auth()->id();
        $user = auth()->user();
        
        // ✅ VERIFICAR SE JÁ ESTÁ FECHADA
        if ($conversation->status === 'closed') {
            return $this->response([
                'success' => false,
                'message' => 'Conversa já está fechada',
                'conversation' => $this->formatConversation($conversation, $currentUserId),
                'notification' => [
                    'type' => 'warning',
                    'title' => 'Aviso',
                    'message' => 'Esta conversa já está fechada.'
                ]
            ], 422);
        }

        // ✅ VERIFICAR PERMISSÃO (usando sua lógica existente)
        if ($conversation->assigned_user_id !== $currentUserId && !$user->super_admin) {
            return $this->response([
                'error' => 'Você não tem permissão para finalizar esta conversa.',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Permissão Negada',
                    'message' => 'Você não tem permissão para finalizar esta conversa.'
                ]
            ], 403);
        }

        // ✅ CONSTRUIR CLOSE_REASON COMPLETO
        $closeReasonFull = $request->input('reason');
        if ($request->filled('resolution')) {
            $closeReasonFull .= "\n\nResolução: " . $request->input('resolution');
        }

        // ✅ DETERMINAR STATUS FINAL (fechada ou arquivada)
        $finalStatus = $request->boolean('archive_conversation') ? 'archived' : 'closed';

        // ✅ ATUALIZAR CONVERSA
        $updateData = [
            'status' => $finalStatus,
            'close_reason' => $closeReasonFull,
            'closed_by' => $request->input('finalized_by', $currentUserId),
            'closed_at' => $request->input('closed_at'),
        ];

        $conversation->update($updateData);

        // ✅ AÇÕES ADICIONAIS

        // 1. Mensagem de despedida
        if ($request->boolean('send_farewell_message') && $request->filled('farewell_message') && $conversation->type !== 'internal') {
            $this->sendFarewellMessage($conversation, $request->input('farewell_message'));
        }

        // 2. Bloquear contato (apenas para spam)
        if ($request->boolean('block_contact') && $conversation->contact_id) {
            $this->blockConversationContact($conversation);
        }

        // 3. Notificar equipe
        if ($request->boolean('notify_team')) {
            $this->notifyTeamConversationClosed($conversation, $user, $request->input('reason'));
        }

        // 4. Solicitar feedback
        if ($request->boolean('request_feedback') && $conversation->type !== 'internal') {
            $this->sendFeedbackRequest($conversation);
        }

        // ✅ MENSAGEM DE SISTEMA (usando seu método existente)
        $this->sendSystemStatusMessage($conversation, $finalStatus, $request->input('reason'), $currentUserId);

        DB::commit();

        // ✅ PUBLICAR NO REDIS (usando seu ChatRedisService)
        $chatRedisService = app(ChatRedisService::class);
        $chatRedisService->publishConversationUpdate($conversation->fresh(), 'conversation_finalized');

        // ✅ LOG DA AÇÃO
        Log::info('Conversa finalizada com dados avançados', [
            'conversation_id' => $id,
            'finalized_by' => $user->id,
            'reason' => $request->input('reason'),
            'has_resolution' => $request->filled('resolution'),
            'archived' => $request->boolean('archive_conversation'),
            'farewell_sent' => $request->boolean('send_farewell_message'),
            'contact_blocked' => $request->boolean('block_contact'),
            'team_notified' => $request->boolean('notify_team'),
            'feedback_requested' => $request->boolean('request_feedback')
        ]);

        $notificationMessage = $finalStatus === 'archived' 
            ? 'Conversa finalizada e arquivada com sucesso!'
            : 'Conversa finalizada com sucesso!';

        return $this->response([
            'success' => true,
            'conversation' => $this->formatConversation($conversation->fresh(), $currentUserId),
            'message' => $notificationMessage,
            'notification' => [
                'type' => 'success',
                'title' => 'Atendimento Finalizado',
                'message' => $notificationMessage
            ]
        ]);

    } catch (\Exception $e) {
        DB::rollBack();
        
        Log::error('Erro ao finalizar conversa com dados avançados', [
            'conversation_id' => $id,
            'user_id' => $currentUserId,
            'request_data' => $request->all(),
            'error' => $e->getMessage()
        ]);

        return $this->response([
            'error' => 'Erro interno do servidor',
            'details' => config('app.debug') ? $e->getMessage() : null,
            'notification' => [
                'type' => 'error',
                'title' => 'Erro',
                'message' => 'Falha ao finalizar conversa.'
            ]
        ], 500);
    }
}

/**
 * Reabre uma conversa fechada
 */
public function reopen(Request $request, $id): JsonResponse
{
    $validator = Validator::make($request->all(), [
        'status' => 'required|in:active',
        'reopen_reason' => 'nullable|string|max:255',
        'reopened_at' => 'required|date',
        'reopened_by' => 'nullable|integer|exists:users,id'
    ]);

    if ($validator->fails()) {
        return $this->response([
            'errors' => $validator->errors(),
            'notification' => [
                'type' => 'error',
                'title' => 'Erro de Validação',
                'message' => 'Dados inválidos para reabrir conversa.'
            ]
        ], 422);
    }

    try {
        $conversation = Conversation::findOrFail($id);
        $currentUserId = auth()->id();
        $user = auth()->user();
        
        // ✅ VERIFICAR SE PODE SER REABERTA
        if (!in_array($conversation->status, ['closed', 'archived'])) {
            return $this->response([
                'success' => false,
                'message' => 'Apenas conversas fechadas ou arquivadas podem ser reabertas',
                'notification' => [
                    'type' => 'warning',
                    'title' => 'Aviso',
                    'message' => 'Esta conversa não pode ser reaberta.'
                ]
            ], 422);
        }

        // ✅ VERIFICAR PERMISSÃO
        if ($conversation->assigned_user_id !== $currentUserId && !$user->super_admin) {
            return $this->response([
                'error' => 'Você não tem permissão para reabrir esta conversa.',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Permissão Negada',
                    'message' => 'Você não tem permissão para reabrir esta conversa.'
                ]
            ], 403);
        }

        // ✅ ATUALIZAR CONVERSA
        $conversation->update([
            'status' => 'active',
            'closed_by' => null,
            'closed_at' => null,
            'close_reason' => null,
            'updated_at' => now()
        ]);

        // ✅ MENSAGEM DE SISTEMA
        $reopenReason = $request->input('reopen_reason', 'Reabertura solicitada');
        $this->sendSystemStatusMessage($conversation, 'active', $reopenReason, $currentUserId);

        // ✅ PUBLICAR NO REDIS
        $chatRedisService = app(ChatRedisService::class);
        $chatRedisService->publishConversationUpdate($conversation->fresh(), 'conversation_reopened');

        // ✅ LOG DA REABERTURA
        Log::info('Conversa reaberta', [
            'conversation_id' => $id,
            'reopened_by' => $user->id,
            'reason' => $reopenReason
        ]);

        return $this->response([
            'success' => true,
            'conversation' => $this->formatConversation($conversation->fresh(), $currentUserId),
            'message' => 'Conversa reaberta com sucesso',
            'notification' => [
                'type' => 'success',
                'title' => 'Atendimento Reaberto',
                'message' => 'Conversa reaberta com sucesso!'
            ]
        ]);

    } catch (\Exception $e) {
        Log::error('Erro ao reabrir conversa', [
            'conversation_id' => $id,
            'user_id' => $currentUserId,
            'error' => $e->getMessage()
        ]);
        
        return $this->response([
            'error' => 'Erro interno do servidor',
            'details' => config('app.debug') ? $e->getMessage() : null,
            'notification' => [
                'type' => 'error',
                'title' => 'Erro',
                'message' => 'Falha ao reabrir conversa.'
            ]
        ], 500);
    }
}

/**
 * Envia mensagem de despedida via Evolution API
 */
/**
 * Envia mensagem de despedida via MessageService
 */
private function sendFarewellMessage(Conversation $conversation, string $message): void
{
    try {
        Log::info('Enviando mensagem de despedida', [
            'conversation_id' => $conversation->id,
            'message_length' => strlen($message)
        ]);

        // ✅ USAR SEU MESSAGESERVICE EXISTENTE
        $messageService = app(\Modules\Chat\Services\MessageService::class);
        
        $messageData = [
            'content' => $message,
            'message_type' => 'text',
            'is_system_message' => true,
            'system_type' => 'farewell_message'
        ];

        $result = $messageService->sendMessage($conversation->id, $messageData);

        Log::info('Mensagem de despedida enviada com sucesso', [
            'conversation_id' => $conversation->id,
            'message_id' => $result['message']['id'] ?? null
        ]);
        
    } catch (\Exception $e) {
        Log::error('Erro ao enviar mensagem de despedida', [
            'conversation_id' => $conversation->id,
            'error' => $e->getMessage()
        ]);
    }
}
/**
 * Bloqueia contato da conversa
 */
private function blockConversationContact(Conversation $conversation): void
{
    try {
        Log::info('Bloqueando contato da conversa', [
            'conversation_id' => $conversation->id,
            'contact_id' => $conversation->contact_id,
            'phone' => $conversation->contact_phone
        ]);

        // ✅ IMPLEMENTAR BLOQUEIO VIA EVOLUTION API
        if ($conversation->evolution_instance_name && $conversation->whatsapp_remote_jid) {
            $evolutionApi = app(\Modules\Chat\Services\EvolutionApiService::class);
            $instance = \Modules\Chat\Models\EvolutionInstance::where('instance_name', $conversation->evolution_instance_name)->first();
            
            if ($instance) {
                $evolutionApi->setInstance($instance);
                $evolutionApi->blockContact($conversation->whatsapp_remote_jid);
                
                Log::info('Contato bloqueado via Evolution API', [
                    'conversation_id' => $conversation->id,
                    'remote_jid' => substr($conversation->whatsapp_remote_jid, 0, 10) . '****'
                ]);
            }
        }

        // ✅ MARCAR NO BANCO LOCAL
        if ($conversation->contact_id) {
            $conversation->contact()->update([
                'is_blocked' => true,
                'blocked_at' => now(),
                'blocked_reason' => 'Bloqueado durante finalização de conversa'
            ]);
        }
        
    } catch (\Exception $e) {
        Log::error('Erro ao bloquear contato', [
            'conversation_id' => $conversation->id,
            'error' => $e->getMessage()
        ]);
    }
}

/**
 * Notifica equipe sobre fechamento da conversa
 */
private function notifyTeamConversationClosed(Conversation $conversation, User $user, ?string $reason): void
{
    try {
        Log::info('Notificando equipe sobre fechamento', [
            'conversation_id' => $conversation->id,
            'closed_by' => $user->id,
            'reason' => $reason
        ]);

        // ✅ BUSCAR USUÁRIOS PARA NOTIFICAR (supervisores, admins, etc.)
        $usersToNotify = User::where('super_admin', true)
            ->orWhere('id', $conversation->assigned_user_id)
            ->where('id', '!=', $user->id)
            ->get();

        foreach ($usersToNotify as $userToNotify) {
            // ✅ IMPLEMENTAR NOTIFICAÇÃO (email, Slack, etc.)
            // Exemplo: Mail::to($userToNotify->email)->send(new ConversationClosedMail($conversation, $user, $reason));
            
            Log::debug('Notificação enviada para usuário', [
                'user_id' => $userToNotify->id,
                'conversation_id' => $conversation->id
            ]);
        }
        
    } catch (\Exception $e) {
        Log::error('Erro ao notificar equipe', [
            'conversation_id' => $conversation->id,
            'error' => $e->getMessage()
        ]);
    }
}

/**
 * Solicita feedback do cliente usando MessageService
 */
private function sendFeedbackRequest(Conversation $conversation): void
{
    try {
        Log::info('Solicitando feedback do cliente', [
            'conversation_id' => $conversation->id,
            'contact_id' => $conversation->contact_id
        ]);

        // ✅ PREPARAR MENSAGEM DE FEEDBACK
        $feedbackMessage = "Olá! Seu atendimento foi finalizado. Que tal avaliar nosso atendimento? Sua opinião é muito importante para nós! 🌟\n\nClique aqui para avaliar: [LINK_FEEDBACK]";

        // ✅ USAR MESSAGESERVICE
        $messageService = app(\Modules\Chat\Services\MessageService::class);
        
        $messageData = [
            'content' => $feedbackMessage,
            'message_type' => 'text',
            'is_system_message' => true,
            'system_type' => 'feedback_request'
        ];

        $messageService->sendMessage($conversation->id, $messageData);

        // ✅ REGISTRAR SOLICITAÇÃO NO BANCO (opcional)
        // Você pode criar uma tabela feedback_requests para rastrear
        
    } catch (\Exception $e) {
        Log::error('Erro ao solicitar feedback', [
            'conversation_id' => $conversation->id,
            'error' => $e->getMessage()
        ]);
    }
}
}