<?php
// modules/Chat/Http/Controllers/Api/MessageController.php - VERSÃO CLEAN

namespace Modules\Chat\Http\Controllers\Api;

use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Exception;
use Modules\Core\Http\Controllers\ApiController;
use Modules\Chat\Services\MessageService;

class MessageController extends ApiController
{
    protected MessageService $messageService;

    public function __construct(MessageService $messageService)
    {
        $this->middleware('auth:sanctum');
        $this->messageService = $messageService;
    }

    /**
     * ✅ LISTAR MENSAGENS - DELEGADO PARA SERVICE
     */
    public function index(Request $request, int $conversationId): JsonResponse
    {
        try {
            $result = $this->messageService->getMessages($conversationId, $request->all());
            return $this->response($result);
        } catch (Exception $e) {
            Log::error('❌ MessageController: Erro ao carregar mensagens', [
                'conversation_id' => $conversationId,
                'error' => $e->getMessage()
            ]);
            return $this->response(['error' => 'Erro interno do servidor'], 500);
        }
    }

    /**
     * ✅ ENVIAR MENSAGEM - DELEGADO PARA SERVICE
     */
    public function store(Request $request, int $conversationId): JsonResponse
    {
        try {
            $result = $this->messageService->sendMessage(
                $conversationId, 
                $request->all(), 
                $request->allFiles()
            );
            return $this->response($result, 201);
        } catch (Exception $e) {
            Log::error('❌ MessageController: Erro ao enviar mensagem', [
                'conversation_id' => $conversationId,
                'error' => $e->getMessage()
            ]);
            return $this->response(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * ✅ MARCAR COMO LIDA - DELEGADO PARA SERVICE
     */
    public function markAsRead(Request $request, int $conversationId): JsonResponse
    {
        try {
            $result = $this->messageService->markAsRead($conversationId);
            return $this->response($result);
        } catch (Exception $e) {
            Log::error('❌ MessageController: Erro ao marcar como lida', [
                'conversation_id' => $conversationId,
                'error' => $e->getMessage()
            ]);
            return $this->response(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * ✅ INDICADOR DE DIGITAÇÃO - DELEGADO PARA SERVICE
     */
    public function typing(Request $request, int $conversationId): JsonResponse
    {
        try {
            $result = $this->messageService->handleTyping(
                $conversationId, 
                $request->boolean('is_typing', true)
            );
            return $this->response($result);
        } catch (Exception $e) {
            return $this->response(['error' => $e->getMessage()], 500);
        }
    }
}