<?php

namespace Modules\Chat\Http\Traits;

use Carbon\Carbon;
use Modules\Chat\Models\Conversation;
use Modules\Users\Models\User;
use Modules\Contacts\Models\Contact;

trait FormatsConversations
{
    /**
     * ✅ FORMATAR CONVERSA - VERSÃO COMPATÍVEL COM FRONTEND
     * Este método pode ser usado por qualquer controller que use este trait.
     */
      protected function formatConversation(Conversation $conversation, int $userId): array
    {
        $isInternal = $conversation->type === 'internal';
        
        // ✅ BUSCAR EMPRESA DO CONTATO (SE EXISTIR E JÁ ESTIVER CARREGADA)
        $companyName = null;
        $companyId = null;
        
if ($conversation->contact && $conversation->contact->relationLoaded('companies')) {
    $companies = $conversation->contact->companies;

    if ($companies->isNotEmpty()) {
        // Se um dia você criar o campo is_primary, esse bloco continua funcionando
        $primaryCompany = $companies->first(function ($company) {
            if (!isset($company->pivot)) {
                return false;
            }

            // Só tenta usar is_primary se realmente existir no pivot
            $attributes = method_exists($company->pivot, 'getAttributes')
                ? $company->pivot->getAttributes()
                : [];

            return array_key_exists('is_primary', $attributes) && (bool) $company->pivot->is_primary === true;
        });

        // Se não tiver is_primary ou nenhuma marcada, pega a primeira mesmo
        if (!$primaryCompany) {
            $primaryCompany = $companies->first();
        }

        if ($primaryCompany) {
            $companyName = $primaryCompany->name;
            $companyId   = $primaryCompany->id;
        }
    }
}

        
        // ✅ DADOS BÁSICOS COMUNS
        $baseData = [
            'id' => $conversation->id,
            'type' => $conversation->type,
            'status' => $conversation->status ?? 'active',
            'last_message' => $conversation->lastMessage?->content ?? '',
            'last_message_at' => $this->formatDate($conversation->lastMessage?->sent_at ?? $conversation->updated_at),
            'unread_count' => $this->getUnreadCountForUser($conversation, $userId),
            'created_at' => $this->formatDate($conversation->created_at),
            'assigned_user_id' => $conversation->assigned_user_id,
            'closed_at' => $this->formatDate($conversation->closed_at),
            'closed_by' => $conversation->closed_by,
            'is_pinned' => (bool) $conversation->is_pinned,
            'is_muted' => (bool) $conversation->is_muted,
            
            // ✅ DADOS DO USUÁRIO ATRIBUÍDO (COMPATIBILIDADE)
            'assignedUser' => $conversation->assignedUser ? [
                'id' => $conversation->assignedUser->id,
                'name' => $conversation->assignedUser->name,
                'email' => $conversation->assignedUser->email,
                'avatar' => $conversation->assignedUser->avatar
            ] : null,
            
            // ✅ MÚLTIPLAS ATRIBUIÇÕES (NOVA FUNCIONALIDADE)
            'assigned_users' => $conversation->assignedUsers->map(function ($user) {
                return [
                    'user' => [
                        'id' => $user->id,
                        'name' => $user->name,
                        'email' => $user->email,
                        'avatar' => $user->avatar ?? null
                    ],
                    'role' => $user->pivot->role,
                    'is_primary' => $user->pivot->is_primary,
                    'assigned_at' => $this->formatDate($user->pivot->assigned_at)
                ];
            })->toArray(),
            
            // ✅ INFORMAÇÕES DE VISIBILIDADE
            'visibility' => [
                'is_assigned_to_user' => $conversation->assigned_user_id === $userId,
                'is_user_in_assignments' => $conversation->assignedUsers->contains('id', $userId),
                'can_user_view' => $conversation->canUserView($userId),
                'total_assigned_users' => $conversation->assignedUsers->count()
            ],
            
            // ✅ NOME DA EMPRESA (CAMPOS DIRETOS PARA FACILITAR USO NO FRONTEND)
            'company_name' => $companyName,
            'company_id' => $companyId,
        ];
        
        if ($isInternal) {
            $otherParticipants = $conversation->participants
                ->filter(fn($p) => $p->id !== $userId)
                ->values();

            $subject = data_get($conversation->metadata, 'subject', 'Chat Interno');
            $participantNames = $otherParticipants->pluck('name')->join(', ');
            
            return array_merge($baseData, [
                'subject' => $subject,
                'contact_name' => $participantNames ?: $subject,
                'description' => data_get($conversation->metadata, 'description'),
                'internal_type' => data_get($conversation->metadata, 'internal_type', 'discussion'),
                'priority' => data_get($conversation->metadata, 'priority', 'normal'),
                'participants' => $otherParticipants->map(function ($user) {
                    return [
                        'id' => $user->id,
                        'name' => $user->name,
                        'email' => $user->email,
                        'avatar' => $user->avatar ?? null, 
                        'role' => $user->pivot->role ?? 'participant',
                        'is_active' => $user->pivot->is_active ?? true,
                        'joined_at' => $this->formatDate($user->pivot->joined_at),
                        'online' => false
                    ];
                })->toArray(),
                'last_message_type' => 'internal',
                'created_by' => data_get($conversation->metadata, 'created_by'),
                'phone' => null,
                'channel_type' => 'internal',
                
                // ✅ CAMPOS PARA COMPATIBILIDADE COM FRONTEND
                'processed_avatar' => $this->getInternalChatAvatar($otherParticipants),
                'processed_display_name' => $participantNames ?: $subject
            ]);
        } else {
            // ✅ CONVERSA EXTERNA - LÓGICA EXISTENTE MANTIDA
            $contactName = 'Contato sem nome';
            $profilePhoto = null;
            $isWhatsAppBusiness = false;
            $phoneNumber = '';
            $contactEmail = null;

            if ($conversation->contact) {
                if ($conversation->contact->display_name) {
                    $contactName = $conversation->contact->display_name;
                } elseif ($conversation->contact->first_name) {
                    $contactName = trim($conversation->contact->first_name . ' ' . ($conversation->contact->last_name ?? ''));
                }
                
                $profilePhoto = $conversation->contact->best_profile_photo ?? 
                               $conversation->contact->avatar_url;
                $isWhatsAppBusiness = $conversation->contact->is_whatsapp_business ?? false;
                $phoneNumber = $conversation->contact->whatsapp_number ?? $conversation->contact_phone ?? '';
                $contactEmail = $conversation->contact->email; // ✅ OBTÉM EMAIL DIRETAMENTE DO CONTATO
            }
            
            if ($contactName === 'Contato sem nome' && $conversation->whatsapp_profile_name) {
                $contactName = $conversation->whatsapp_profile_name;
            }
            
            if ($contactName === 'Contato sem nome' && !empty($conversation->contact_name) && $conversation->contact_name !== 'Contato sem nome') {
                $contactName = $conversation->contact_name;
            }
            
            if ($contactName === 'Contato sem nome') {
                $phone = $phoneNumber ?: $conversation->contact_phone ?: '';
                if ($phone) {
                    $contactName = $this->formatPhoneForDisplay($phone);
                }
            }
            
                     return array_merge($baseData, [
                'contact_name' => $contactName,
                'phone' => $phoneNumber ?: $conversation->contact_phone ?: '',
                'avatar_url' => $profilePhoto, 
                'is_whatsapp_business' => (bool)$isWhatsAppBusiness, 
                'whatsapp_profile_picture' => $conversation->whatsapp_profile_picture,
                'last_message_type' => 'external',
                'channel_type' => $conversation->chatChannel?->type ?? 'evolution_api',
                'subject' => null,
                'participants' => [],
                'contact' => $conversation->contact ? [
                    'id' => $conversation->contact->id,
                    'display_name' => $conversation->contact->display_name,
                    'first_name' => $conversation->contact->first_name,
                    'last_name' => $conversation->contact->last_name,
                    'whatsapp_number' => $conversation->contact->whatsapp_number,
                    'email' => $contactEmail,
                    'best_profile_photo' => $conversation->contact->best_profile_photo,
                    'profile_photo_source' => $conversation->contact->profile_photo_source,
                    
                    // ✅ DADOS DA EMPRESA
                    'company' => $companyName ? [
                        'id' => $companyId,
                        'name' => $companyName,
                    ] : null,
                ] : null,
                
                // ✅ CAMPOS PARA COMPATIBILIDADE COM FRONTEND
                'processed_avatar' => $profilePhoto,
                'processed_display_name' => $contactName
            ]);
        }
    }

    /**
     * ✅ OBTER AVATAR PARA CHAT INTERNO
     */
    protected function getInternalChatAvatar($participants): ?string
    {
        if ($participants->isEmpty()) {
            return null;
        }
        
        // Usar avatar do primeiro participante
        $firstParticipant = $participants->first();
        return $firstParticipant->avatar ?? null;
    }

    /**
     * ✅ FORMATAR TELEFONE PARA EXIBIÇÃO
     */
    protected function formatPhoneForDisplay(string $phone): string
    {
        $clean = preg_replace('/[^0-9]/', '', $phone);
        
        if (strlen($clean) === 13 && str_starts_with($clean, '55')) {
            // Formato: 5566999887766 -> +55 (66) 99988-7766
            $ddd = substr($clean, 2, 2);
            $numero = substr($clean, 4);
            if (strlen($numero) === 9) {
                return "+55 ({$ddd}) " . substr($numero, 0, 5) . "-" . substr($numero, 5);
            }
        } elseif (strlen($clean) === 11) {
            // Formato: 66999887766 -> (66) 99988-7766
            $ddd = substr($clean, 0, 2);
            $numero = substr($clean, 2);
            if (strlen($numero) === 9) {
                return "({$ddd}) " . substr($numero, 0, 5) . "-" . substr($numero, 5);
            }
        } elseif (strlen($clean) === 10) {
            // Formato: 6699887766 -> (66) 9988-7766
            $ddd = substr($clean, 0, 2);
            $numero = substr($clean, 2);
            return "({$ddd}) " . substr($numero, 0, 4) . "-" . substr($numero, 4);
        }
        
        return $phone;
    }

    /**
     * ✅ MÉTODO AUXILIAR PARA FORMATAR DATAS
     */
    protected function formatDate($date): ?string
    {
        if (!$date) {
            return null;
        }

        if (is_string($date)) {
            try {
                $carbonDate = Carbon::parse($date);
                return $carbonDate->toISOString();
            } catch (\Exception $e) {
                return $date;
            }
        }

        if ($date instanceof Carbon || $date instanceof \DateTime) {
            return $date->toISOString();
        }

        try {
            return Carbon::parse($date)->toISOString();
        } catch (\Exception $e) {
            // Log::warning('⚠️ Não foi possível formatar data:', ['date' => $date, 'type' => gettype($date), 'error' => $e->getMessage()]);
            return null;
        }
    }

    /**
     * ✅ CONTAR MENSAGENS NÃO LIDAS PARA USUÁRIO ESPECÍFICO
     */
    protected function getUnreadCountForUser(Conversation $conversation, int $userId): int
    {
        if ($conversation->type !== 'internal') {
            return $conversation->unread_count ?? 0;
        }

        // Contar mensagens não lidas para o usuário na conversa interna
        return \Modules\Chat\Models\Message::where('conversation_id', $conversation->id)
            ->where('sender_id', '!=', $userId)
            ->whereNull('read_at')
            ->count();
    }
}