<?php

namespace Modules\Chat\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Modules\Contacts\Models\Contact;
use Modules\Chat\Models\EvolutionInstance;
use Modules\Chat\Models\Conversation;
use Modules\Chat\Services\ContactProfilePhotoService;
use Exception;

class FetchContactPhotoJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected Contact $contact;
    protected string $phoneNumber;
    protected int $conversationId;
    protected bool $forceUpdate;

    public int $tries = 3;
    public int $timeout = 120;
    public array $backoff = [30, 60, 120];

    public function __construct(Contact $contact, string $phoneNumber, int $conversationId, bool $forceUpdate = false)
    {
        $this->contact = $contact;
        $this->phoneNumber = $phoneNumber;
        $this->conversationId = $conversationId;
        $this->forceUpdate = $forceUpdate;
        $this->onQueue('photos');
    }

    public function handle(): void
    {
        $cacheKey = "photo_job_pending_{$this->contact->id}";
        
        try {
            Log::info('📸 [JOB] Iniciando execução do job de foto', [
                'contact_id' => $this->contact->id,
                'contact_name' => $this->contact->first_name,
                'phone' => $this->maskPhone($this->phoneNumber),
                'conversation_id' => $this->conversationId,
                'attempt' => $this->attempts(),
                'force_update' => $this->forceUpdate,
                'job_id' => $this->job->getJobId() ?? 'unknown'
            ]);

            // ✅ REFRESH DO CONTATO
            $this->contact->refresh();
            
            // ✅ VERIFICAR SE AINDA PRECISA (COM OPÇÃO DE FORÇAR)
            if (!$this->forceUpdate && !$this->contact->needsWhatsappProfilePhotoUpdate()) {
                Log::info('📸 [JOB] Foto já atualizada, job cancelado', [
                    'contact_id' => $this->contact->id,
                    'last_update' => $this->contact->whatsapp_profile_photo_updated_at,
                    'has_photo' => !empty($this->contact->whatsapp_profile_photo_path)
                ]);
                return;
            }

            // ✅ BUSCAR E VALIDAR CONVERSA
            $conversation = Conversation::with(['chatChannel'])->find($this->conversationId);
            if (!$conversation) {
                throw new Exception("Conversa não encontrada: {$this->conversationId}");
            }

            if (!$conversation->chatChannel) {
                throw new Exception("Chat channel não encontrado para conversa: {$this->conversationId}");
            }

            // ✅ EXTRAIR INSTANCE_NAME
            $channelConfig = $conversation->chatChannel->config ?? [];
            $instanceName = $channelConfig['instance_name'] ?? null;
            
            if (!$instanceName) {
                throw new Exception("Instance name não encontrado no config do canal: {$conversation->chatChannel->id}");
            }

            // ✅ BUSCAR E VALIDAR INSTÂNCIA
            $instance = EvolutionInstance::where('instance_name', $instanceName)->first();
            if (!$instance) {
                throw new Exception("Instância Evolution não encontrada: {$instanceName}");
            }

            // // ✅ VERIFICAR STATUS DA INSTÂNCIA
            // if (!$instance->is_online) {
            //     Log::warning('⚠️ [JOB] Instância offline, reagendando job', [
            //         'instance' => $instanceName,
            //         'contact_id' => $this->contact->id,
            //         'conversation_id' => $this->conversationId,
            //         'attempt' => $this->attempts()
            //     ]);
                
            //     // Reagendar com delay maior se muitas tentativas
            //     $delay = $this->attempts() > 1 ? 600 : 300; // 10min ou 5min
            //     $this->release($delay);
            //     return;
            // }

            // ✅ EXECUTAR BUSCA DE FOTO
            $photoService = app(ContactProfilePhotoService::class);
            
            Log::info('📸 [JOB] Executando busca de foto de perfil', [
                'contact_id' => $this->contact->id,
                'contact_name' => $this->contact->first_name,
                'phone' => $this->maskPhone($this->phoneNumber),
                'instance' => $instanceName,
                'conversation_id' => $this->conversationId,
                'attempt' => $this->attempts()
            ]);

            $success = $photoService->fetchAndSaveContactProfilePhoto(
                $this->contact, 
                $this->phoneNumber, 
                $instance
            );

            if ($success) {
                Log::info('✅ [JOB] Foto de perfil obtida com sucesso', [
                    'contact_id' => $this->contact->id,
                    'contact_name' => $this->contact->first_name,
                    'phone' => $this->maskPhone($this->phoneNumber),
                    'instance' => $instanceName,
                    'conversation_id' => $this->conversationId,
                    'photo_path' => $this->contact->fresh()->whatsapp_profile_photo_path
                ]);
            } else {
                Log::info('📸 [JOB] Nenhuma foto encontrada para contato', [
                    'contact_id' => $this->contact->id,
                    'phone' => $this->maskPhone($this->phoneNumber),
                    'instance' => $instanceName,
                    'conversation_id' => $this->conversationId
                ]);
            }

        } catch (Exception $e) {
            Log::error('❌ [JOB] Erro no job de busca de foto', [
                'contact_id' => $this->contact->id,
                'phone' => $this->maskPhone($this->phoneNumber),
                'conversation_id' => $this->conversationId,
                'attempt' => $this->attempts(),
                'max_tries' => $this->tries,
                'error' => $e->getMessage(),
                'error_line' => $e->getLine(),
                'error_file' => basename($e->getFile())
            ]);

            // ✅ SE ÚLTIMA TENTATIVA, MARCAR ERRO
            if ($this->attempts() >= $this->tries) {
                $this->contact->update([
                    'whatsapp_profile_photo_updated_at' => now(),
                    'whatsapp_profile_metadata' => [
                        'last_check' => now(),
                        'has_photo' => false,
                        'error' => $e->getMessage(),
                        'phone_checked' => $this->phoneNumber,
                        'job_failed' => true,
                        'attempts_made' => $this->attempts()
                    ]
                ]);
                
                Log::error('❌ [JOB] Marcando contato com erro após esgotar tentativas', [
                    'contact_id' => $this->contact->id,
                    'attempts_made' => $this->attempts()
                ]);
            }

            throw $e;
        } finally {
            // ✅ SEMPRE LIMPAR CACHE NO FINAL
            Cache::forget($cacheKey);
            
            Log::debug('🧹 [JOB] Cache de job pendente limpo', [
                'contact_id' => $this->contact->id,
                'cache_key' => $cacheKey
            ]);
        }
    }

    public function failed(\Throwable $exception): void
    {
        $cacheKey = "photo_job_pending_{$this->contact->id}";
        Cache::forget($cacheKey);
        
        Log::error('❌ [JOB] Job de foto falhou permanentemente', [
            'contact_id' => $this->contact->id,
            'phone' => $this->maskPhone($this->phoneNumber),
            'conversation_id' => $this->conversationId,
            'error' => $exception->getMessage(),
            'attempts_made' => $this->attempts()
        ]);
    }

    private function maskPhone(string $phone): string
    {
        if (strlen($phone) >= 8) {
            return substr($phone, 0, 4) . '****' . substr($phone, -2);
        }
        return '****';
    }
}