<?php
// modules/Chat/Jobs/SendWhatsAppMessageJob.php

namespace Modules\Chat\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Modules\Chat\Models\Message;
use Modules\Chat\Models\Conversation;
use Modules\Chat\Services\EvolutionApiService;
use Modules\Users\Models\User;
use Exception;

class SendWhatsAppMessageJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 3;
    public int $timeout = 60;
    public array $backoff = [10, 30, 60];

    public function __construct(
        public Message $message,
        public Conversation $conversation,
        public ?string $attachmentPath = null
    ) {
        $this->onQueue('whatsapp-messages');
        $this->delay = now()->addSeconds($this->conversation->id % 5);
    }

    public function handle(): void
    {
        try {
            // ✅ BUSCAR E VALIDAR INSTÂNCIA
            $instance = $this->getEvolutionInstance();
            if (!$instance || !$instance->is_online) {
                $this->failWithError('Instância Evolution não encontrada ou offline');
                return;
            }

            // ✅ OBTER E VALIDAR NÚMERO
            $targetNumber = $this->getTargetNumber();
            if (!$targetNumber) {
                $this->failWithError('Número de destino não encontrado');
                return;
            }

            $whatsappValidation = $this->validateWhatsAppNumber($targetNumber);
            if (!$whatsappValidation['exists']) {
                $this->failWithError("Número WhatsApp inválido: " . substr($targetNumber, 0, 6) . '****');
                return;
            }

            // ✅ ENVIAR MENSAGEM COM PREFIXO
            $evolutionApi = app(EvolutionApiService::class)->setInstance($instance);
            $targetJid = $whatsappValidation['jid'] ?? $targetNumber;

            $result = $this->sendMessage($evolutionApi, $targetJid);
            $this->processResponse($result, $whatsappValidation);

            Log::info('✅ Mensagem WhatsApp enviada', [
                'message_id' => $this->message->id,
                'whatsapp_id' => $result['key']['id'] ?? null,
                'has_user_prefix' => $this->shouldAddUserPrefix()
            ]);

        } catch (Exception $e) {
            Log::error('❌ Erro no Job WhatsApp', [
                'message_id' => $this->message->id,
                'error' => $e->getMessage()
            ]);

            $this->message->update(['delivery_status' => 'failed', 'status' => 'failed']);
            throw $e;
        }
    }

    // ===== MÉTODOS PARA PREFIXO DO USUÁRIO =====

    /**
     * ✅ PREPARAR CONTEÚDO DA MENSAGEM COM PREFIXO DO USUÁRIO
     */
    private function prepareMessageContent(): string
    {
        $originalContent = $this->message->content;

        // Verificar se deve adicionar prefixo
        if (!$this->shouldAddUserPrefix()) {
            return $originalContent;
        }

        // Obter nome do usuário
        $userName = $this->getUserName();
        if (!$userName) {
            return $originalContent;
        }

        // ✅ ADICIONAR PREFIXO COM NOME DO USUÁRIO
        $prefix = "*{$userName}*";
        
        // Verificar se já tem o prefixo para evitar duplicação
        if (str_starts_with(trim($originalContent), $prefix)) {
            return $originalContent;
        }

        return $prefix . "\n" . $originalContent;
    }

    /**
     * ✅ VERIFICAR SE DEVE ADICIONAR PREFIXO DO USUÁRIO
     */
    private function shouldAddUserPrefix(): bool
    {
        // Só adicionar prefixo se:
        // 1. Mensagem foi enviada por um usuário (não sistema)
        // 2. Não é uma mensagem de sistema
        // 3. Configuração está habilitada
        
        if ($this->message->sender_type !== '\Modules\Users\Models\User') {
            return false;
        }

        if ($this->message->message_type === 'system') {
            return false;
        }

        // Verificar configuração global
        $addPrefix = config('chat.add_user_prefix_to_messages', true);
        if (!$addPrefix) {
            return false;
        }

        // Verificar se o usuário tem preferência específica
        $user = $this->getUser();
        if ($user && isset($user->preferences['chat_add_name_prefix'])) {
            return (bool) $user->preferences['chat_add_name_prefix'];
        }

        return true;
    }

    /**
     * ✅ OBTER NOME DO USUÁRIO
     */
    private function getUserName(): ?string
    {
        $user = $this->getUser();
        if (!$user) {
            return null;
        }

        // Prioridade: first_name > name > email (parte antes do @)
        if (!empty($user->first_name)) {
            return $user->first_name;
        }

        if (!empty($user->name)) {
            // Se name contém espaço, pegar só o primeiro nome
            $nameParts = explode(' ', trim($user->name));
            return $nameParts[0];
        }

        if (!empty($user->email)) {
            // Pegar parte antes do @ do email
            $emailParts = explode('@', $user->email);
            return ucfirst($emailParts[0]);
        }

        return null;
    }

    /**
     * ✅ OBTER USUÁRIO QUE ENVIOU A MENSAGEM
     */
    private function getUser(): ?User
    {
        if ($this->message->sender_type !== '\Modules\Users\Models\User' || !$this->message->sender_id) {
            return null;
        }

        return User::find($this->message->sender_id);
    }

    // ===== MÉTODOS EXISTENTES (ATUALIZADOS) =====

    private function getEvolutionInstance(): ?\Modules\Chat\Models\EvolutionInstance
    {
        // Via relacionamento direto
        if ($this->conversation->relationLoaded('evolutionInstance') && $this->conversation->evolutionInstance) {
            return $this->conversation->evolutionInstance;
        }

        // Via campo evolution_instance_name
        if ($this->conversation->evolution_instance_name) {
            return \Modules\Chat\Models\EvolutionInstance::where('instance_name', $this->conversation->evolution_instance_name)->first();
        }

        // Via canal
        if ($this->conversation->chat_channel_id) {
            $channel = $this->conversation->chatChannel ?? $this->conversation->load('chatChannel')->chatChannel;
            
            if ($channel && $channel->type === 'evolution_api') {
                $instanceName = $channel->getConfig('instance_name');
                if ($instanceName) {
                    return \Modules\Chat\Models\EvolutionInstance::where('instance_name', $instanceName)->first();
                }
            }
        }

        return null;
    }

    private function getTargetNumber(): ?string
    {
        // 1. JID existente
        if ($this->conversation->whatsapp_remote_jid) {
            return $this->conversation->whatsapp_remote_jid;
        }

        // 2. Telefone da conversa
        if ($this->conversation->contact_phone) {
            return $this->formatPhone($this->conversation->contact_phone);
        }

        // 3. Telefone do contato
        if ($this->conversation->contact) {
            if ($this->conversation->contact->whatsapp_number) {
                return $this->formatPhone($this->conversation->contact->whatsapp_number);
            }
            
            $firstPhone = $this->conversation->contact->phones?->first();
            if ($firstPhone) {
                return $this->formatPhone($firstPhone->number);
            }
        }

        return null;
    }

    private function formatPhone(string $phone): string
    {
        $phone = preg_replace('/[^0-9]/', '', $phone);

        // Remove código 55 se presente e número muito longo
        if (strlen($phone) >= 13 && str_starts_with($phone, '55')) {
            $phone = substr($phone, 2);
        }

        // Adiciona código do país se necessário
        if (strlen($phone) == 10 || strlen($phone) == 11) {
            $phone = '55' . $phone;
        }

        // Adiciona 9 do celular se necessário
        if (strlen($phone) == 12) {
            $areaCode = substr($phone, 2, 2);
            $number = substr($phone, 4);
            
            if (in_array($number[0], ['6', '7', '8', '9']) && strlen($number) == 8) {
                $phone = '55' . $areaCode . '9' . $number;
            }
        }

        return $phone;
    }

    private function validateWhatsAppNumber(string $number): array
    {
        try {
            $evolutionApi = app(EvolutionApiService::class)->setInstance($this->getEvolutionInstance());
            return $evolutionApi->checkSingleWhatsAppNumber($number);
        } catch (Exception $e) {
            // Se falhar validação, assume que existe
            return ['exists' => true, 'number' => $number, 'jid' => null, 'validation_skipped' => true];
        }
    }

    private function sendMessage(EvolutionApiService $evolutionApi, string $targetJid): array
    {
        // ✅ PREPARAR CONTEÚDO COM PREFIXO DO USUÁRIO
        $messageContent = $this->prepareMessageContent();

        if ($this->attachmentPath) {
            if ($this->message->message_type === 'audio') {
                // Enviar texto separado para áudio (com prefixo se aplicável)
                if ($messageContent) {
                    try {
                        $evolutionApi->sendTextMessage($targetJid, $messageContent);
                    } catch (Exception $e) {
                        Log::warning('Erro ao enviar texto antes do áudio', ['error' => $e->getMessage()]);
                    }
                }
                return $evolutionApi->sendFile($targetJid, $this->attachmentPath, '');
            } else {
                // Para outros tipos de arquivo, usar o conteúdo com prefixo como caption
                return $evolutionApi->sendFile($targetJid, $this->attachmentPath, $messageContent ?: '');
            }
        } else {
            // Mensagem de texto com prefixo
            return $evolutionApi->sendTextMessage($targetJid, $messageContent);
        }
    }

    private function processResponse(array $result, array $validation): void
    {
        $whatsappMessageId = $result['key']['id'] ?? null;
        $remoteJid = $result['key']['remoteJid'] ?? $validation['jid'] ?? null;
        $status = $result['status'] ?? 'PENDING';

        // Atualizar mensagem
        $this->message->update([
            'whatsapp_message_id' => $whatsappMessageId,
            'delivery_status' => $this->mapStatus($status),
            'status' => 'sent'
        ]);

        // Atualizar conversa com JID se não tiver
        $updateData = ['last_message_at' => $this->message->sent_at];
        if ($remoteJid && !$this->conversation->whatsapp_remote_jid) {
            $updateData['whatsapp_remote_jid'] = $remoteJid;
            Log::info('📱 JID capturado', [
                'conversation_id' => $this->conversation->id,
                'jid' => substr($remoteJid, 0, 10) . '****'
            ]);
        }
        
        $this->conversation->update($updateData);
    }

    private function mapStatus(string $status): string
    {
        return match (strtoupper($status)) {
            'PENDING' => 'pending',
            'SENT' => 'sent',
            'DELIVERED' => 'delivered',
            'READ' => 'read',
            'FAILED' => 'failed',
            default => 'sent'
        };
    }

    private function failWithError(string $message): void
    {
        Log::error('❌ Job WhatsApp falhou', [
            'message_id' => $this->message->id,
            'error' => $message
        ]);

        $this->message->update([
            'delivery_status' => 'failed',
            'status' => 'failed'
        ]);

        $this->fail(new Exception($message));
    }

    public function failed(\Throwable $exception): void
    {
        $this->message->update([
            'delivery_status' => 'failed',
            'status' => 'failed'
        ]);
    }
}