<?php
// modules/Chat/Models/Conversation.php - VERSÃO FINAL COMPLETA

namespace Modules\Chat\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\DB;
use Modules\Contacts\Models\Contact;
use Modules\Users\Models\User;

class Conversation extends Model
{
    use SoftDeletes;

    // ✅ CONSTANTES PARA TIPOS DE CONVERSA
    const TYPE_INTERNAL = 'internal';
    const TYPE_EXTERNAL = 'external';
    const TYPE_WHATSAPP = 'whatsapp';
    const TYPE_EMAIL = 'email';
    const TYPE_TELEGRAM = 'telegram';
    const TYPE_WEBCHAT = 'webchat';

    // ✅ CONSTANTES PARA STATUS
    const STATUS_ACTIVE = 'active';
    const STATUS_CLOSED = 'closed';
    const STATUS_PENDING = 'pending';
    const STATUS_ARCHIVED = 'archived';

    protected $fillable = [
        'contact_id', 'chat_channel_id', 'external_id', 'contact_phone',
        'contact_identifier', 'status', 'type', 'assigned_user_id',
        'last_message_at', 'metadata', 'unread_count', 'is_pinned', 
        'is_muted', 'closed_by', 'closed_at', 'close_reason',
        'whatsapp_chat_id', 'whatsapp_remote_jid', 'evolution_instance_name',
        'whatsapp_status', 'last_whatsapp_activity', 'whatsapp_profile_name',
        'whatsapp_profile_picture', 'is_whatsapp_business', 'whatsapp_labels'
    ];

    protected $casts = [
        'last_message_at' => 'datetime',
        'metadata' => 'array',
        'unread_count' => 'integer',
        'is_pinned' => 'boolean',
        'is_muted' => 'boolean',
        'closed_at' => 'datetime',
        'last_whatsapp_activity' => 'datetime',
        'is_whatsapp_business' => 'boolean',
        'whatsapp_labels' => 'array'
    ];

    // ✅ RELACIONAMENTOS
    public function contact(): BelongsTo
    {
        return $this->belongsTo(Contact::class);
    }

    public function chatChannel(): BelongsTo
    {
        return $this->belongsTo(ChatChannel::class);
    }

    public function assignedUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'assigned_user_id');
    }

    public function messages(): HasMany
    {
        return $this->hasMany(Message::class)->orderBy('sent_at');
    }

    public function lastMessage(): HasOne
    {
        return $this->hasOne(Message::class)->latestOfMany('sent_at');
    }

    /**
     * ✅ PARTICIPANTES (PARA CHAT INTERNO)
     */
    public function participants(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'conversation_participants')
                    ->withPivot(['role', 'joined_at', 'left_at', 'last_read_at', 'is_active'])
                    ->withTimestamps();
    }

    /**
     * ✅ MÚLTIPLAS ATRIBUIÇÕES
     */
    public function assignedUsers(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'conversation_assignments')
                    ->withPivot(['role', 'assigned_at', 'assigned_by', 'is_primary'])
                    ->withTimestamps()
                    ->orderByPivot('is_primary', 'desc') // ✅ Garante que o primário venha primeiro
                    ->orderByPivot('assigned_at', 'asc');
    }

        public function primaryAssignedUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'assigned_user_id');
    }

        /**
     * ✅ ATRIBUIR MÚLTIPLOS USUÁRIOS
     */
    public function assignMultipleUsers(array $userIds, array $roles = [], int $assignedBy = null): bool
    {
        try {
            $assignedBy = $assignedBy ?? auth()->id();
            $assignmentData = [];
            $primaryUserId = null;

            foreach ($userIds as $index => $userId) {
                $role = $roles[$index] ?? ($index === 0 ? 'primary' : 'secondary');
                $isPrimary = $role === 'primary';
                
                if ($isPrimary) {
                    $primaryUserId = $userId;
                }

                $assignmentData[$userId] = [
                    'role' => $role,
                    'assigned_at' => now(),
                    'assigned_by' => $assignedBy,
                    'is_primary' => $isPrimary,
                    'created_at' => now(),
                    'updated_at' => now()
                ];
            }

            // ✅ SYNC PARA SUBSTITUIR ATRIBUIÇÕES EXISTENTES
            $this->assignedUsers()->sync($assignmentData);

            // ✅ ATUALIZAR CAMPO PRINCIPAL PARA COMPATIBILIDADE
            $this->update(['assigned_user_id' => $primaryUserId]);

            return true;
        } catch (\Exception $e) {
            \Log::error('Erro ao atribuir múltiplos usuários:', [
                'conversation_id' => $this->id,
                'user_ids' => $userIds,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * ✅ VERIFICAR SE USUÁRIO PODE VER A CONVERSA
     */
    public function canUserView(int $userId): bool
    {
        $user = User::find($userId);
        
        // ✅ SUPERUSER VÊ TUDO
        if ($user && $user->super_admin) {
            return true;
        }

        // ✅ CONVERSA EXTERNA SEM ATRIBUIÇÃO OU COM ATRIBUIÇÃO
        if ($this->isExternal()) {
             // Não atribuída a ninguém, todos podem ver
            if (!$this->assigned_user_id && $this->assignedUsers()->count() === 0) {
                return true;
            }
            // Atribuída a ele (simples)
            if ($this->assigned_user_id === $userId) {
                return true;
            }
            // Atribuída a ele (múltipla)
            if ($this->assignedUsers()->where('user_id', $userId)->exists()) {
                return true;
            }
        }

        // ✅ PARTICIPANTE EM CHAT INTERNO
        if ($this->isInternal()) {
            return $this->participants()
                       ->where('user_id', $userId)
                       ->wherePivot('is_active', true) // ✅ Usar wherePivot
                       ->exists();
        }

        return false;
    }


        /**
     * ✅ OBTER TODOS OS USUÁRIOS QUE PODEM VER A CONVERSA
     */
    public function getViewableUserIds(): array
    {
        $userIds = [];

        // ✅ ATRIBUIÇÃO SIMPLES
        if ($this->assigned_user_id) {
            $userIds[] = $this->assigned_user_id;
        }

        // ✅ MÚLTIPLAS ATRIBUIÇÕES
        $assignedUserIds = $this->assignedUsers()->pluck('user_id')->toArray();
        $userIds = array_merge($userIds, $assignedUserIds);

        // ✅ PARTICIPANTES INTERNOS
        if ($this->type === 'internal') {
            $participantIds = $this->participants()
                ->wherePivot('is_active', true)
                ->pluck('users.id')
                ->toArray();

            $userIds = array_merge($userIds, $participantIds);
        }


        // ✅ SUPERUSERS SEMPRE PODEM VER
        $superUserIds = User::where('super_admin', true)->pluck('id')->toArray();
        $userIds = array_merge($userIds, $superUserIds);

        return array_unique($userIds);
    }

    /**
     * ✅ SCOPE PARA CONVERSAS VISÍVEIS AO USUÁRIO
     */
    public function scopeVisibleToUser($query, int $userId)
    {
        $user = User::find($userId);
        
        // ✅ SUPERUSER VÊ TUDO
        if ($user && $user->super_admin) {
            return $query;
        }

        return $query->where(function ($q) use ($userId) {
            // Conversas externas (sem atribuição ou atribuídas a ele)
            $q->where(function ($subQ) use ($userId) {
                $subQ->where('type', self::TYPE_EXTERNAL) // Ou outros tipos externos
                     ->where(function($assignedQ) use ($userId) {
                         // Não atribuída a ninguém
                         $assignedQ->whereNull('assigned_user_id')
                                   ->whereDoesntHave('assignedUsers')
                                   // OU atribuída ao usuário (simples)
                                   ->orWhere('assigned_user_id', $userId)
                                   // OU atribuída ao usuário (múltipla)
                                   ->orWhereHas('assignedUsers', fn($a) => $a->where('user_id', $userId));
                     });
            })
            // OU conversas internas onde é participante
            ->orWhere(function($internalQ) use ($userId) {
                $internalQ->where('type', self::TYPE_INTERNAL)
                          ->whereHas('participants', fn($p) => $p->where('user_id', $userId)->wherePivot('is_active', true)); // ✅ Usar wherePivot
            });
        });
    }

    /**
     * ✅ SCOPE PARA CONVERSAS NÃO ATRIBUÍDAS
     */
    public function scopeUnassigned($query)
    {
        return $query->whereNull('assigned_user_id')
                     ->whereDoesntHave('assignedUsers');
    }

    /**
     * ✅ SCOPE PARA CONVERSAS COM MÚLTIPLAS ATRIBUIÇÕES
     */
    public function scopeWithMultipleAssignments($query)
    {
        return $query->whereHas('assignedUsers');
    }

    public function evolutionInstance(): BelongsTo
    {
        return $this->belongsTo(EvolutionInstance::class, 'evolution_instance_name', 'instance_name');
    }

    // ✅ MÉTODOS DE VERIFICAÇÃO DE TIPO
    public function isInternal(): bool
    {
        return $this->type === self::TYPE_INTERNAL;
    }

    public function isExternal(): bool
    {
        return in_array($this->type, [self::TYPE_EXTERNAL, self::TYPE_WHATSAPP, self::TYPE_EMAIL, self::TYPE_TELEGRAM, self::TYPE_WEBCHAT]);
    }

    public function isWhatsApp(): bool
    {
        return $this->type === self::TYPE_WHATSAPP || 
               ($this->chatChannel && $this->chatChannel->type === 'evolution_api');
    }

    public function isEmail(): bool
    {
        return $this->type === self::TYPE_EMAIL;
    }

    public function isTelegram(): bool
    {
        return $this->type === self::TYPE_TELEGRAM;
    }

    public function isWebChat(): bool
    {
        return $this->type === self::TYPE_WEBCHAT;
    }

    // ✅ MÉTODOS PARA MÚLTIPLAS ATRIBUIÇÕES (FUTURO)
    public function assignToUser(int $userId, string $role = 'agent', bool $isPrimary = false): bool
    {
        try {
            $this->assignedUsers()->syncWithoutDetaching([
                $userId => [
                    'role' => $role,
                    'assigned_at' => now(),
                    'assigned_by' => auth()->id(),
                    'is_primary' => $isPrimary
                ]
            ]);

            // Se é primário, atualizar o campo assigned_user_id para compatibilidade
            if ($isPrimary) {
                $this->update(['assigned_user_id' => $userId]);
            }

            return true;
        } catch (\Exception $e) {
            \Log::error('Erro ao atribuir usuário à conversa', [
                'conversation_id' => $this->id,
                'user_id' => $userId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    public function removeAssignment(int $userId): bool
    {
        try {
            $this->assignedUsers()->detach($userId);
            
            // Se era o usuário primário, limpar
            if ($this->assigned_user_id === $userId) {
                $this->update(['assigned_user_id' => null]);
            }

            return true;
        } catch (\Exception $e) {
            return false;
        }
    }

    public function getAssignedUserIds(): array
    {
        return $this->assignedUsers()->pluck('users.id')->toArray();
    }

    public function isAssignedToUser(int $userId): bool
    {
        return $this->assignedUsers()->where('user_id', $userId)->exists() ||
               $this->assigned_user_id === $userId;
    }

    // ✅ MÉTODOS PARA CHAT INTERNO
    public function addParticipant(int $userId, string $role = 'participant'): bool
    {
        if (!$this->isInternal()) return false;
        
        try {
            $this->participants()->syncWithoutDetaching([
                $userId => [
                    'role' => $role,
                    'is_active' => true,
                    'joined_at' => now()
                ]
            ]);
            
            $this->updateMetadata('participant_count', $this->participants()->where('is_active', true)->count());
            return true;
        } catch (\Exception $e) {
            return false;
        }
    }

    public function removeParticipant(int $userId): bool
    {
        if (!$this->isInternal()) return false;
        
        try {
            $this->participants()->updateExistingPivot($userId, [
                'is_active' => false,
                'left_at' => now()
            ]);
            
            $this->updateMetadata('participant_count', $this->participants()->where('is_active', true)->count());
            return true;
        } catch (\Exception $e) {
            return false;
        }
    }

    // ✅ MÉTODOS AUXILIARES
    public function updateMetadata(string $key, $value): void
    {
        $metadata = $this->metadata ?? [];
        $metadata[$key] = $value;
        $this->update(['metadata' => $metadata]);
    }

    public function getDisplayName(): string
    {
        if ($this->isInternal()) {
            return data_get($this->metadata, 'subject', 'Chat Interno');
        }

        return $this->contact?->full_name ?? 
               $this->whatsapp_profile_name ?? 
               $this->contact_phone ?? 
               'Conversa Externa';
    }

    // ✅ SCOPES
    public function scopeForUser($query, $userId)
    {
        return $query->where(function ($q) use ($userId) {
            // Conversas atribuídas diretamente
            $q->where('assigned_user_id', $userId)
              // OU conversas com múltiplas atribuições
              ->orWhereHas('assignedUsers', function($subQ) use ($userId) {
                  $subQ->where('user_id', $userId);
              })
              // OU conversas internas com participação
                ->orWhere(function($subQ) use ($userId) {
                    $subQ->where('type', self::TYPE_INTERNAL)
                        ->whereHas('participants', function($partQ) use ($userId) {
                            $partQ->where('users.id', $userId)
                                ->wherePivot('is_active', true);
                        });
                });
        });
    }

    public function scopeByChannel($query, string $channelType)
    {
        return $query->whereHas('chatChannel', function($q) use ($channelType) {
            $q->where('type', $channelType);
        });
    }

    public function scopeByStatus($query, string $status)
    {
        return $query->where('status', $status);
    }

    public function scopeActive($query)
    {
        return $query->where('status', self::STATUS_ACTIVE);
    }
}