<?php
// modules/Chat/Models/EvolutionWebhook.php

namespace Modules\Chat\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Carbon\Carbon;

class EvolutionWebhook extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'instance_name',
        'webhook_id',
        'event_type',
        'event_subtype',
        'payload',
        'headers',
        'source_ip',
        'user_agent',
        'status',
        'received_at',
        'processed_at',
        'processing_attempts',
        'next_attempt_at',
        'processing_result',
        'error_message',
        'error_details',
        'error_code',
        'conversation_id',
        'message_id',
        'contact_id',
        'whatsapp_chat_id',
        'whatsapp_message_id',
        'whatsapp_remote_jid',
        'is_duplicate',
        'duplicate_hash',
        'priority'
    ];

    protected $casts = [
        'payload' => 'array',
        'headers' => 'array',
        'received_at' => 'datetime',
        'processed_at' => 'datetime',
        'next_attempt_at' => 'datetime',
        'processing_result' => 'array',
        'error_details' => 'array',
        'is_duplicate' => 'boolean',
        'processing_attempts' => 'integer',
        'priority' => 'integer'
    ];

    // ===== RELATIONSHIPS =====

    public function instance(): BelongsTo
    {
        return $this->belongsTo(EvolutionInstance::class, 'instance_name', 'instance_name');
    }

    public function conversation(): BelongsTo
    {
        return $this->belongsTo(Conversation::class);
    }

    public function message(): BelongsTo
    {
        return $this->belongsTo(Message::class);
    }

    public function contact(): BelongsTo
    {
        return $this->belongsTo(\Modules\Contacts\Models\Contact::class);
    }

    // ===== SCOPES =====

    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    public function scopeFailed($query)
    {
        return $query->where('status', 'failed');
    }

    public function scopeProcessed($query)
    {
        return $query->where('status', 'processed');
    }

    public function scopeReadyForRetry($query)
    {
        return $query->where('status', 'failed')
                    ->where('processing_attempts', '<', 5)
                    ->where(function($q) {
                        $q->whereNull('next_attempt_at')
                          ->orWhere('next_attempt_at', '<=', now());
                    });
    }

    public function scopeByEventType($query, string $eventType)
    {
        return $query->where('event_type', $eventType);
    }

    public function scopeByInstance($query, string $instanceName)
    {
        return $query->where('instance_name', $instanceName);
    }

    public function scopeRecent($query, int $hours = 24)
    {
        return $query->where('received_at', '>=', now()->subHours($hours));
    }

    // ===== ACCESSORS =====

    public function getProcessingTimeAttribute(): ?int
    {
        if (!$this->processed_at || !$this->received_at) {
            return null;
        }
        
        return $this->received_at->diffInSeconds($this->processed_at);
    }

    public function getIsExpiredAttribute(): bool
    {
        // Webhook expira após 24 horas sem processamento
        return $this->received_at->addHours(24)->isPast();
    }

    public function getCanRetryAttribute(): bool
    {
        return $this->status === 'failed' && 
               $this->processing_attempts < 5 && 
               !$this->is_expired &&
               (!$this->next_attempt_at || $this->next_attempt_at->isPast());
    }

    // ===== MÉTODOS PÚBLICOS =====

    public function markAsProcessing(): bool
    {
        return $this->update([
            'status' => 'processing',
            'processing_attempts' => $this->processing_attempts + 1
        ]);
    }

    public function markAsProcessed(array $result = []): bool
    {
        return $this->update([
            'status' => 'processed',
            'processed_at' => now(),
            'processing_result' => $result,
            'error_message' => null,
            'error_details' => null,
            'error_code' => null
        ]);
    }

    public function markAsFailed(string $errorMessage, array $errorDetails = [], string $errorCode = null): bool
    {
        $nextAttempt = $this->calculateNextAttempt();
        
        return $this->update([
            'status' => 'failed',
            'error_message' => $errorMessage,
            'error_details' => $errorDetails,
            'error_code' => $errorCode,
            'next_attempt_at' => $nextAttempt
        ]);
    }

    public function markAsDuplicate(int $originalWebhookId): bool
    {
        return $this->update([
            'status' => 'duplicate',
            'is_duplicate' => true,
            'processing_result' => [
                'duplicate_of' => $originalWebhookId,
                'detected_at' => now()->toISOString()
            ]
        ]);
    }

    public function markAsIgnored(string $reason): bool
    {
        return $this->update([
            'status' => 'ignored',
            'processing_result' => [
                'ignored_reason' => $reason,
                'ignored_at' => now()->toISOString()
            ]
        ]);
    }

    public function generateDuplicateHash(): string
    {
        // Gerar hash baseado em campos únicos para detectar duplicatas
        $data = [
            'instance_name' => $this->instance_name,
            'event_type' => $this->event_type,
            'whatsapp_message_id' => $this->whatsapp_message_id,
            'whatsapp_chat_id' => $this->whatsapp_chat_id
        ];
        
        return md5(json_encode($data));
    }

    public function extractDataFromPayload(): array
    {
        $payload = $this->payload;
        $extracted = [];
        
        // Extrair dados baseado no tipo de evento
        switch ($this->event_type) {
            case 'messages':
                $extracted = $this->extractMessageData($payload);
                break;
                
            case 'message_status':
                $extracted = $this->extractStatusData($payload);
                break;
                
            case 'connection_status':
                $extracted = $this->extractConnectionData($payload);
                break;
                
            case 'qrcode_updated':
                $extracted = $this->extractQrCodeData($payload);
                break;
        }
        
        return $extracted;
    }

    // ===== MÉTODOS PRIVADOS =====

    private function calculateNextAttempt(): Carbon
    {
        // Backoff exponencial: 1min, 5min, 15min, 1h, 4h
        $delays = [1, 5, 15, 60, 240]; // em minutos
        $attempt = min($this->processing_attempts, count($delays) - 1);
        
        return now()->addMinutes($delays[$attempt]);
    }

    private function extractMessageData(array $payload): array
    {
        $data = $payload['data'] ?? [];
        $key = $data['key'] ?? [];
        $message = $data['message'] ?? [];
        
        return [
            'whatsapp_message_id' => $key['id'] ?? null,
            'whatsapp_chat_id' => $this->extractChatId($key['remoteJid'] ?? ''),
            'whatsapp_remote_jid' => $key['remoteJid'] ?? null,
            'whatsapp_key_from_me' => $key['fromMe'] ?? false,
            'message_type' => $this->detectMessageType($message),
            'content' => $this->extractMessageContent($message),
            'timestamp' => $data['messageTimestamp'] ?? null
        ];
    }

    private function extractStatusData(array $payload): array
    {
        $data = $payload['data'] ?? [];
        
        return [
            'whatsapp_message_id' => $data['id'] ?? null,
            'whatsapp_chat_id' => $this->extractChatId($data['remoteJid'] ?? ''),
            'whatsapp_remote_jid' => $data['remoteJid'] ?? null,
            'delivery_status' => $this->mapDeliveryStatus($data['status'] ?? ''),
            'timestamp' => $data['datetime'] ?? null
        ];
    }

    private function extractConnectionData(array $payload): array
    {
        return [
            'connection_state' => $payload['state'] ?? null,
            'instance_data' => $payload['instance'] ?? []
        ];
    }

    private function extractQrCodeData(array $payload): array
    {
        return [
            'qr_code' => $payload['qrcode'] ?? null,
            'qr_base64' => $payload['base64'] ?? null
        ];
    }

    private function extractChatId(string $remoteJid): string
    {
        // Remover @s.whatsapp.net ou @g.us para obter apenas o número/ID
        return str_replace(['@s.whatsapp.net', '@g.us'], '', $remoteJid);
    }

    private function detectMessageType(array $message): string
    {
        if (isset($message['imageMessage'])) return 'image';
        if (isset($message['audioMessage'])) return 'audio';
        if (isset($message['videoMessage'])) return 'video';
        if (isset($message['documentMessage'])) return 'document';
        if (isset($message['stickerMessage'])) return 'sticker';
        if (isset($message['locationMessage'])) return 'location';
        if (isset($message['contactMessage'])) return 'contact';
        
        return 'text';
    }

    private function extractMessageContent(array $message): string
    {
        // Extrair conteúdo baseado no tipo
        if (isset($message['conversation'])) {
            return $message['conversation'];
        }
        
        if (isset($message['extendedTextMessage']['text'])) {
            return $message['extendedTextMessage']['text'];
        }
        
        // Para outros tipos, retornar tipo da mensagem
        $type = $this->detectMessageType($message);
        return $type === 'text' ? '' : "[{$type}]";
    }

    private function mapDeliveryStatus(string $status): string
    {
        $map = [
            'PENDING' => 'sending',
            'SENT' => 'sent',
            'RECEIVED' => 'delivered',
            'READ' => 'read',
            'FAILED' => 'failed'
        ];
        
        return $map[$status] ?? 'pending';
    }

    // ===== BOOT =====

    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($webhook) {
            // Gerar hash para detectar duplicatas
            $webhook->duplicate_hash = $webhook->generateDuplicateHash();
            
            // Definir prioridade baseada no tipo de evento
            if (!$webhook->priority) {
                $priorities = [
                    'messages' => 1,
                    'message_status' => 2,
                    'connection_status' => 3,
                    'qrcode_updated' => 4
                ];
                
                $webhook->priority = $priorities[$webhook->event_type] ?? 5;
            }
        });
    }
}