<?php
// modules/Chat/app/Providers/ChatServiceProvider.php

namespace Modules\Chat\Providers;

use Closure;
use Modules\Core\Support\ModuleServiceProvider;
use Modules\Core\Menu\MenuItem;
use Modules\Core\Settings\SettingsMenuItem;
use Modules\Chat\Services\EvolutionSyncService;
use Modules\Chat\Services\ChatRedisService;
use Modules\Chat\Services\WhatsAppContactService;
use Modules\Chat\Services\ContactProfilePhotoService;
use Modules\Chat\Services\EvolutionApiService;

class ChatServiceProvider extends ModuleServiceProvider
{
    public function boot(): void
    {
        $this->loadViewsFrom(module_path('Chat', 'resources/views'), 'chat');
        $this->loadRoutesFrom(module_path('Chat', 'routes/web.php'));
        $this->loadRoutesFrom(module_path('Chat', 'routes/api.php'));
        $this->loadTranslationsFrom(module_path('Chat', 'resources/lang'), 'chat');
        $this->loadMigrationsFrom(module_path('Chat', 'database/migrations'));
    }

    public function register(): void
    {
        if (class_exists(\Modules\Chat\Providers\RouteServiceProvider::class)) {
            $this->app->register(\Modules\Chat\Providers\RouteServiceProvider::class);
        }
        
          $this->app->singleton(EvolutionSyncService::class, function ($app) {
                return new EvolutionSyncService(
                    $app->make(EvolutionApiService::class),
                    $app->make(ChatRedisService::class),
                    $app->make(WhatsAppContactService::class), 
                    $app->make(ContactProfilePhotoService::class)
                );
            });
        }

    protected function moduleName(): string
    {
        return 'Chat';
    }

    protected function moduleNameLower(): string
    {
        return 'chat';
    }

    protected function setup(): void
    {
        \Modules\Core\Facades\Innoclapps::vite([
            'resources/js/app.js',
            'resources/js/global-chat.js',
            'resources/css/app.css'
        ], 'modules/' . $this->moduleNameLower() . '/build');
    }

    public function menu(): array
    {
        return [
            MenuItem::make('Chat', '/chat')
                ->icon('ChatBubbleLeftRight')
                ->position(25)
                ->badge(function () {
                    if (!auth()->check()) return null;
                    
                    try {
                        return \Modules\Chat\Models\Conversation::whereHas('messages', function($q) {
                            $q->where('sender_id', '!=', auth()->id())
                              ->whereNull('read_at');
                        })->count() ?: null;
                    } catch (\Exception $e) {
                        return null;
                    }
                })
                ->badgeVariant('danger')
        ];
    }

    protected function settingsMenu(): SettingsMenuItem
    {
        return SettingsMenuItem::make('chat', 'Chat Settings')
            ->path('/settings/chat')
            ->icon('ChatBubbleLeftRight')
            ->order(25);
    }

    protected function scriptData(): Closure
    {
        return function () {
            if (!auth()->check()) return [];

            $user = auth()->user();
            
            return [
                'chat' => [
                    'user_id' => $user->id,
                    'user_name' => $user->name,
                    'user_email' => $user->email,
                    'is_super_admin' => $user->super_admin ?? false,
                    'websocket_url' => config('chat.websocket_url', request()->getSchemeAndHttpHost() . ':9001'),
                    'enabled' => config('chat.enabled', true),
                    'notifications_enabled' => config('chat.notifications', true),
                    'sound_enabled' => config('chat.sound', true),
                    'debug' => config('app.debug', false),
                    'csrf_token' => csrf_token(),
                    
                    // ✅ DADOS COMPLETOS DO USUÁRIO USANDO O SISTEMA EXISTENTE
                    'user' => [
                        'id' => $user->id,
                        'name' => $user->name,
                        'email' => $user->email,
                        'avatar' => $user->avatar,
                        'avatar_url' => $user->avatar_url, // ✅ Usa o accessor do HasAvatar trait
                        'super_admin' => $user->super_admin ?? false,
                        'timezone' => $user->timezone,
                        'locale' => $user->locale
                    ]
                ]
            ];
        };
    }
}