<?php
// modules/Chat/app/Resources/Conversation.php
namespace Modules\Chat\Resources;

use Modules\Core\Contracts\Resources\Tableable;
use Modules\Core\Contracts\Resources\WithResourceRoutes;
use Modules\Core\Contracts\Resources\Importable;
use Modules\Core\Contracts\Resources\Exportable;
use Modules\Core\Resource\Resource;
use Modules\Core\Table\Column;
use Modules\Core\Table\Table;
use Modules\Core\Filters\SelectFilter;
use Modules\Core\Filters\DateFilter;
use Modules\Core\Filters\BelongsToFilter;
use Modules\Core\Actions\Action;
use Modules\Core\Actions\ActionFields;
use Modules\Core\Http\Requests\ActionRequest;
use Illuminate\Database\Eloquent\Builder;
use Modules\Core\Http\Requests\ResourceRequest;
use Modules\Chat\Http\Resources\ConversationResource;
use Illuminate\Support\Collection;

class Conversation extends Resource implements WithResourceRoutes, Tableable, Importable, Exportable
{
    /**
     * The model the resource is related to.
     */
    public static string $model = 'Modules\Chat\Models\Conversation';

    /**
     * Get the json resource that should be used for json response.
     */
    public function jsonResource(): string
    {
        return ConversationResource::class;
    }

    /**
     * Get the displayable label of the resource.
     */
    public static function label(): string
    {
        return 'Conversas';
    }

    /**
     * Get the displayable singular label of the resource.
     */
    public static function singularLabel(): string
    {
        return 'Conversa';
    }

    /**
     * ✅ IMPLEMENTAÇÃO DA TABELA MANTENDO SUA LÓGICA
     */
    public function table(Builder $query, ResourceRequest $request): Table
    {
        // ✅ APLICAR SUA LÓGICA DE VISIBILIDADE EXISTENTE
        $userId = auth()->id();
        $user = auth()->user();
        $isAdmin = $user->super_admin ?? false;
        $showAll = $request->boolean('show_all', false);

        // ✅ SUA LÓGICA DE VISIBILIDADE MANTIDA
        if (!($isAdmin && $showAll)) {
            $query->visibleToUser($userId);
        }

        return Table::make($query, $request)
            ->withViews()
            ->orderBy('last_message_at', 'desc')
            ->setPerPage(20)
            
            // ✅ VISÕES BASEADAS NA SUA LÓGICA
            ->withDefaultView('active', [
                'name' => 'Ativas',
                'query' => ['status' => 'active'],
                'icon' => 'ChatBubbleOvalLeft'
            ])
            ->withDefaultView('unread', [
                'name' => 'Não Lidas',
                'query' => ['has_unread' => true],
                'icon' => 'ExclamationCircle'
            ])
            ->withDefaultView('internal', [
                'name' => 'Internas',
                'query' => ['type' => 'internal'],
                'icon' => 'Users'
            ])
            ->withDefaultView('my_assignments', [
                'name' => 'Minhas Atribuições',
                'query' => ['assigned_to_me' => true],
                'icon' => 'User'
            ])
            
            // ✅ COLUNAS BASEADAS NO SEU formatConversation
            ->addColumn(
                Column::make('contact_name', 'Contato/Assunto')
                    ->width('300px')
                    ->route('/chat?conversation={id}')
                    ->primary()
                    ->sortable()
                    ->formatUsing(function ($model) {
                        if ($model->type === 'internal') {
                            return data_get($model->metadata, 'subject', 'Chat Interno');
                        }
                        return $model->contact_name ?: 
                               $model->whatsapp_profile_name ?: 
                               $this->formatPhoneForDisplay($model->contact_phone ?: '') ?: 
                               'Sem nome';
                    })
            )
            ->addColumn(
                Column::make('last_message', 'Última Mensagem')
                    ->formatUsing(function ($model) {
                        return str($model->lastMessage?->content ?? '')->limit(60);
                    })
            )
            ->addColumn(
                Column::make('type', 'Tipo')
                    ->badge([
                        'internal' => 'blue',
                        'external' => 'green'
                    ])
                    ->formatUsing(function ($value) {
                        return $value === 'internal' ? 'Interna' : 'Externa';
                    })
                    ->sortable()
            )
            ->addColumn(
                Column::make('status', 'Status')
                    ->badge([
                        'active' => 'success',
                        'closed' => 'danger',
                        'archived' => 'warning',
                        'pending' => 'info'
                    ])
                    ->formatUsing(function ($value) {
                        return match($value) {
                            'active' => 'Ativa',
                            'closed' => 'Finalizada',
                            'archived' => 'Arquivada',
                            'pending' => 'Pendente',
                            default => ucfirst($value)
                        };
                    })
                    ->sortable()
            )
            ->addColumn(
                Column::make('assignedUser.name', 'Atribuído a')
                    ->formatUsing(function ($model) {
                        if ($model->assignedUser) {
                            return $model->assignedUser->name;
                        }
                        
                        // ✅ VERIFICAR MÚLTIPLAS ATRIBUIÇÕES
                        if ($model->assignedUsers && $model->assignedUsers->count() > 0) {
                            $primary = $model->assignedUsers->where('pivot.is_primary', true)->first();
                            if ($primary) {
                                return $primary->name . ' (Principal)';
                            }
                            return $model->assignedUsers->first()->name . ' +' . ($model->assignedUsers->count() - 1);
                        }
                        
                        return 'Não atribuído';
                    })
                    ->sortable()
            )
            ->addColumn(
                Column::make('unread_count', 'Não Lidas')
                    ->badge(['0' => 'gray', '*' => 'danger'])
                    ->formatUsing(function ($value) {
                        return $value > 0 ? $value : '-';
                    })
                    ->sortable()
            )
            ->addColumn(
                Column::make('last_message_at', 'Última Atividade')
                    ->dateTime()
                    ->sortable()
            );
    }

    /**
     * ✅ FILTROS BASEADOS NA SUA LÓGICA
     */
    public function filters(ResourceRequest $request): array
    {
        return [
            SelectFilter::make('status', 'Status')
                ->options([
                    'active' => 'Ativas',
                    'closed' => 'Finalizadas',
                    'archived' => 'Arquivadas',
                    'pending' => 'Pendentes'
                ]),
            
            SelectFilter::make('type', 'Tipo')
                ->options([
                    'internal' => 'Internas',
                    'external' => 'Externas'
                ]),
                
            // ✅ FILTRO PERSONALIZADO PARA NÃO LIDAS
            SelectFilter::make('has_unread', 'Mensagens não lidas')
                ->options([
                    '1' => 'Com mensagens não lidas',
                    '0' => 'Todas lidas'
                ])
                ->apply(function ($query, $value) {
                    if ($value === '1') {
                        $query->where('unread_count', '>', 0);
                    } elseif ($value === '0') {
                        $query->where('unread_count', 0);
                    }
                }),
                
            // ✅ FILTRO PARA ATRIBUIÇÕES
            SelectFilter::make('assigned_to_me', 'Atribuições')
                ->options([
                    '1' => 'Atribuídas a mim',
                    '0' => 'Não atribuídas'
                ])
                ->apply(function ($query, $value) use ($request) {
                    $userId = auth()->id();
                    if ($value === '1') {
                        $query->where(function ($q) use ($userId) {
                            $q->where('assigned_user_id', $userId)
                              ->orWhereHas('assignedUsers', function ($subQ) use ($userId) {
                                  $subQ->where('user_id', $userId);
                              });
                        });
                    } elseif ($value === '0') {
                        $query->whereNull('assigned_user_id')
                              ->whereDoesntHave('assignedUsers');
                    }
                }),
                
            DateFilter::make('last_message_at', 'Última atividade'),
            
            BelongsToFilter::make('assigned_user_id', 'assignedUser', 'Atribuído a')
                ->endpoint('/api/chat/users')
        ];
    }

    /**
     * ✅ AÇÕES PERSONALIZADAS BASEADAS NO SEU MENU DE CONTEXTO
     */
    public function actions(ResourceRequest $request): array
    {
        return [
            // ✅ AÇÃO PARA ARQUIVAR
            new class extends Action {
                public function name(): string {
                    return 'Arquivar Conversas';
                }
                
                public function handle(Collection $models, ActionFields $fields) {
                    $models->each->update(['status' => 'archived']);
                }
                
                public function authorizedToRun(ActionRequest $request, $model): bool {
                    return $request->user()->can('update', $model);
                }
            },
            
            // ✅ AÇÃO PARA FINALIZAR
            new class extends Action {
                public function name(): string {
                    return 'Finalizar Conversas';
                }
                
                public function handle(Collection $models, ActionFields $fields) {
                    $models->each(function ($conversation) {
                        $conversation->update([
                            'status' => 'closed',
                            'closed_at' => now(),
                            'closed_by' => auth()->id()
                        ]);
                    });
                }
            },
            
            // ✅ AÇÃO PARA ATRIBUIR EM MASSA
            new class extends Action {
                public function name(): string {
                    return 'Atribuir a Usuário';
                }
                
                public function fields(): array {
                    return [
                        \Modules\Core\Fields\BelongsTo::make('user_id', 'Usuário')
                            ->endpoint('/api/chat/users')
                            ->required()
                    ];
                }
                
                public function handle(Collection $models, ActionFields $fields) {
                    $models->each->update(['assigned_user_id' => $fields->user_id]);
                }
            },
            
            // ✅ AÇÃO PARA MARCAR COMO LIDA
            new class extends Action {
                public function name(): string {
                    return 'Marcar como Lidas';
                }
                
                public function handle(Collection $models, ActionFields $fields) {
                    foreach ($models as $conversation) {
                        \Modules\Chat\Models\Message::where('conversation_id', $conversation->id)
                            ->where('sender_id', '!=', auth()->id())
                            ->whereNull('read_at')
                            ->update(['read_at' => now()]);
                            
                        $conversation->update(['unread_count' => 0]);
                    }
                }
            }
        ];
    }

    /**
     * ✅ HELPER PARA FORMATAR TELEFONE (REUTILIZANDO SUA LÓGICA)
     */
    private function formatPhoneForDisplay(string $phone): string
    {
        $clean = preg_replace('/[^0-9]/', '', $phone);
        
        if (strlen($clean) === 13 && str_starts_with($clean, '55')) {
            $ddd = substr($clean, 2, 2);
            $numero = substr($clean, 4);
            if (strlen($numero) === 9) {
                return "+55 ({$ddd}) " . substr($numero, 0, 5) . "-" . substr($numero, 5);
            }
        } elseif (strlen($clean) === 11) {
            $ddd = substr($clean, 0, 2);
            $numero = substr($clean, 2);
            if (strlen($numero) === 9) {
                return "({$ddd}) " . substr($numero, 0, 5) . "-" . substr($numero, 5);
            }
        } elseif (strlen($clean) === 10) {
            $ddd = substr($clean, 0, 2);
            $numero = substr($clean, 2);
            return "({$ddd}) " . substr($numero, 0, 4) . "-" . substr($numero, 4);
        }
        
        return $phone;
    }
}