<?php

namespace Modules\Chat\Services;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Redis;
use Modules\Chat\Models\Message;
use Modules\Chat\Models\Conversation;
use Exception;

class ChatRedisService
{
    protected $redis;
    protected string $streamName;
    protected array $eventTypes = [
        'NEW_WHATSAPP_MESSAGE' => 'new_whatsapp_message',
        'NEW_INTERNAL_MESSAGE' => 'new_message',
        'WHATSAPP_STATUS_UPDATE' => 'whatsapp_status_update',
        'USER_TYPING' => 'chat.user.typing',
        'CONVERSATION_READ' => 'chat.conversation.read',
        'EVOLUTION_CONNECTION_STATUS' => 'evolution_connection_status',
        'EVOLUTION_QRCODE_UPDATED' => 'evolution_qrcode_updated',
        'CONVERSATION_UPDATED' => 'chat.conversation.updated', 
        'NEW_CONVERSATION_CREATED' => 'new_conversation_created', 
        'OUTBOUND_WHATSAPP_MESSAGE' => 'outbound_whatsapp_message', 
    ];
    
    public function __construct()
    {
        try {
            $this->redis = Redis::connection('default');
            $this->streamName = config('chat.redis.stream_name', 'chat_messages');
            
            $ping = $this->redis->ping();
            Log::info('✅ Redis conectado para integração Node.js', [
                'ping' => $ping,
                'stream_name_full' => $this->streamName,
                'stream_name_base' => 'chat-messages'
            ]);
        } catch (Exception $e) {
            Log::error('❌ Erro conectando Redis: ' . $e->getMessage());
            $this->redis = null;
        }
    }

    public function getEventTypes(): array
    {
        return $this->eventTypes;
    }

    public function getEventType(string $key): ?string
    {
        return $this->eventTypes[$key] ?? null;
    }
    
    public function getStreamName(): string
    {
        return $this->streamName;
    }

    /**
     * ✅ MÉTODO PRINCIPAL PARA PUBLICAR NO STREAM
     */
    public function publishToStream(string $event, array $data, ?string $conversationSpecificStream = null)
    {
        try {
            if (!$this->isRedisAvailable()) {
                Log::error('🔥 Redis não disponível para publicação para stream', ['event' => $event]);
                return false;
            }

            // ✅ GARANTIR DADOS ESSENCIAIS
            $data['source'] = $data['source'] ?? 'laravel'; 
            $data['timestamp'] = $data['timestamp'] ?? now()->toISOString();
            $data['event_id'] = uniqid('evt_', true);

            $fields = [
                'event' => $event,
                'data' => json_encode($data, JSON_UNESCAPED_UNICODE),
                'created_at' => (string)now()->timestamp,
                'event_id' => $data['event_id'],
            ];
            
            // ✅ PUBLICAR NO STREAM PRINCIPAL
            $messageId = $this->redis->xadd($this->streamName, '*', $fields);
            
            if (!$messageId) {
                Log::error('❌ Falha ao publicar evento no Redis Stream', [
                    'stream' => $this->streamName, 
                    'event' => $event
                ]);
                return false;
            }

            // ✅ PUBLICAR NO STREAM DA CONVERSA SE ESPECIFICADO
            if ($conversationSpecificStream) {
                $conversationMessageId = $this->redis->xadd($conversationSpecificStream, '*', $fields);
                
                // ✅ LIMPAR STREAMS ANTIGOS
                $this->redis->xtrim($conversationSpecificStream, 'MAXLEN', '~', 1000); 
            }
            
            // ✅ LIMPAR STREAM PRINCIPAL
            $this->redis->xtrim($this->streamName, 'MAXLEN', '~', 10000); 

            Log::debug('✅ Evento publicado no Redis Stream', [
                'stream' => $this->streamName,
                'event' => $event,
                'message_id' => $messageId,
                'conversation_stream' => $conversationSpecificStream,
                'data_size' => strlen(json_encode($data))
            ]);

            return $messageId;

        } catch (Exception $e) {
            Log::error('❌ Exceção ao publicar evento no Redis Stream: ' . $e->getMessage(), [
                'stream' => $this->streamName,
                'event' => $event,
                'error' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * ✅ PUBLICAR ATUALIZAÇÃO DE CONVERSA
     */
    public function publishConversationUpdate(Conversation $conversation, string $changeType = 'general_update'): void
    {
        try {
            $conversation->loadMissing(['assignedUser:id,name,email,avatar', 'participants:id,name,email,avatar']);

            $eventData = [
                'type' => $changeType,
                'conversation_id' => $conversation->id,
                'conversation' => $conversation->toArray(),
                'timestamp' => now()->toISOString()
            ];

            $this->publishToStream(
                $this->eventTypes['CONVERSATION_UPDATED'],
                $eventData,
                "conversation_{$conversation->id}"
            );

            Log::debug('📡 Evento de atualização de conversa publicado', [
                'conversation_id' => $conversation->id,
                'change_type' => $changeType
            ]);

        } catch (Exception $e) {
            Log::error('❌ Erro ao publicar evento de atualização de conversa: ' . $e->getMessage(), [
                'conversation_id' => $conversation->id,
                'error' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * ✅ PUBLICAR MENSAGEM WHATSAPP - USA toFrontendArray()
     */
    public function publishWhatsAppMessage(Message $message, Conversation $conversation): void
    {
        try {
            $message->loadMissing(['sender', 'attachments']); 
            $conversation->loadMissing(['assignedUser:id,name,email,avatar', 'participants:id,name,email,avatar']);

            $eventData = [
                'type' => 'whatsapp_message',
                'conversation_id' => $conversation->id,
                'message' => $message->toFrontendArray(), // ✅ USA URLs SEGURAS
                'conversation' => $conversation->toArray(),
                'timestamp' => now()->toISOString(),
                'message_type' => $message->message_type,
                'from_me' => $message->whatsapp_key_from_me,
                'sender_id' => $message->sender_id,
                'sender_type' => $message->sender_type,
            ];

            $messageId = $this->publishToStream(
                $this->eventTypes['NEW_WHATSAPP_MESSAGE'], 
                $eventData, 
                "conversation_{$conversation->id}"
            );

            if ($messageId) {
                Log::debug('📡 Evento WhatsApp publicado com URLs seguras', [
                    'message_id' => $message->id,
                    'conversation_id' => $conversation->id,
                    'has_attachments' => $message->attachments->count() > 0,
                    'redis_message_id' => $messageId,
                    'from_me' => $message->whatsapp_key_from_me
                ]);
            }

        } catch (Exception $e) {
            Log::error('❌ Erro ao publicar evento WhatsApp: ' . $e->getMessage(), [
                'message_id' => $message->id,
                'conversation_id' => $conversation->id ?? null,
                'error' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * ✅ PUBLICAR MENSAGEM INTERNA - USA toFrontendArray()
     */
    public function publishInternalMessage(Message $message, Conversation $conversation): void
    {
        try {
            $message->loadMissing(['sender', 'attachments']);
            $conversation->loadMissing(['assignedUser:id,name,email,avatar', 'participants:id,name,email,avatar']);

            $eventData = [
                'type' => 'internal_message',
                'conversation_id' => $conversation->id,
                'message' => $message->toFrontendArray(), // ✅ USA URLs SEGURAS
                'conversation' => $conversation->toArray(),
                'timestamp' => now()->toISOString(),
                'message_type' => $message->message_type,
                'sender_id' => $message->sender_id,
                'sender_type' => $message->sender_type,
            ];

            $messageId = $this->publishToStream(
                $this->eventTypes['NEW_INTERNAL_MESSAGE'], 
                $eventData, 
                "conversation_{$conversation->id}"
            );

            if ($messageId) {
                Log::debug('📡 Evento Mensagem Interna publicado', [
                    'message_id' => $message->id,
                    'conversation_id' => $conversation->id,
                    'redis_message_id' => $messageId
                ]);
            }

        } catch (Exception $e) {
            Log::error('❌ Erro ao publicar evento Mensagem Interna: ' . $e->getMessage(), [
                'message_id' => $message->id,
                'conversation_id' => $conversation->id ?? null,
                'error' => $e->getTraceAsString()
            ]);
        }
    }
    
    /**
     * ✅ PUBLICAR STATUS UPDATE DO WHATSAPP
     */
    public function publishWhatsAppStatusUpdate(Message $message, string $oldStatus, string $newStatus): void
    {
        try {
            $eventData = [
                'type' => 'whatsapp_status_update',
                'conversation_id' => $message->conversation_id,
                'message_id' => $message->id,
                'whatsapp_message_id' => $message->whatsapp_message_id,
                'old_status' => $oldStatus,
                'new_status' => $newStatus,
                'updated_at' => now()->toISOString(),
                'timestamp' => now()->toISOString(),
                'sender_id' => $message->sender_id,
                'from_me' => $message->whatsapp_key_from_me,
            ];

            $messageId = $this->publishToStream(
                $this->eventTypes['WHATSAPP_STATUS_UPDATE'], 
                $eventData, 
                "conversation_{$message->conversation_id}"
            );

            if ($messageId) {
                Log::debug('📡 Status update publicado via Redis', [
                    'message_id' => $message->id,
                    'whatsapp_message_id' => $message->whatsapp_message_id,
                    'status_change' => "{$oldStatus} → {$newStatus}",
                    'redis_message_id' => $messageId,
                    'conversation_id' => $message->conversation_id
                ]);
            }

        } catch (Exception $e) {
            Log::error('❌ Erro ao publicar status WhatsApp: ' . $e->getMessage(), [
                'message_id' => $message->id,
                'conversation_id' => $message->conversation_id ?? null,
                'error' => $e->getTraceAsString()
            ]);
        }
    }

    public function notifyUserTyping(int $userId, string $userName, int $conversationId, bool $isTyping = true): void
    {
        try {
            $eventData = [
                'conversation_id' => $conversationId,
                'user_id' => $userId,
                'user_name' => $userName,
                'is_typing' => $isTyping,
                'timestamp' => now()->toISOString()
            ];

            $this->publishToStream($this->eventTypes['USER_TYPING'], $eventData, "conversation_{$conversationId}");

        } catch (Exception $e) {
            Log::error('❌ Erro ao publicar indicador de digitação: ' . $e->getMessage(), [
                'user_id' => $userId,
                'conversation_id' => $conversationId,
                'error' => $e->getTraceAsString()
            ]);
        }
    }

    public function notifyConversationRead(int $conversationId, int $userId): void
    {
        try {
            $eventData = [
                'conversation_id' => $conversationId,
                'user_id' => $userId,
                'read_at' => now()->toISOString()
            ];

            $this->publishToStream($this->eventTypes['CONVERSATION_READ'], $eventData, "conversation_{$conversationId}");

        } catch (Exception $e) {
            Log::error('❌ Erro ao publicar conversa lida: ' . $e->getMessage(), [
                'conversation_id' => $conversationId,
                'user_id' => $userId,
                'error' => $e->getTraceAsString()
            ]);
        }
    }

    public function isRedisAvailable(): bool
    {
        try {
            return $this->redis !== null && ($this->redis->ping() === 'PONG' || $this->redis->ping() === true);
        } catch (Exception $e) {
            Log::error('❌ Redis não disponível: ' . $e->getMessage());
            return false;
        }
    }

    public function notifyNewMessage(Message $message, Conversation $conversation): void
    {
        if ($conversation->isWhatsApp()) {
            $this->publishWhatsAppMessage($message, $conversation);
        } else {
            $this->publishInternalMessage($message, $conversation);
        }
    }

    /**
     * ✅ MÉTODOS PARA DEBUG E MONITORAMENTO
     */
    public function getStreamInfo(): array
    {
        try {
            if (!$this->isRedisAvailable()) {
                return ['error' => 'Redis não disponível'];
            }

            $info = $this->redis->xinfo('STREAM', $this->streamName);
            
            return [
                'stream_name' => $this->streamName,
                'length' => $info[1] ?? 0,
                'first_entry' => $info[5] ?? null,
                'last_entry' => $info[7] ?? null,
                'available' => true
            ];
        } catch (Exception $e) {
            return [
                'stream_name' => $this->streamName,
                'error' => $e->getMessage(),
                'available' => false
            ];
        }
    }

    public function getEventTypesCount(): array
    {
        return array_count_values($this->eventTypes);
    }
}