<?php
// modules/Chat/app/Services/ContactProfilePhotoService.php - VERSÃO OTIMIZADA

namespace Modules\Chat\Services;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Http;
use Modules\Contacts\Models\Contact;
use Modules\Chat\Models\EvolutionInstance;
use Carbon\Carbon;
use Exception;

class ContactProfilePhotoService
{
    protected EvolutionApiService $evolutionApi;

    public function __construct(EvolutionApiService $evolutionApi)
    {
        $this->evolutionApi = $evolutionApi;
    }

    /**
     * ✅ BUSCAR E SALVAR FOTO DE PERFIL DO CONTATO
     */
    public function fetchAndSaveContactProfilePhoto(Contact $contact, string $phoneNumber, EvolutionInstance $instance): bool
    {
        try {
            // ✅ VERIFICAR SE PRECISA ATUALIZAR (CACHE DE 24H)
            if (!$contact->needsWhatsappProfilePhotoUpdate()) {
                Log::debug('📸 Foto de perfil do contato ainda válida (cache)', [
                    'contact_id' => $contact->id,
                    'phone' => $this->maskPhone($phoneNumber),
                    'last_update' => $contact->whatsapp_profile_photo_updated_at
                ]);
                return $contact->hasWhatsappProfilePhoto();
            }

            $this->evolutionApi->setInstance($instance);

            Log::info('📸 Buscando foto de perfil do contato', [
                'contact_id' => $contact->id,
                'phone' => $this->maskPhone($phoneNumber),
                'contact_name' => $contact->first_name
            ]);

            // ✅ BUSCAR PERFIL VIA API (MÉTODO CORRIGIDO)
            $profileData = $this->fetchContactProfileFromApi($phoneNumber);
            
            if (!$profileData || empty($profileData['profilePictureUrl'])) {
                Log::debug('📸 Nenhuma foto de perfil encontrada para contato', [
                    'contact_id' => $contact->id,
                    'phone' => $this->maskPhone($phoneNumber)
                ]);
                
                // ✅ MARCAR COMO TENTATIVA REALIZADA
                $contact->update([
                    'whatsapp_profile_photo_updated_at' => now(),
                    'whatsapp_profile_metadata' => [
                        'last_check' => now(), 
                        'has_photo' => false,
                        'phone_checked' => $phoneNumber
                    ]
                ]);
                
                return false;
            }

            // ✅ BAIXAR E SALVAR FOTO
            $photoPath = $this->downloadAndSaveContactPhoto(
                $profileData['profilePictureUrl'], 
                $contact,
                $phoneNumber
            );

            if ($photoPath) {
                // ✅ ATUALIZAR CONTATO
                $contact->update([
                    'whatsapp_profile_photo_url' => $profileData['profilePictureUrl'],
                    'whatsapp_profile_photo_path' => $photoPath,
                    'whatsapp_profile_photo_updated_at' => now(),
                    'whatsapp_profile_metadata' => [
                        'last_check' => now(),
                        'has_photo' => true,
                        'original_url' => $profileData['profilePictureUrl'],
                        'file_size' => Storage::disk('public')->size($photoPath) ?? 0,
                        'phone_checked' => $phoneNumber,
                        'wuid' => $profileData['wuid'] ?? null
                    ]
                ]);

                Log::info('✅ Foto de perfil do contato salva com sucesso', [
                    'contact_id' => $contact->id,
                    'photo_path' => $photoPath,
                    'file_size' => Storage::disk('public')->size($photoPath),
                    'wuid' => $profileData['wuid'] ?? 'unknown'
                ]);

                return true;
            }

            return false;
        } catch (Exception $e) {
            Log::error('❌ Erro ao buscar foto de perfil do contato', [
                'contact_id' => $contact->id,
                'phone' => $this->maskPhone($phoneNumber),
                'error' => $e->getMessage()
            ]);
            
            // ✅ MARCAR ERRO PARA NÃO TENTAR NOVAMENTE POR UM TEMPO
            $contact->update([
                'whatsapp_profile_photo_updated_at' => now(),
                'whatsapp_profile_metadata' => [
                    'last_check' => now(),
                    'has_photo' => false,
                    'error' => $e->getMessage(),
                    'phone_checked' => $phoneNumber
                ]
            ]);
            
            return false;
        }
    }

    /**
     * ✅ BUSCAR PERFIL VIA API EVOLUTION - CONFORME DOCUMENTAÇÃO
     */
    protected function fetchContactProfileFromApi(string $phoneNumber): ?array
    {
        try {
            $instance = $this->evolutionApi->getInstance();
            
            if (!$instance) {
                throw new Exception('Instância Evolution não configurada');
            }

            // ✅ LIMPAR E FORMATAR NÚMERO
            $cleanPhone = preg_replace('/[^0-9]/', '', $phoneNumber);
            
            // ✅ FORMATOS TESTADOS EM ORDEM DE PRIORIDADE
            $phoneFormats = [
                $cleanPhone,                    // Formato original
                '55' . $cleanPhone,            // Com código do país
            ];

            foreach ($phoneFormats as $phone) {
                $result = $this->tryFetchProfilePicture($phone, $instance);
                if ($result) {
                    Log::info('✅ Foto de perfil encontrada', [
                        'phone_format_used' => $this->maskPhone($phone),
                        'original_phone' => $this->maskPhone($phoneNumber),
                        'wuid' => $result['wuid'] ?? 'unknown',
                        'has_url' => !empty($result['profilePictureUrl'])
                    ]);
                    return $result;
                }
                
                // ✅ DELAY MENOR ENTRE TENTATIVAS
                usleep(300000); // 0.3 segundos
            }

            Log::debug('📸 Nenhuma foto encontrada após tentar todos os formatos', [
                'phone' => $this->maskPhone($phoneNumber),
                'formats_tried' => count($phoneFormats)
            ]);

            return null;
        } catch (Exception $e) {
            Log::warning('⚠️ Erro ao buscar perfil do contato via API', [
                'phone' => $this->maskPhone($phoneNumber),
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

private function tryFetchProfilePicture(string $phoneNumber, EvolutionInstance $instance): ?array
{
    try {
        // ✅ ENDPOINT CORRETO CONFORME DOCUMENTAÇÃO
        $url = rtrim($instance->api_url, '/') . '/chat/fetchProfile/' . $instance->instance_name;
        
        // ✅ PAYLOAD CONFORME DOCUMENTAÇÃO
        $payload = [
            'number' => $phoneNumber
        ];
        
        Log::debug('📡 Buscando perfil via endpoint correto', [
            'url' => $url,
            'phone' => $this->maskPhone($phoneNumber),
            'instance' => $instance->instance_name
        ]);
        
        $response = Http::withHeaders([
            'apikey' => $instance->api_key,
            'Content-Type' => 'application/json',
            'Accept' => 'application/json'
        ])->timeout(20)->post($url, $payload);

        Log::debug('📡 Resposta da Evolution API', [
            'phone' => $this->maskPhone($phoneNumber),
            'status' => $response->status(),
            'success' => $response->successful(),
            'has_body' => !empty($response->body()),
            'response_size' => strlen($response->body())
        ]);

        if ($response->successful()) {
            $data = $response->json();
            
            // ✅ VERIFICAR ESTRUTURA DA RESPOSTA
            Log::debug('📡 Estrutura da resposta', [
                'phone' => $this->maskPhone($phoneNumber),
                'response_keys' => $data ? array_keys($data) : [],
                'full_response' => $data
            ]);
            
            // ✅ BUSCAR FOTO EM POSSÍVEIS CAMPOS
            $profilePictureUrl = $data['profilePictureUrl'] ?? 
                                $data['picture'] ?? 
                                $data['profilePicture'] ?? 
                                $data['avatar'] ?? null;
            
            if ($profilePictureUrl && !empty($profilePictureUrl)) {
                Log::info('✅ Foto de perfil encontrada via endpoint correto', [
                    'phone' => $this->maskPhone($phoneNumber),
                    'url_preview' => substr($profilePictureUrl, 0, 50) . '...',
                    'url_valid' => filter_var($profilePictureUrl, FILTER_VALIDATE_URL) !== false
                ]);
                
                return [
                    'profilePictureUrl' => $profilePictureUrl,
                    'wuid' => $data['wuid'] ?? $phoneNumber,
                    'endpoint_used' => '/chat/fetchProfile',
                    'raw_data' => $data
                ];
            } else {
                Log::debug('📸 Resposta sem profilePictureUrl válida', [
                    'phone' => $this->maskPhone($phoneNumber),
                    'response_keys' => $data ? array_keys($data) : [],
                    'profilePictureUrl_exists' => isset($data['profilePictureUrl']),
                    'profilePictureUrl_empty' => empty($data['profilePictureUrl'] ?? ''),
                    'full_response' => $data
                ]);
            }
        } else {
            Log::warning('⚠️ Resposta não bem-sucedida da Evolution API', [
                'phone' => $this->maskPhone($phoneNumber),
                'status' => $response->status(),
                'body_preview' => substr($response->body(), 0, 200)
            ]);
        }

        return null;
        
    } catch (Exception $e) {
        Log::debug('⚠️ Erro ao tentar buscar foto', [
            'phone' => $this->maskPhone($phoneNumber),
            'error' => $e->getMessage(),
            'error_type' => get_class($e)
        ]);
        return null;
    }
}

    /**
     * ✅ BAIXAR E SALVAR FOTO COM VALIDAÇÕES ROBUSTAS
     */
    protected function downloadAndSaveContactPhoto(string $photoUrl, Contact $contact, string $phoneNumber): ?string
    {
        try {
            Log::debug('📥 Baixando foto de perfil do contato', [
                'contact_id' => $contact->id,
                'url_preview' => substr($photoUrl, 0, 50) . '...'
            ]);

            // ✅ VALIDAR URL
            if (!filter_var($photoUrl, FILTER_VALIDATE_URL)) {
                throw new Exception('URL inválida fornecida');
            }

            // ✅ BAIXAR COM HEADERS APROPRIADOS PARA WHATSAPP
            $response = Http::withHeaders([
                'User-Agent' => 'WhatsApp/2.23.24.76 Mozilla/5.0 (compatible; ConcordCRM/1.0)',
                'Accept' => 'image/webp,image/apng,image/*,*/*;q=0.8',
                'Accept-Language' => 'pt-BR,pt;q=0.9,en;q=0.8',
                'Cache-Control' => 'no-cache',
                'Pragma' => 'no-cache'
            ])->timeout(30)->retry(2, 1000)->get($photoUrl);
            
            if (!$response->successful()) {
                throw new Exception("HTTP {$response->status()}: Falha ao baixar foto");
            }

            $imageData = $response->body();
            
            // ✅ VALIDAÇÕES DE SEGURANÇA
            if (strlen($imageData) < 500) {
                throw new Exception('Imagem muito pequena (possível erro)');
            }
            
            if (strlen($imageData) > 10 * 1024 * 1024) { // 10MB
                throw new Exception('Imagem muito grande (>10MB)');
            }

            // ✅ VALIDAR SE É IMAGEM REAL
            $imageInfo = @getimagesizefromstring($imageData);
            if (!$imageInfo) {
                // ✅ TENTAR DETECTAR POR HEADER BINÁRIO
                $header = bin2hex(substr($imageData, 0, 10));
                if (!str_starts_with($header, 'ffd8ff') && // JPEG
                    !str_starts_with($header, '89504e47') && // PNG
                    !str_starts_with($header, '52494646')) { // WEBP
                    throw new Exception("Dados não são uma imagem válida. Header: {$header}");
                }
                
                // ✅ ASSUMIR JPEG SE NÃO CONSEGUIR DETECTAR
                $mimeType = 'image/jpeg';
                $extension = 'jpg';
            } else {
                $mimeType = $imageInfo['mime'];
                $allowedTypes = ['image/jpeg', 'image/png', 'image/webp'];
                
                if (!in_array($mimeType, $allowedTypes)) {
                    throw new Exception("Formato não suportado: {$mimeType}");
                }

                // ✅ DETERMINAR EXTENSÃO
                $extension = match($mimeType) {
                    'image/jpeg' => 'jpg',
                    'image/png' => 'png', 
                    'image/webp' => 'webp',
                    default => 'jpg'
                };
            }

            // ✅ GERAR NOME ÚNICO
            $timestamp = time();
            $hash = substr(md5($imageData), 0, 8);
            $fileName = "contact_profile_{$contact->id}_{$timestamp}_{$hash}.{$extension}";
            $filePath = "contact-profile-photos/{$fileName}";

            // ✅ CRIAR DIRETÓRIO
            Storage::disk('public')->makeDirectory('contact-profile-photos');

            // ✅ REMOVER FOTO ANTERIOR SE EXISTIR
            if ($contact->whatsapp_profile_photo_path && 
                Storage::disk('public')->exists($contact->whatsapp_profile_photo_path)) {
                Storage::disk('public')->delete($contact->whatsapp_profile_photo_path);
            }

            // ✅ SALVAR NOVA FOTO
            $saved = Storage::disk('public')->put($filePath, $imageData);
            
            if (!$saved) {
                throw new Exception('Falha ao salvar arquivo no storage');
            }

            Log::info('✅ Foto de perfil do contato baixada e salva', [
                'contact_id' => $contact->id,
                'file_path' => $filePath,
                'file_size' => strlen($imageData),
                'format' => $extension,
                'dimensions' => $imageInfo ? "{$imageInfo[0]}x{$imageInfo[1]}" : 'unknown'
            ]);

            return $filePath;

        } catch (Exception $e) {
            Log::error('❌ Erro ao baixar foto de perfil do contato', [
                'contact_id' => $contact->id,
                'url_preview' => substr($photoUrl, 0, 50) . '...',
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * ✅ BUSCAR FOTOS EM LOTE COM CONTROLE DE RATE LIMIT OTIMIZADO
     */
    public function fetchMultipleContactProfilePhotos(array $contactData, EvolutionInstance $instance): array
    {
        $results = [];
        $successCount = 0;
        $errorCount = 0;

        Log::info('📸 Iniciando busca de fotos de perfil em lote', [
            'contact_count' => count($contactData),
            'instance' => $instance->instance_name
        ]);

        foreach ($contactData as $index => $data) {
            $contact = $data['contact'];
            $phoneNumber = $data['phone'];
            
            try {
                $success = $this->fetchAndSaveContactProfilePhoto($contact, $phoneNumber, $instance);
                $results[$contact->id] = [
                    'success' => $success,
                    'contact_id' => $contact->id,
                    'contact_name' => $contact->full_name,
                    'phone' => $this->maskPhone($phoneNumber),
                    'processed_at' => now()->toISOString()
                ];
                
                if ($success) {
                    $successCount++;
                } else {
                    $errorCount++;
                }
                
                // ✅ RATE LIMITING MAIS CONSERVADOR
                if (($index + 1) % 3 === 0) {
                    Log::debug('⏸️ Pausa para rate limiting', [
                        'processed' => $index + 1,
                        'remaining' => count($contactData) - ($index + 1)
                    ]);
                    sleep(3); // Pausa de 3 segundos a cada 3 contatos
                } else {
                    sleep(1); // 1 segundo entre cada contato
                }
                
            } catch (Exception $e) {
                $errorCount++;
                $results[$contact->id] = [
                    'success' => false,
                    'error' => $e->getMessage(),
                    'contact_id' => $contact->id,
                    'contact_name' => $contact->full_name,
                    'phone' => $this->maskPhone($phoneNumber),
                    'processed_at' => now()->toISOString()
                ];
                
                Log::error('❌ Erro ao processar contato em lote', [
                    'contact_id' => $contact->id,
                    'contact_name' => $contact->full_name,
                    'phone' => $this->maskPhone($phoneNumber),
                    'error' => $e->getMessage()
                ]);
            }
        }

        Log::info('✅ Busca de fotos em lote finalizada', [
            'total_processed' => count($contactData),
            'success_count' => $successCount,
            'error_count' => $errorCount,
            'success_rate' => round(($successCount / count($contactData)) * 100, 2) . '%'
        ]);

        return [
            'results' => $results,
            'summary' => [
                'total' => count($contactData),
                'success' => $successCount,
                'errors' => $errorCount,
                'success_rate' => round(($successCount / count($contactData)) * 100, 2)
            ]
        ];
    }

    // ===== MÉTODOS AUXILIARES =====

    /**
     * 🎭 MASCARAR TELEFONE PARA LOGS
     */
    protected function maskPhone(string $phone): string
    {
        if (empty($phone)) {
            return '****';
        }
        
        // ✅ REMOVER CARACTERES ESPECIAIS PARA MASCARAR
        $cleanPhone = preg_replace('/[^0-9]/', '', $phone);
        
        if (strlen($cleanPhone) >= 8) {
            return substr($cleanPhone, 0, 4) . '****' . substr($cleanPhone, -2);
        }
        
        return '****';
    }

    /**
     * 🧪 TESTAR ENDPOINT ESPECÍFICO
     */
    public function testProfilePictureEndpoint(string $phoneNumber, EvolutionInstance $instance): array
    {
        try {
            $cleanPhone = preg_replace('/[^0-9]/', '', $phoneNumber);
            
            $url = rtrim($instance->api_url, '/') . '/chat/fetchProfilePictureUrl/' . $instance->instance_name;
            
            $payload = [
                'number' => $cleanPhone
            ];
            
            $startTime = microtime(true);
            
            $response = Http::withHeaders([
                'apikey' => $instance->api_key,
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ])->timeout(15)->post($url, $payload);
            
            $responseTime = round((microtime(true) - $startTime) * 1000, 2);
            
            return [
                'success' => $response->successful(),
                'status_code' => $response->status(),
                'response_time_ms' => $responseTime,
                'has_profile_picture' => $response->successful() && 
                                       $response->json() && 
                                       !empty($response->json()['profilePictureUrl'] ?? ''),
                'response_data' => $response->json(),
                'url_tested' => $url,
                'phone_tested' => $this->maskPhone($cleanPhone)
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'phone_tested' => $this->maskPhone($phoneNumber)
            ];
        }
    }

    /**
     * 📊 OBTER ESTATÍSTICAS DE FOTOS
     */
    public function getPhotoStatistics(): array
    {
        $totalContacts = Contact::count();
        $contactsWithWhatsappPhoto = Contact::whereNotNull('whatsapp_profile_photo_path')->count();
        $contactsWithWhatsappUrl = Contact::whereNotNull('whatsapp_profile_photo_url')->count();
        $contactsWithManualPhoto = Contact::whereNotNull('avatar')->count();
        
        return [
            'total_contacts' => $totalContacts,
            'with_whatsapp_photo' => $contactsWithWhatsappPhoto,
            'with_whatsapp_url' => $contactsWithWhatsappUrl,
            'with_manual_photo' => $contactsWithManualPhoto,
            'whatsapp_photo_percentage' => $totalContacts > 0 ? round(($contactsWithWhatsappPhoto / $totalContacts) * 100, 2) : 0,
            'manual_photo_percentage' => $totalContacts > 0 ? round(($contactsWithManualPhoto / $totalContacts) * 100, 2) : 0,
        ];
    }
}