<?php
// modules/Chat/Services/EvolutionApiService.php
// ✅ VERSÃO CORRIGIDA PARA EVOLUTION API v2.3.1 - SEGUINDO PADRÕES CONCORDCRM

namespace Modules\Chat\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Modules\Chat\Models\EvolutionInstance;
use Exception;

class EvolutionApiService
{
    protected ?EvolutionInstance $instance = null;
    protected int $timeout = 30;
    protected int $retryTimes = 3;

    public function __construct()
    {
        $this->timeout = 30;
        $this->retryTimes = 3;
    }

    public function setInstance(EvolutionInstance $instance): self
    {
        $this->instance = $instance;
        return $this;
    }

    public function getInstance(): EvolutionInstance
    {
        if (!$this->instance) {
            throw new Exception('Evolution instance not set. Use setInstance() first.');
        }
        return $this->instance;
    }

    /**
     * 🏗️ Criar instância na Evolution API (se não existir)
     */
    public function createInstanceIfNotExists(): array
    {
        $instance = $this->getInstance();
        try {
            Log::info('🏗️ Criando/verificando instância na Evolution API', [
                'instance_name' => $instance->instance_name
            ]);

            // ✅ PRIMEIRO VERIFICAR SE JÁ EXISTE
            try {
                // Aumentar o timeout para este GET para dar mais tempo para a API responder
                $existingInstances = $this->makeRequest('GET', '/instance/fetchInstances', [], 30);

                // Verificar se a instância já existe
                if (is_array($existingInstances)) {
                    foreach ($existingInstances as $existingInstance) {
                        // ✅ Lógica CORRIGIDA: Verificar 'instanceName' OU 'name'
                        $foundInstanceName = $existingInstance['instanceName'] ?? $existingInstance['name'] ?? null;
                        if ($foundInstanceName === $instance->instance_name) {
                            Log::info('ℹ️ Instância já existe na Evolution API', [
                                'instance_name' => $instance->instance_name,
                                'status' => $existingInstance['status'] ?? 'unknown'
                            ]);
                            return [
                                'success' => true,
                                'message' => 'Instance already exists',
                                'instance' => $existingInstance
                            ];
                        }
                    }
                }
            } catch (Exception $e) {
                // Captura o erro da requisição, mas não impede a tentativa de criação
                Log::debug('⚠️ Erro ao verificar instâncias existentes (continuando com criação)', [
                    'error' => $e->getMessage()
                ]);
            }

            // ✅ SE NÃO EXISTE (ou a verificação falhou), TENTAR CRIAR
            $payload = [
                'instanceName' => $instance->instance_name,
                'integration' => 'WHATSAPP-BAILEYS',
                'qrcode' => true
            ];

            // ✅ ADICIONAR WEBHOOK SE CONFIGURADO
            if ($instance->webhook_url) {
                $payload['webhook'] = [
                    'url' => $instance->webhook_url,
                    'byEvents' => true,
                    'base64' => true,
                    'events' => [
                        'QRCODE_UPDATED',
                        'MESSAGES_UPSERT',
                        'MESSAGES_UPDATE',
                        'CONNECTION_UPDATE',
                        'SEND_MESSAGE'
                    ]
                ];
            }

            $response = $this->makeRequest('POST', '/instance/create', $payload);

            Log::info('✅ Instância criada na Evolution API', [
                'instance_name' => $instance->instance_name,
                'instance_id' => $response['instance']['instanceId'] ?? 'unknown',
                'status' => $response['instance']['status'] ?? 'unknown'
            ]);

            return $response;
        } catch (Exception $e) {
            // ✅ TRATAR ERROS ESPECÍFICOS DE INSTÂNCIA JÁ EXISTENTE
            $errorMessage = $e->getMessage();

            if (
                str_contains($errorMessage, 'already in use') ||
                str_contains($errorMessage, 'already exists') ||
                str_contains($errorMessage, '409') // Conflict
            ) {

                Log::info('ℹ️ Instância já existe (detectado por erro na criação)', [
                    'instance_name' => $instance->instance_name
                ]);

                // ✅ TENTAR OBTER DADOS DA INSTÂNCIA EXISTENTE
                try {
                    $statusResponse = $this->makeRequest('GET', "/instance/connectionState/{$instance->instance_name}");
                    return [
                        'success' => true,
                        'message' => 'Instance already exists',
                        'instance' => $statusResponse['instance'] ?? ['instanceName' => $instance->instance_name]
                    ];
                } catch (Exception $statusException) {
                    // Se não conseguir nem o status, retorna com o nome da instância
                    return [
                        'success' => true,
                        'message' => 'Instance already exists (status check failed)',
                        'instance' => ['instanceName' => $instance->instance_name]
                    ];
                }
            }

            Log::error('❌ Erro ao criar instância na Evolution API', [
                'instance_name' => $instance->instance_name,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * 🔗 Obtém o status da conexão da instância.
     */
    public function getConnectionStatus(): array
    {
        $instance = $this->getInstance();
        try {
            Log::info('📊 Verificando status da instância Evolution', [
                'instance_name' => $instance->instance_name
            ]);

            // ✅ PRIMEIRO GARANTIR QUE A INSTÂNCIA EXISTE
            $this->createInstanceIfNotExists();

            // ✅ ENDPOINT CORRETO: /instance/connectionState/{instanceName}
            $response = $this->makeRequest('GET', "/instance/connectionState/{$instance->instance_name}");

            $instanceData = $response['instance'] ?? $response;
            $state = $instanceData['state'] ?? $instanceData['status'] ?? 'close';
            $connected = ($state === 'open');

            Log::info('✅ Status da instância obtido', [
                'instance_name' => $instance->instance_name,
                'state' => $state,
                'connected' => $connected
            ]);

            return [
                'state' => $state,
                'connected' => $connected,
                'instance_name' => $instance->instance_name,
                'full_response' => ['instance' => $instanceData]
            ];
        } catch (Exception $e) {
            Log::error('❌ Erro ao verificar status Evolution', [
                'instance_name' => $instance->instance_name,
                'error' => $e->getMessage()
            ]);

            return [
                'connected' => false,
                'state' => 'disconnected',
                'full_response' => ['instance' => []]
            ];
        }
    }

    /**
     * 📱 Obtém o QR Code da instância.
     */
    public function getQrCode(): array
    {
        $instance = $this->getInstance();
        try {
            Log::info('📱 Obtendo QR Code', [
                'instance_name' => $instance->instance_name
            ]);

            // ✅ PRIMEIRO GARANTIR QUE A INSTÂNCIA EXISTE
            $this->createInstanceIfNotExists();

            // ✅ ENDPOINT CORRETO: /instance/connect/{instanceName}
            $response = $this->makeRequest('GET', "/instance/connect/{$instance->instance_name}", [], 60);

            $qrData = [
                'qrcode' => $response['qrcode'] ?? $response['base64'] ?? null,
                'base64' => $response['base64'] ?? $response['qrcode'] ?? null,
                'generated_at' => now()->toISOString()
            ];

            Log::info('✅ QR Code obtido', [
                'instance_name' => $instance->instance_name,
                'has_qrcode' => !empty($qrData['qrcode'])
            ]);

            return $qrData;
        } catch (Exception $e) {
            Log::error('❌ Erro ao obter QR Code', [
                'instance_name' => $instance->instance_name,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * 🔌 Conecta a instância.
     */
    public function connectInstance(): array
    {
        $instance = $this->getInstance();
        try {
            Log::info('🔌 Conectando instância', [
                'instance_name' => $instance->instance_name
            ]);

            // ✅ PRIMEIRO GARANTIR QUE A INSTÂNCIA EXISTE
            $this->createInstanceIfNotExists();

            // ✅ ENDPOINT CORRETO: /instance/connect/{instanceName}
            $response = $this->makeRequest('GET', "/instance/connect/{$instance->instance_name}");

            Log::info('✅ Comando de conexão enviado', [
                'instance_name' => $instance->instance_name
            ]);

            return $response;
        } catch (Exception $e) {
            Log::error('❌ Erro ao conectar instância', [
                'instance_name' => $instance->instance_name,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * 🔌 Desconecta a instância.
     */
    public function disconnectInstance(): array
    {
        $instance = $this->getInstance();
        try {
            Log::info('🔌 Desconectando instância', [
                'instance_name' => $instance->instance_name
            ]);

            // ✅ ENDPOINT CORRETO: /instance/logout/{instanceName}
            $response = $this->makeRequest('DELETE', "/instance/logout/{$instance->instance_name}");

            Log::info('✅ Instância desconectada', [
                'instance_name' => $instance->instance_name
            ]);

            return $response;
        } catch (Exception $e) {
            Log::error('❌ Erro ao desconectar instância', [
                'instance_name' => $instance->instance_name,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * 🪝 Configura o webhook na Evolution API.
     */
    public function setupWebhook(?array $events = null): array
    {
        $instance = $this->getInstance();
        try {
            $events = $events ?: [
                'QRCODE_UPDATED',
                'MESSAGES_UPSERT',
                'MESSAGES_UPDATE',
                'CONNECTION_UPDATE',
                'SEND_MESSAGE'
            ];
            $webhookUrl = $instance->webhook_url ?? $instance->getWebhookEndpoint();

            Log::info('🪝 Configurando webhook', [
                'instance_name' => $instance->instance_name,
                'webhook_url' => $webhookUrl,
                'events' => $events
            ]);

            // ✅ PRIMEIRO GARANTIR QUE A INSTÂNCIA EXISTE
            $this->createInstanceIfNotExists();

            // ✅ PAYLOAD CORRETO PARA WEBHOOK
            $payload = [
                'url' => $webhookUrl,
                'byEvents' => true,
                'base64' => true,
                'events' => $events
            ];

            // ✅ ENDPOINT CORRETO: /webhook/set/{instanceName}
            $response = $this->makeRequest('POST', "/webhook/set/{$instance->instance_name}", $payload);

            Log::info('✅ Webhook configurado', [
                'instance_name' => $instance->instance_name,
                'events' => $events
            ]);

            return $response;
        } catch (Exception $e) {
            Log::error('❌ Erro ao configurar webhook', [
                'instance_name' => $instance->instance_name,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * 🧪 Testa a conexão com a Evolution API.
     */
    public function testConnection(): array
    {
        $instance = $this->getInstance();
        $startTime = microtime(true);

        try {
            // ✅ PRIMEIRO GARANTIR QUE A INSTÂNCIA EXISTE
            $this->createInstanceIfNotExists();

            // ✅ USAR ENDPOINT SIMPLES PARA TESTE
            $response = $this->makeRequest('GET', "/instance/connectionState/{$instance->instance_name}", [], 10);
            $responseTime = round((microtime(true) - $startTime) * 1000, 2);

            return [
                'success' => true,
                'connected' => ($response['instance']['status'] ?? 'close') === 'open',
                'state' => $response['instance']['status'] ?? 'close',
                'response_time' => $responseTime,
                'instance_name' => $instance->instance_name,
                'message' => 'Conexão bem-sucedida',
                'full_response' => $response
            ];
        } catch (Exception $e) {
            $responseTime = round((microtime(true) - $startTime) * 1000, 2);

            return [
                'success' => false,
                'connected' => false,
                'response_time' => $responseTime,
                'instance_name' => $instance->instance_name,
                'message' => 'Erro de conexão: ' . $e->getMessage(),
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * ✅ OBTER INFORMAÇÕES DO GRUPO - CORRIGIDO
     */
    public function getGroupInfo(string $groupJid): array
    {
        try {
            $instance = $this->getInstance();

            Log::info('👥 Obtendo informações do grupo', [
                'group_jid' => $this->maskJid($groupJid),
                'instance' => $instance->instance_name
            ]);

            // ✅ ENDPOINT CORRETO CONFORME DOCUMENTAÇÃO
            $endpoint = "/group/findGroupInfos/{$instance->instance_name}";
            $queryParams = ['groupJid' => $groupJid];

            Log::debug('🔗 Fazendo requisição para grupo', [
                'instance' => $instance->instance_name,
                'endpoint' => $endpoint,
                'groupJid' => $this->maskJid($groupJid),
                'full_url' => $this->maskUrl(rtrim($instance->api_url, '/') . $endpoint . '?groupJid=' . urlencode($groupJid))
            ]);

            // ✅ FAZER REQUISIÇÃO GET COM QUERY PARAMETER
            $response = $this->makeRequest('GET', $endpoint, $queryParams);

            // ✅ VERIFICAR SE A RESPOSTA CONTÉM OS DADOS ESPERADOS
            if (empty($response) || (!isset($response['subject']) && !isset($response['name']))) {
                Log::warning('⚠️ Resposta da API não contém informações do grupo', [
                    'group_jid' => $this->maskJid($groupJid),
                    'response' => $response,
                    'response_keys' => is_array($response) ? array_keys($response) : 'not_array'
                ]);

                throw new Exception('Informações do grupo não encontradas na resposta da API');
            }

            Log::info('✅ Informações do grupo obtidas via API', [
                'group_jid' => $this->maskJid($groupJid),
                'group_subject' => $response['subject'] ?? 'N/A',
                'participants_count' => $response['size'] ?? 0,
                'instance' => $instance->instance_name,
                'response_structure' => array_keys($response)
            ]);

            return $response;
        } catch (Exception $e) {
            Log::error('❌ Erro ao obter informações do grupo: ' . $e->getMessage(), [
                'group_jid' => $this->maskJid($groupJid),
                'instance' => $this->instance->instance_name,
                'error' => $e->getMessage()
            ]);

            throw $e;
        }
    }

    /**
     * ✅ MÉTODO GENÉRICO PARA CONTATOS - CORRIGIDO
     */
    public function getContactInfo(string $contactJid): array
    {
        try {
            // ✅ VERIFICAR SE É GRUPO PRIMEIRO
            if (str_ends_with($contactJid, '@g.us')) {
                Log::debug('🔄 Detectado grupo, redirecionando para getGroupInfo', [
                    'contact_jid' => $this->maskJid($contactJid)
                ]);
                return $this->getGroupInfo($contactJid);
            }

            // ✅ ENDPOINT PARA CONTATOS INDIVIDUAIS
            Log::info('👤 Obtendo informações do contato individual', [
                'contact_jid' => $this->maskJid($contactJid),
                'instance' => $this->instance->instance_name
            ]);

            $response = $this->makeRequest('GET', "/chat/findContact/{$contactJid}");

            Log::info('✅ Informações do contato obtidas via API', [
                'contact_jid' => $this->maskJid($contactJid),
                'contact_name' => $response['pushName'] ?? $response['name'] ?? 'N/A',
                'instance' => $this->instance->instance_name
            ]);

            return $response;
        } catch (Exception $e) {
            Log::error('❌ Erro ao obter informações do contato: ' . $e->getMessage(), [
                'contact_jid' => $this->maskJid($contactJid),
                'instance' => $this->instance->instance_name,
                'error' => $e->getMessage()
            ]);

            throw $e;
        }
    }

    /**
     * ✅ LISTAR TODOS OS GRUPOS
     */
    public function getAllGroups(): array
    {
        try {
            $response = $this->makeRequest('GET', "/group/findGroupInfos/{$this->instance->instance_name}");

            Log::info('✅ Lista de grupos obtida', [
                'groups_count' => is_array($response) ? count($response) : 0,
                'instance' => $this->instance->instance_name
            ]);

            return is_array($response) ? $response : [];
        } catch (Exception $e) {
            Log::error('❌ Erro ao listar grupos: ' . $e->getMessage(), [
                'instance' => $this->instance->instance_name,
                'error' => $e->getMessage()
            ]);

            return [];
        }
    }

    /**
     * ✅ OBTER PARTICIPANTES DO GRUPO
     */
    public function getGroupParticipants(string $groupJid): array
    {
        try {
            $groupInfo = $this->getGroupInfo($groupJid);
            return $groupInfo['participants'] ?? [];
        } catch (Exception $e) {
            Log::error('❌ Erro ao obter participantes do grupo: ' . $e->getMessage(), [
                'group_jid' => $this->maskJid($groupJid),
                'error' => $e->getMessage()
            ]);

            return [];
        }
    }

    /**
     * 🔧 Faz uma requisição HTTP para a Evolution API.
     * ✅ SEM CRIAÇÃO AUTOMÁTICA DE INSTÂNCIA
     */
    protected function makeRequest(string $method, string $endpoint, array $data = [], ?int $timeout = null): array
    {
        try {
            if (!$this->instance) {
                throw new Exception('Instância Evolution não configurada');
            }

            $url = rtrim($this->instance->api_url, '/') . $endpoint;
            $timeout = $timeout ?? $this->timeout;

            // ✅ SEPARAR QUERY PARAMS DOS DADOS DO BODY
            $queryParams = [];
            $bodyData = [];

            // ✅ PARA GRUPOS, TRATAR groupJid COMO QUERY PARAMETER
            if (str_contains($endpoint, '/group/findGroupInfos/') && !empty($data['groupJid'])) {
                $queryParams['groupJid'] = $data['groupJid'];
                $bodyData = array_diff_key($data, ['groupJid' => '']);
            } else {
                $bodyData = $data;
            }

            Log::debug('🔗 Fazendo requisição Evolution API', [
                'instance' => $this->instance->instance_name,
                'method' => $method,
                'url' => $this->maskUrl($url),
                'endpoint' => $endpoint,
                'has_query_params' => !empty($queryParams),
                'has_body_data' => !empty($bodyData),
                'query_params' => $queryParams ? array_keys($queryParams) : []
            ]);

            // ✅ USAR LARAVEL HTTP CLIENT
            $httpClient = Http::withHeaders([
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'apikey' => $this->instance->api_key,
            ])
                ->timeout($timeout)
                ->retry($this->retryTimes, 1000);

            // ✅ FAZER REQUISIÇÃO BASEADA NO MÉTODO COM QUERY PARAMS
            $response = match (strtoupper($method)) {
                'GET' => $httpClient->get($url, $queryParams),
                'POST' => $httpClient->post($url, $bodyData),
                'PUT' => $httpClient->put($url, $bodyData),
                'PATCH' => $httpClient->patch($url, $bodyData),
                'DELETE' => $httpClient->delete($url, $bodyData),
                default => throw new Exception("Método HTTP não suportado: {$method}")
            };

            if (!$response->successful()) {
                throw new Exception("HTTP {$response->status()}: {$response->body()}");
            }

            $decodedResponse = $response->json();

            if (is_null($decodedResponse)) {
                throw new Exception('Resposta JSON inválida');
            }

            Log::debug('✅ Resposta Evolution API recebida', [
                'instance' => $this->instance->instance_name,
                'status' => $response->status(),
                'response_size' => strlen($response->body()),
                'has_data' => !empty($decodedResponse)
            ]);

            return $decodedResponse;
        } catch (Exception $e) {
            Log::error('❌ Erro na requisição Evolution API', [
                'instance' => $this->instance->instance_name ?? 'unknown',
                'method' => $method,
                'endpoint' => $endpoint,
                'error' => $e->getMessage(),
                'status' => isset($response) ? $response->status() : 'unknown',
                'response_body' => isset($response) ? $response->body() : null,
                'url' => isset($url) ? $this->maskUrl($url) : 'N/A'
            ]);

            throw new Exception("Erro na comunicação com a Evolution API: " . $e->getMessage());
        }
    }

    /**
     * ✅ MASCARAR JID PARA LOGS
     */
    private function maskJid(string $jid): string
    {
        if (empty($jid)) return '';

        $parts = explode('@', $jid);
        $number = $parts[0] ?? '';
        $domain = $parts[1] ?? '';

        if (strlen($number) >= 8) {
            $masked = substr($number, 0, 4) . '****' . substr($number, -2);
            return $masked . ($domain ? "@{$domain}" : '');
        }

        return '****' . ($domain ? "@{$domain}" : '');
    }

    /**
     * ✅ MASCARAR URL PARA LOGS (MÉTODO FALTANTE ADICIONADO)
     */
    private function maskUrl(string $url): string
    {
        if (empty($url)) return '';

        // Mascarar API key na URL se presente
        $maskedUrl = preg_replace('/apikey=[^&\\s]+/', 'apikey=****', $url);

        // Mascarar números de telefone longos
        $maskedUrl = preg_replace('/(\\d{4})\\d{4,}(\\d{2})/', '$1****$2', $maskedUrl);

        return $maskedUrl;
    }

    // ===== GETTERS =====

    public function getInstanceName(): string
    {
        return $this->getInstance()->instance_name;
    }

    public function getInstanceModel(): EvolutionInstance
    {
        return $this->getInstance();
    }

    public function isConnected(): bool
    {
        return $this->getInstance()->is_online;
    }

    /**
     * 📋 Buscar todas as instâncias da Evolution API (método estático)
     */
    public static function fetchInstancesFromEvolutionApi(): array
    {
        try {
            Log::info('📋 Buscando instâncias diretamente da Evolution API');

            // ✅ USAR CONFIGURAÇÕES DO ENV
            $apiUrl = config('chat.evolution.api_url') ?? env('EVOLUTION_API_URL');
            $apiKey = config('chat.evolution.api_key') ?? env('EVOLUTION_API_KEY');

            if (!$apiUrl || !$apiKey) {
                throw new Exception('Evolution API URL ou API Key não configuradas');
            }

            // ✅ PROCESSAR URL
            if (!str_starts_with($apiUrl, 'http://') && !str_starts_with($apiUrl, 'https://')) {
                $apiUrl = 'https://' . $apiUrl;
            }

            $url = rtrim($apiUrl, '/') . '/instance/fetchInstances';

            // ✅ USAR LARAVEL HTTP CLIENT (PADRÃO CONCORDCRM)
            $response = Http::withHeaders([
                'apikey' => $apiKey,
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ])
                ->timeout(15)
                ->retry(2, 1000)
                ->get($url);

            if (!$response->successful()) {
                throw new Exception("Erro na Evolution API - HTTP {$response->status()}: " . $response->body());
            }

            $instances = $response->json() ?? [];

            Log::info('✅ Instâncias obtidas da Evolution API', [
                'count' => count($instances),
                'instances' => array_map(fn($i) => [
                    'name' => $i['name'] ?? 'unknown',
                    'connectionStatus' => $i['connectionStatus'] ?? 'unknown',
                    'profileName' => $i['profileName'] ?? null
                ], $instances)
            ]);

            return $instances;
        } catch (Exception $e) {
            Log::error('❌ Erro ao buscar instâncias da Evolution API', [
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * 📎 Envia arquivo/mídia (imagem, documento, vídeo, etc.)
     */
    public function sendMediaMessage(string $remoteJid, string $mediaPath, string $caption = '', array $options = []): array
    {
        $instance = $this->getInstance();
        try {
            Log::info('📎 Enviando arquivo/mídia', [
                'instance_name' => $instance->instance_name,
                'remote_jid' => $this->maskJid($remoteJid),
                'media_path' => basename($mediaPath),
                'caption_preview' => substr($caption, 0, 50),
                'file_exists' => file_exists($mediaPath)
            ]);

            // ✅ VERIFICAR SE INSTÂNCIA ESTÁ CONECTADA
            if (!$instance->is_online) {
                throw new Exception("Instância '{$instance->instance_name}' não está conectada");
            }

            // ✅ VERIFICAR SE ARQUIVO EXISTE
            if (!file_exists($mediaPath)) {
                throw new Exception("Arquivo não encontrado: {$mediaPath}");
            }

            // ✅ OBTER INFORMAÇÕES DO ARQUIVO
            $fileInfo = $this->getFileInfo($mediaPath);

            // ✅ CONVERTER ARQUIVO PARA BASE64
            $mediaBase64 = base64_encode(file_get_contents($mediaPath));

            // ✅ PAYLOAD PARA ENVIO DE MÍDIA - CORRIGIDO
            $payload = [
                'number' => $remoteJid,
                'mediatype' => $fileInfo['media_type'],
                'mimetype' => $fileInfo['mime_type'],
                'media' => $mediaBase64,
                'fileName' => $fileInfo['filename'],
                'delay' => $options['delay'] ?? 0,
                'linkPreview' => $options['linkPreview'] ?? false,
            ];

            // ✅ ADICIONAR CAPTION APENAS SE NÃO ESTIVER VAZIO
            if (!empty($caption)) {
                $payload['caption'] = $caption;
            }

            // ✅ ADICIONAR MENTIONED APENAS SE TIVER VALORES
            if (!empty($options['mentioned']) && is_array($options['mentioned'])) {
                $payload['mentioned'] = $options['mentioned'];
            }

            // ✅ ADICIONAR MENTIONS EVERYONE APENAS SE TRUE
            if (!empty($options['mentionsEveryOne'])) {
                $payload['mentionsEveryOne'] = true;
            }

            // ✅ ADICIONAR QUOTED SE FORNECIDO
            if (!empty($options['quoted'])) {
                $payload['quoted'] = $options['quoted'];
            }

            // ✅ ENDPOINT PARA ENVIAR MÍDIA
            $response = $this->makeRequest('POST', "/message/sendMedia/{$instance->instance_name}", $payload);

            Log::info('✅ Arquivo/mídia enviado', [
                'instance_name' => $instance->instance_name,
                'remote_jid' => $this->maskJid($remoteJid),
                'message_id' => $response['key']['id'] ?? 'unknown',
                'file_size' => $fileInfo['file_size'],
                'mime_type' => $fileInfo['mime_type']
            ]);

            return $response;
        } catch (Exception $e) {
            Log::error('❌ Erro ao enviar arquivo/mídia', [
                'instance_name' => $instance->instance_name,
                'remote_jid' => $this->maskJid($remoteJid),
                'media_path' => basename($mediaPath),
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * 🎵 Envia áudio do WhatsApp (PTT - Push to Talk)
     */
    public function sendWhatsAppAudio(string $remoteJid, string $audioPath, array $options = []): array
    {
        $instance = $this->getInstance();
        try {
            Log::info('🎵 Enviando áudio do WhatsApp', [
                'instance_name' => $instance->instance_name,
                'remote_jid' => $this->maskJid($remoteJid),
                'audio_path' => basename($audioPath),
                'file_exists' => file_exists($audioPath)
            ]);

            // ✅ VERIFICAR SE INSTÂNCIA ESTÁ CONECTADA
            if (!$instance->is_online) {
                throw new Exception("Instância '{$instance->instance_name}' não está conectada");
            }

            // ✅ VERIFICAR SE ARQUIVO EXISTE
            if (!file_exists($audioPath)) {
                throw new Exception("Arquivo de áudio não encontrado: {$audioPath}");
            }

            // ✅ VERIFICAR SE É ARQUIVO DE ÁUDIO
            $fileInfo = $this->getFileInfo($audioPath);
            if (!str_starts_with($fileInfo['mime_type'], 'audio/') && $fileInfo['mime_type'] !== 'video/webm') {
                throw new Exception("Arquivo não é um áudio válido: {$fileInfo['mime_type']}");
            }

            // ✅ CONVERTER ÁUDIO PARA BASE64
            $audioBase64 = base64_encode(file_get_contents($audioPath));

            // ✅ PAYLOAD CORRETO PARA WHATSAPP AUDIO - SEGUINDO DOCUMENTAÇÃO
            $payload = [
                'number' => $remoteJid,
                'audio' => $audioBase64, // Base64 do áudio
                'delay' => $options['delay'] ?? 0,
                'linkPreview' => $options['linkPreview'] ?? false,
            ];

            // ✅ ADICIONAR MENTIONED APENAS SE TIVER VALORES
            if (!empty($options['mentioned']) && is_array($options['mentioned'])) {
                $payload['mentioned'] = $options['mentioned'];
            }

            // ✅ ADICIONAR MENTIONS EVERYONE APENAS SE TRUE
            if (!empty($options['mentionsEveryOne'])) {
                $payload['mentionsEveryOne'] = true;
            }

            // ✅ ADICIONAR QUOTED SE FORNECIDO
            if (!empty($options['quoted'])) {
                $payload['quoted'] = $options['quoted'];
            }

            // ✅ ENDPOINT CORRETO PARA WHATSAPP AUDIO
            $response = $this->makeRequest('POST', "/message/sendWhatsAppAudio/{$instance->instance_name}", $payload);

            Log::info('✅ Áudio do WhatsApp enviado', [
                'instance_name' => $instance->instance_name,
                'remote_jid' => $this->maskJid($remoteJid),
                'message_id' => $response['key']['id'] ?? 'unknown',
                'file_size' => $fileInfo['file_size']
            ]);

            return $response;
        } catch (Exception $e) {
            Log::error('❌ Erro ao enviar áudio do WhatsApp', [
                'instance_name' => $instance->instance_name,
                'remote_jid' => $this->maskJid($remoteJid),
                'audio_path' => basename($audioPath),
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }
    /**
     * 💬 Envia mensagem de texto - CORRIGIDO
     */
    public function sendTextMessage(string $remoteJid, string $text): array
    {
        $instance = $this->getInstance();
        try {
            Log::info('💬 Enviando mensagem de texto', [
                'instance_name' => $instance->instance_name,
                'remote_jid' => $this->maskJid($remoteJid),
                'text_preview' => substr($text, 0, 50)
            ]);

            // ✅ VERIFICAR SE INSTÂNCIA ESTÁ CONECTADA
            if (!$instance->is_online) {
                throw new Exception("Instância '{$instance->instance_name}' não está conectada");
            }

            // ✅ PAYLOAD PARA MENSAGEM DE TEXTO - CORRIGIDO
            $payload = [
                'number' => $remoteJid,
                'text' => $text,
                'delay' => 0,
                'linkPreview' => false,
            ];

            // ✅ NÃO INCLUIR mentioned E mentionsEveryOne SE VAZIOS

            // ✅ ENDPOINT PARA ENVIAR TEXTO
            $response = $this->makeRequest('POST', "/message/sendText/{$instance->instance_name}", $payload);

            Log::info('✅ Mensagem de texto enviada', [
                'instance_name' => $instance->instance_name,
                'remote_jid' => $this->maskJid($remoteJid),
                'message_id' => $response['key']['id'] ?? 'unknown'
            ]);

            return $response;
        } catch (Exception $e) {
            Log::error('❌ Erro ao enviar mensagem de texto', [
                'instance_name' => $instance->instance_name,
                'remote_jid' => $this->maskJid($remoteJid),
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * 📄 Método genérico para envio de arquivos (detecta tipo automaticamente)
     */
    public function sendFile(string $remoteJid, string $filePath, string $caption = '', array $options = []): array
    {
        try {
            $fileInfo = $this->getFileInfo($filePath);

            Log::info('📄 Enviando arquivo (auto-detectado)', [
                'file_path' => basename($filePath),
                'mime_type' => $fileInfo['mime_type'],
                'media_type' => $fileInfo['media_type'],
                'file_size' => $fileInfo['file_size']
            ]);

            // ✅ PARA ÁUDIOS, USAR ENDPOINT ESPECÍFICO DE WHATSAPP AUDIO
            if ($fileInfo['media_type'] === 'audio' || $fileInfo['mime_type'] === 'video/webm') {
                // ✅ SEMPRE USAR WHATSAPP AUDIO PARA ÁUDIOS
                Log::info('🎵 Detectado áudio, usando endpoint sendWhatsAppAudio', [
                    'mime_type' => $fileInfo['mime_type']
                ]);

                return $this->sendWhatsAppAudio($remoteJid, $filePath, $options);
            }

            // ✅ PARA OUTROS TIPOS (IMAGEM, VÍDEO, DOCUMENTO), USAR ENDPOINT DE MÍDIA
            return $this->sendMediaMessage($remoteJid, $filePath, $caption, $options);
        } catch (Exception $e) {
            Log::error('❌ Erro ao enviar arquivo', [
                'file_path' => basename($filePath),
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * 🔍 Obter informações do arquivo
     */
    private function getFileInfo(string $filePath): array
    {
        try {
            if (!file_exists($filePath)) {
                throw new Exception("Arquivo não encontrado: {$filePath}");
            }

            $fileSize = filesize($filePath);
            $mimeType = mime_content_type($filePath) ?: 'application/octet-stream';
            $filename = basename($filePath);

            // ✅ DETERMINAR TIPO DE MÍDIA BASEADO NO MIME TYPE
            $mediaType = match (true) {
                str_starts_with($mimeType, 'image/') => 'image',
                str_starts_with($mimeType, 'video/') => 'video',
                str_starts_with($mimeType, 'audio/') => 'audio',
                $mimeType === 'application/pdf' => 'document',
                str_starts_with($mimeType, 'application/') => 'document',
                str_starts_with($mimeType, 'text/') => 'document',
                default => 'document'
            };

            return [
                'filename' => $filename,
                'file_size' => $fileSize,
                'mime_type' => $mimeType,
                'media_type' => $mediaType,
                'file_path' => $filePath
            ];
        } catch (Exception $e) {
            throw new Exception("Erro ao obter informações do arquivo: " . $e->getMessage());
        }
    }

    /**
     * ✅ VERIFICAR SE NÚMEROS EXISTEM NO WHATSAPP
     */
    public function checkWhatsAppNumbers(array $numbers): array
    {
        $instance = $this->getInstance();
        try {
            Log::info('🔍 Verificando se números existem no WhatsApp', [
                'instance_name' => $instance->instance_name,
                'numbers_count' => count($numbers),
                'numbers_preview' => array_map(fn($n) => substr($n, 0, 6) . '****', $numbers)
            ]);

            // ✅ ENDPOINT PARA VERIFICAR NÚMEROS
            $response = $this->makeRequest('POST', "/chat/whatsappNumbers/{$instance->instance_name}", [
                'numbers' => $numbers
            ]);

            Log::info('✅ Verificação de números concluída', [
                'instance_name' => $instance->instance_name,
                'total_checked' => count($response),
                'results' => array_map(fn($r) => [
                    'number' => substr($r['number'] ?? '', 0, 6) . '****',
                    'exists' => $r['exists'] ?? false,
                    'jid' => $r['jid'] ? substr($r['jid'], 0, 10) . '****' : null
                ], $response)
            ]);

            return $response;
        } catch (Exception $e) {
            Log::error('❌ Erro ao verificar números WhatsApp', [
                'instance_name' => $instance->instance_name,
                'numbers_count' => count($numbers),
                'error' => $e->getMessage()
            ]);

            throw $e;
        }
    }

    /**
     * ✅ VERIFICAR SE UM ÚNICO NÚMERO EXISTE NO WHATSAPP
     */
    public function checkSingleWhatsAppNumber(string $number): array
    {
        try {
            $results = $this->checkWhatsAppNumbers([$number]);
            return $results[0] ?? ['exists' => false, 'number' => $number, 'jid' => null];
        } catch (Exception $e) {
            Log::error('❌ Erro ao verificar número único WhatsApp', [
                'number' => substr($number, 0, 6) . '****',
                'error' => $e->getMessage()
            ]);

            return [
                'exists' => false,
                'number' => $number,
                'jid' => null,
                'error' => $e->getMessage()
            ];
        }
    }
}
