<?php

namespace Modules\Chat\Services;

use Illuminate\Support\Facades\Log;
use Modules\Chat\Models\EvolutionWebhook;
use Modules\Chat\Models\EvolutionInstance;
use Modules\Chat\Models\Message;
use Modules\Chat\Models\Conversation;
use Modules\Contacts\Models\Contact;
use Illuminate\Support\Facades\Cache; 
use Exception;

class EvolutionWebhookProcessorService
{
    protected ChatRedisService $chatRedis;
    protected EvolutionApiService $evolutionApi;
    protected EvolutionSyncService $evolutionSyncService;

    public function __construct(
        ChatRedisService $chatRedis,
        EvolutionApiService $evolutionApi,
        EvolutionSyncService $evolutionSyncService
    ) {
        $this->chatRedis = $chatRedis;
        $this->evolutionApi = $evolutionApi;
        $this->evolutionSyncService = $evolutionSyncService;
    }

    public function processWebhook(EvolutionWebhook $webhook): array
    {
        try {
            $instance = EvolutionInstance::where('instance_name', $webhook->instance_name)->first();

            if (!$instance) {
                return ['action' => 'error', 'reason' => "Instância não encontrada: {$webhook->instance_name}"];
            }

            $this->evolutionSyncService->setInstance($instance);
            $normalizedEventType = $this->normalizeEventType($webhook->event_type);

            $result = match ($normalizedEventType) {
                'messages_upsert', 'messages' => $this->processMessageEvent($webhook),
                'messages_update', 'message_status' => $this->processMessageUpdateEvent($webhook),
                'connection_update', 'connection_status' => $this->processConnectionUpdateEvent($webhook),
                'qrcode_updated' => $this->processQrCodeUpdateEvent($webhook),
                'application_startup', 'instance_status' => $this->processInstanceStatusEvent($webhook),
                'chats_update', 'chats_upsert' => $this->processChatUpdateEvent($webhook),
                'contacts_update', 'contacts_upsert' => $this->processContactUpdateEvent($webhook),

                'messages_set', 'contacts_set', 'chats_set' => ['action' => 'ignored', 'reason' => 'bulk_event_not_processed'],
                'presence_update' => ['action' => 'ignored', 'reason' => 'presence_not_tracked'],
                'send_message' => ['action' => 'ignored', 'reason' => 'outbound_message_confirmation'],

                default => ['action' => 'unsupported', 'reason' => "Tipo de evento não suportado: {$webhook->event_type}"],
            };

            $this->updateWebhookRelationships($webhook, $result);

            if (in_array($result['action'], ['outbound_message_created', 'message_created', 'status_updated'])) {
                $this->publishToFrontend($webhook, $result);
            }

            return $result;
        } catch (Exception $e) {
            Log::error('Erro ao processar webhook', [
                'webhook_db_id' => $webhook->id,
                'error' => $e->getMessage()
            ]);

            return ['action' => 'error', 'reason' => $e->getMessage()];
        }
    }

    protected function processMessageEvent(EvolutionWebhook $webhook): array
    {
        try {
            $payload = $webhook->payload;
            $data = $payload['data'] ?? [];
            $key = $data['key'] ?? [];
            $message = $data['message'] ?? [];

            $originalFromMe = $key['fromMe'] ?? false;
            $originalRemoteJid = $key['remoteJid'] ?? '';
            $participant = $key['participant'] ?? null;

            $isFromMe = $this->determineIfFromMe($originalFromMe, $participant, $webhook->instance_name);

            if (empty($originalRemoteJid)) {
                return ['action' => 'error', 'reason' => 'remoteJid não encontrado'];
            }

            $normalizedRemoteJid = $this->normalizeJid($originalRemoteJid);
            $isGroup = str_ends_with($normalizedRemoteJid, '@g.us');

            if ($isFromMe) {
                return $this->handleOutgoingMessage($webhook, $data, $key, $message, $normalizedRemoteJid, $isGroup);
            }

            return $this->handleIncomingMessage($webhook, $data, $key, $message, $normalizedRemoteJid, $isGroup);
        } catch (Exception $e) {
            Log::error('Erro ao processar mensagem', [
                'webhook_id' => $webhook->id,
                'error' => $e->getMessage()
            ]);

            return ['action' => 'error', 'reason' => $e->getMessage()];
        }
    }

    protected function handleOutgoingMessage(EvolutionWebhook $webhook, array $data, array $key, array $message, string $normalizedRemoteJid, bool $isGroup): array
    {
        try {
            // Ignorar mensagens para próprios LIDs
            $cleanNumber = preg_replace('/[^0-9]/', '', $normalizedRemoteJid);
            $yourLids = ['90744203251776', '236485311766672'];

            if (in_array($cleanNumber, $yourLids)) {
                return ['action' => 'ignored', 'reason' => 'self_to_self_message'];
            }

            // Buscar ou criar conversa
            $conversation = $this->findOrCreateConversation($webhook, $normalizedRemoteJid, $cleanNumber, $isGroup, $data, $key, true);

            if (!$conversation) {
                return ['action' => 'error', 'reason' => 'Falha ao criar/encontrar conversa'];
            }

            // Criar mensagem
            $messageModel = $this->createMessage($conversation, $data, $key, $message, $isGroup, true);

            // Disparar job de foto se necessário
            if (!$isGroup && $conversation->contact_id) {
                $this->dispatchContactPhotoJobIfNeeded($conversation->contact, $this->extractPhoneFromJid($normalizedRemoteJid), $conversation);
            }

            return [
                'action' => 'outbound_message_created',
                'conversation_id' => $conversation->id,
                'message_id' => $messageModel->id,
                'contact_id' => $conversation->contact_id,
                'from_me' => true,
                'is_group' => $isGroup
            ];
        } catch (Exception $e) {
            Log::error('Erro ao processar mensagem enviada', [
                'webhook_id' => $webhook->id,
                'error' => $e->getMessage()
            ]);

            return ['action' => 'error', 'reason' => $e->getMessage()];
        }
    }

    protected function handleIncomingMessage(EvolutionWebhook $webhook, array $data, array $key, array $message, string $normalizedRemoteJid, bool $isGroup): array
    {
        try {
            // Buscar conversa existente (incluindo fechadas)
            $conversation = Conversation::where('evolution_instance_name', $webhook->instance_name)
                ->where(function ($query) use ($normalizedRemoteJid) {
                    $query->where('whatsapp_remote_jid', $normalizedRemoteJid)
                        ->orWhere('whatsapp_chat_id', $normalizedRemoteJid);
                })
                ->orderBy('updated_at', 'desc')
                ->first();

            // Lógica de reabertura ou nova conversa
            if ($conversation && in_array($conversation->status, ['closed', 'archived'])) {
                $conversation = $this->handleClosedConversation($conversation, $webhook, $data, $key, $isGroup, $normalizedRemoteJid);
            }

            // Criar nova conversa se não existir
            if (!$conversation) {
                $conversation = $this->findOrCreateConversation($webhook, $normalizedRemoteJid, null, $isGroup, $data, $key, false);
            }

            if (!$conversation) {
                return ['action' => 'error', 'reason' => 'Falha ao criar/encontrar conversa'];
            }

            // Criar mensagem
            $messageModel = $this->createMessage($conversation, $data, $key, $message, $isGroup, false);

            // Disparar job de foto se necessário
            if (!$isGroup && $conversation->contact_id) {
                $this->dispatchContactPhotoJobIfNeeded($conversation->contact, $this->extractPhoneFromJid($normalizedRemoteJid), $conversation);
            }

            return [
                'action' => 'message_created',
                'conversation_id' => $conversation->id,
                'message_id' => $messageModel->id,
                'contact_id' => $messageModel->sender_id,
                'from_me' => false,
                'is_group' => $isGroup
            ];
        } catch (Exception $e) {
            Log::error('Erro ao processar mensagem recebida', [
                'webhook_id' => $webhook->id,
                'error' => $e->getMessage()
            ]);

            return ['action' => 'error', 'reason' => $e->getMessage()];
        }
    }

    private function findOrCreateConversation(EvolutionWebhook $webhook, string $normalizedRemoteJid, ?string $cleanNumber, bool $isGroup, array $data, array $key, bool $fromMe): ?Conversation
    {
        // Buscar conversa existente
        $conversation = Conversation::where('evolution_instance_name', $webhook->instance_name)
            ->where(function ($query) use ($normalizedRemoteJid, $cleanNumber) {
                $query->where('whatsapp_remote_jid', $normalizedRemoteJid)
                    ->orWhere('whatsapp_chat_id', $normalizedRemoteJid);

                if ($cleanNumber && strlen($cleanNumber) >= 10) {
                    $query->orWhere('whatsapp_chat_id', $cleanNumber)
                        ->orWhere('whatsapp_remote_jid', $cleanNumber);
                }
            })
            ->first();

        if ($conversation) {
            return $conversation;
        }

        // Criar nova conversa
        $chatData = [
            'id' => $normalizedRemoteJid,
            'name' => $fromMe ? null : ($data['pushName'] ?? null),
            'isGroup' => $isGroup,
            'participant' => $key['participant'] ?? null,
            'groupName' => $isGroup ? $this->getGroupName($normalizedRemoteJid, $webhook->instance_name) : null,
            'senderName' => $isGroup && !$fromMe ? ($data['pushName'] ?? null) : null,
            'fromMe' => $fromMe,
        ];

        $chatSyncResult = $this->evolutionSyncService->syncSingleChat($chatData);
        return Conversation::find($chatSyncResult['conversation_id']);
    }

    private function handleClosedConversation(Conversation $conversation, EvolutionWebhook $webhook, array $data, array $key, bool $isGroup, string $normalizedRemoteJid): ?Conversation
    {
        $closedAt = $conversation->closed_at;
        $oneHourAgo = now()->subHour();

        if ($closedAt && $closedAt->lt($oneHourAgo)) {
            // Criar nova conversa
            return $this->createNewConversationFromClosed($conversation, $webhook, $data, $key, $isGroup);
        } else {
            // Reabrir conversa
            $conversation->update([
                'status' => 'active',
                'closed_by' => null,
                'closed_at' => null,
                'close_reason' => null,
                'updated_at' => now()
            ]);

            $this->createSystemMessage($conversation, 'conversation_auto_reopened', 'Conversa reaberta automaticamente.');

            try {
                $this->chatRedis->publishConversationUpdate($conversation->fresh(), 'conversation_auto_reopened');
            } catch (Exception $e) {
                // Silenciar erro de publicação
            }

            return $conversation;
        }
    }

    private function createNewConversationFromClosed(Conversation $oldConversation, EvolutionWebhook $webhook, array $data, array $key, bool $isGroup): ?Conversation
    {
        try {
            $normalizedRemoteJid = $this->normalizeJid($key['remoteJid'] ?? '');

            $chatData = [
                'id' => $normalizedRemoteJid,
                'name' => $data['pushName'] ?? $oldConversation->whatsapp_profile_name ?? null,
                'isGroup' => $isGroup,
                'participant' => $key['participant'] ?? null,
                'groupName' => $isGroup ? $this->getGroupName($normalizedRemoteJid, $webhook->instance_name) : null,
                'senderName' => $isGroup ? ($data['pushName'] ?? null) : null,
                'fromMe' => false,
            ];

            $chatSyncResult = $this->evolutionSyncService->syncSingleChat($chatData);
            $newConversation = Conversation::find($chatSyncResult['conversation_id']);

            if ($newConversation) {
                // Copiar informações relevantes
                $updateData = [];
                if ($oldConversation->contact_id) $updateData['contact_id'] = $oldConversation->contact_id;
                if ($oldConversation->assigned_user_id) $updateData['assigned_user_id'] = $oldConversation->assigned_user_id;

                if (!empty($updateData)) {
                    $newConversation->update($updateData);
                }

                $this->createSystemMessage($newConversation, 'new_conversation_from_closed', 'Nova conversa iniciada.');
            }

            return $newConversation;
        } catch (Exception $e) {
            Log::error('Erro ao criar nova conversa', ['error' => $e->getMessage()]);
            return null;
        }
    }

    private function createMessage(Conversation $conversation, array $data, array $key, array $message, bool $isGroup, bool $fromMe): Message
    {
        $messageDataForSync = [
            'key' => $key,
            'message' => $message,
            'messageTimestamp' => $data['messageTimestamp'] ?? time(),
            'pushName' => $data['pushName'] ?? null,
            'status' => $data['status'] ?? null,
            'messageType' => $data['messageType'] ?? 'conversation',
            'instanceId' => $data['instanceId'] ?? null,
            'source' => $data['source'] ?? 'webhook',
            'base64' => $message['base64'] ?? null,
            'fromMe' => $fromMe,
            'participant' => $key['participant'] ?? null,
            'isGroup' => $isGroup,
            'groupName' => $isGroup ? $this->getGroupName($this->normalizeJid($key['remoteJid'] ?? ''), $data['instanceId'] ?? '') : null,
            'senderName' => $isGroup && !$fromMe ? ($data['pushName'] ?? null) : null
        ];

        return $this->evolutionSyncService->createMessageFromApiData($conversation, $messageDataForSync);
    }

    private function createSystemMessage(Conversation $conversation, string $systemType, string $content, array $metadata = []): void
    {
        try {
            Message::create([
                'conversation_id' => $conversation->id,
                'sender_type' => 'system',
                'sender_id' => null,
                'content' => $content,
                'message_type' => 'system',
                'is_from_contact' => false,
                'status' => 'sent',
                'sent_at' => now(),
                'metadata' => array_merge(['is_system_message' => true, 'system_type' => $systemType], $metadata)
            ]);

            $conversation->update(['last_message_at' => now()]);
        } catch (Exception $e) {
            // Silenciar erro
        }
    }

    protected function getGroupName(string $groupJid, string $instanceName): ?string
    {
        try {
            $instance = EvolutionInstance::where('instance_name', $instanceName)->first();
            if (!$instance) return 'Grupo WhatsApp';

            $this->evolutionApi->setInstance($instance);
            $groupInfo = $this->evolutionApi->getGroupInfo($groupJid);
            return $groupInfo['subject'] ?? $groupInfo['name'] ?? 'Grupo WhatsApp';
        } catch (Exception $e) {
            // Fallback para banco local
            $conversation = Conversation::where('whatsapp_chat_id', $groupJid)->where('type', 'external')->first();
            return $conversation?->whatsapp_profile_name ?? 'Grupo WhatsApp';
        }
    }

    protected function determineIfFromMe(bool $originalFromMe, ?string $participant, string $instanceName): bool
    {
        if ($originalFromMe) return true;
        if (!$participant) return $originalFromMe;

        $yourLids = ['90744203251776', '236485311766672'];
        $participantNumber = preg_replace('/[^0-9]/', '', $participant);

        if (in_array($participantNumber, $yourLids)) {
            return true;
        }

        $instance = EvolutionInstance::where('instance_name', $instanceName)->first();
        if ($instance) {
            $instanceNumber = preg_replace('/[^0-9]/', '', $instance->whatsapp_number ?? '');
            $yourNumbers = [$instanceNumber, '556696322859', '6696322859'];

            foreach ($yourNumbers as $yourNumber) {
                if (!empty($yourNumber) && $participantNumber === $yourNumber) {
                    return true;
                }
            }
        }

        return $originalFromMe;
    }

    protected function processMessageUpdateEvent(EvolutionWebhook $webhook): array
    {
        try {
            $data = $webhook->payload['data'] ?? [];
            $whatsappMessageId = $data['keyId'] ?? $data['messageId'] ?? null;
            $newEvolutionStatus = $data['status'] ?? null;

            if (!$whatsappMessageId || !$newEvolutionStatus) {
                return ['action' => 'ignored', 'reason' => 'dados_insuficientes'];
            }

            $message = Message::where('whatsapp_message_id', $whatsappMessageId)->first();
            if (!$message) {
                return ['action' => 'message_not_found', 'whatsapp_message_id' => $whatsappMessageId];
            }

            $mappedNewStatus = $this->mapWhatsAppStatusToEnum($newEvolutionStatus);
            $currentDbStatus = $message->delivery_status;

            if ($mappedNewStatus === 'failed' && $currentDbStatus !== 'failed') {
                $this->updateMessageStatus($message, $mappedNewStatus);
                return ['action' => 'status_updated', 'message_id' => $message->id, 'old_status' => $currentDbStatus, 'new_status' => $mappedNewStatus];
            }

            if ($this->getStatusRank($mappedNewStatus) <= $this->getStatusRank($currentDbStatus)) {
                return ['action' => 'no_change', 'current_status' => $currentDbStatus];
            }

            $this->updateMessageStatus($message, $mappedNewStatus);
            return ['action' => 'status_updated', 'message_id' => $message->id, 'old_status' => $currentDbStatus, 'new_status' => $mappedNewStatus];
        } catch (Exception $e) {
            return ['action' => 'error', 'reason' => $e->getMessage()];
        }
    }

    private function updateMessageStatus(Message $message, string $newStatus): void
    {
        $message->update([
            'delivery_status' => $newStatus,
            'status' => $newStatus,
            'delivered_at' => ($newStatus === 'delivered' && !$message->delivered_at) ? now() : $message->delivered_at,
            'read_at' => ($newStatus === 'read' && !$message->read_at) ? now() : $message->read_at
        ]);

        try {
            $this->chatRedis->publishWhatsAppStatusUpdate($message, $message->getOriginal('delivery_status'), $newStatus);
        } catch (Exception $e) {
            // Silenciar erro de publicação
        }
    }

    protected function publishToFrontend(EvolutionWebhook $webhook, array $result): void
    {
        if (!$this->chatRedis->isRedisAvailable()) return;

        try {
            $eventData = [
                'original_webhook_event' => $webhook->event_type,
                'instance_name' => $webhook->instance_name,
                'result_action' => $result['action'] ?? 'processed',
                'timestamp' => now()->toISOString(),
                'webhook_db_id' => $webhook->id,
            ];

            if (in_array($webhook->event_type, ['messages', 'messages_upsert', 'messages.upsert']) && isset($result['message_id'])) {
                $message = Message::with(['conversation', 'attachments'])->find($result['message_id']);
                if ($message && $message->conversation) {
                    $this->publishMessageEvent($message, $eventData);
                }
            }

            if (in_array($webhook->event_type, ['messages_update', 'messages.update']) && isset($result['message_id'])) {
                $message = Message::find($result['message_id']);
                if ($message) {
                    $this->publishStatusEvent($message, $result, $eventData);
                }
            }
        } catch (Exception $e) {
            // Silenciar erro de publicação
        }
    }

    private function publishMessageEvent(Message $message, array $eventData): void
    {
        $conversation = $message->conversation;
        $isFromMe = $message->whatsapp_key_from_me;
        $messageMetadata = is_string($message->metadata) ? json_decode($message->metadata, true) : $message->metadata;
        $messageSource = $messageMetadata['source'] ?? 'unknown';

        $externalAppSources = ['desktop', 'android', 'ios', 'web'];
        
        if ($isFromMe && in_array($messageSource, $externalAppSources)) {
            $eventTypeToPublish = $this->chatRedis->getEventType('OUTBOUND_WHATSAPP_MESSAGE');
            $sourceIdentifier = 'whatsapp_external_app';
        } else {
            $eventTypeToPublish = $this->chatRedis->getEventType('NEW_WHATSAPP_MESSAGE');
            $sourceIdentifier = $isFromMe ? 'crm_sent_message' : 'whatsapp_incoming';
        }

        $eventData['event_type'] = $eventTypeToPublish;
        $eventData['source'] = $sourceIdentifier;
        $eventData['message'] = $message->toFrontendArray();
        $eventData['conversation'] = $conversation->toArray();
        $eventData['conversation_id'] = $conversation->id;

        $this->chatRedis->publishToStream($eventTypeToPublish, $eventData);
        $this->chatRedis->publishToStream($eventTypeToPublish, $eventData, "conversation_{$conversation->id}");
    }

    private function publishStatusEvent(Message $message, array $result, array $eventData): void
    {
        $eventData['event_type'] = $this->chatRedis->getEventType('WHATSAPP_STATUS_UPDATE');
        $eventData['message_id'] = $result['message_id'];
        $eventData['conversation_id'] = $message->conversation_id;
        $eventData['old_status'] = $result['old_status'] ?? null;
        $eventData['new_status'] = $result['new_status'] ?? null;

        $this->chatRedis->publishToStream($eventData['event_type'], $eventData);
        $this->chatRedis->publishToStream($eventData['event_type'], $eventData, "conversation_{$message->conversation_id}");
    }

    protected function dispatchContactPhotoJobIfNeeded(Contact $contact, string $phoneNumber, Conversation $conversation): void
    {
        if (!$contact->needsWhatsappProfilePhotoUpdate() || !$conversation->chat_channel_id) {
            return;
        }

        $cacheKey = "photo_job_pending_{$contact->id}";
        if (Cache::get($cacheKey, false)) {
            return;
        }

        Cache::put($cacheKey, true, 600);

        try {
            \Modules\Chat\Jobs\FetchContactPhotoJob::dispatch($contact, $phoneNumber, $conversation->id, false)
                ->onQueue('photos')
                ->delay(now()->addSeconds(rand(5, 15)));
        } catch (Exception $e) {
            Cache::forget($cacheKey);
        }
    }

    // ===== MÉTODOS AUXILIARES =====

    private function normalizeJid(string $jid): string
    {
        if (str_contains($jid, '@lid')) {
            $numbers = preg_replace('/[^0-9]/', '', explode('@', $jid)[0]);
            if (strlen($numbers) >= 10) {
                return $numbers . '@s.whatsapp.net';
            }
        }

        if (str_ends_with($jid, '@s.whatsapp.net') || str_ends_with($jid, '@g.us')) {
            return $jid;
        }

        $numbers = preg_replace('/[^0-9]/', '', explode('@', $jid)[0]);
        if (strlen($numbers) >= 10) {
            return $numbers . '@s.whatsapp.net';
        }

        return $jid;
    }

    protected function normalizeEventType(string $eventType): string
    {
        $eventMapping = [
            'MESSAGES-UPSERT' => 'messages_upsert', 'MESSAGES-UPDATE' => 'messages_update',
            'CONNECTION-UPDATE' => 'connection_update', 'QRCODE-UPDATED' => 'qrcode_updated',
            'APPLICATION-STARTUP' => 'application_startup', 'CHATS-UPDATE' => 'chats_update',
            'CHATS-UPSERT' => 'chats_upsert', 'CONTACTS-UPDATE' => 'contacts_update',
            'CONTACTS-UPSERT' => 'contacts_upsert', 'messages' => 'messages_upsert',
            'message_status' => 'messages_update', 'connection_status' => 'connection_update',
            'instance_status' => 'application_startup', 'messages.upsert' => 'messages_upsert',
            'messages.update' => 'messages_update',
        ];

        return $eventMapping[$eventType] ?? str_replace(['-', '.'], '_', strtolower($eventType));
    }

    private function getStatusRank(string $status): int
    {
        return match ($status) {
            'sending' => 10, 'sent' => 20, 'delivered' => 30, 'read' => 40, 'failed' => 100, default => 0,
        };
    }

    private function mapWhatsAppStatusToEnum(string $whatsappStatus): string
    {
        $mapping = [
            'PENDING' => 'sending', 'SERVER_ACK' => 'sent', 'DELIVERY_ACK' => 'delivered',
            'READ' => 'read', 'ERROR' => 'failed', 'FAILED' => 'failed'
        ];

        return $mapping[$whatsappStatus] ?? 'sent';
    }

    protected function updateWebhookRelationships(EvolutionWebhook $webhook, array $result): void
    {
        $updates = [];
        if (isset($result['message_id'])) $updates['message_id'] = $result['message_id'];
        if (isset($result['conversation_id'])) $updates['conversation_id'] = $result['conversation_id'];
        if (isset($result['contact_id'])) $updates['contact_id'] = $result['contact_id'];

        if (!empty($updates)) {
            $webhook->update($updates);
        }
    }

    protected function maskJid(string $jid): string
    {
        if (empty($jid)) return '';
        $parts = explode('@', $jid);
        $number = $parts[0] ?? '';
        $domain = $parts[1] ?? '';

        if (strlen($number) >= 8) {
            $masked = substr($number, 0, 4) . '****' . substr($number, -2);
            return $masked . ($domain ? "@{$domain}" : '');
        }

        return '****' . ($domain ? "@{$domain}" : '');
    }

    private function extractPhoneFromJid(string $jid): string
    {
        $phone = preg_replace('/[^0-9]/', '', explode('@', $jid)[0]);
        if (strlen($phone) === 11 && !str_starts_with($phone, '55')) {
            $phone = '55' . $phone;
        }
        return $phone;
    }

    // ===== MÉTODOS DE EVENTOS SIMPLIFICADOS =====

    protected function processConnectionUpdateEvent(EvolutionWebhook $webhook): array
    {
        return $this->evolutionSyncService->handleConnectionStatusUpdate($webhook->payload['data'] ?? []);
    }

    protected function processQrCodeUpdateEvent(EvolutionWebhook $webhook): array
    {
        return $this->evolutionSyncService->handleQrCodeUpdate($webhook->payload['data'] ?? []);
    }

    protected function processInstanceStatusEvent(EvolutionWebhook $webhook): array
    {
        return $this->evolutionSyncService->handleInstanceStatusUpdate($webhook->payload['data'] ?? []);
    }

    protected function processChatUpdateEvent(EvolutionWebhook $webhook): array
    {
        return $this->evolutionSyncService->handleChatUpdateEvent($webhook->payload['data'] ?? []);
    }

    protected function processContactUpdateEvent(EvolutionWebhook $webhook): array
    {
        return $this->evolutionSyncService->handleContactUpdateEvent($webhook->payload['data'] ?? []);
    }
}