<?php
// modules/Chat/Services/MessageService.php

namespace Modules\Chat\Services;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Exception;
use Modules\Chat\Models\Conversation;
use Modules\Chat\Models\Message;
use Modules\Chat\Models\MessageAttachment;
use Modules\Users\Models\User;
use Modules\Chat\Jobs\SendWhatsAppMessageJob;

class MessageService
{
    public function __construct(
        protected ChatRedisService $redis,
        protected EvolutionApiService $evolution,
        protected FileService $fileService
    ) {}

    /**
     * ✅ OBTER MENSAGENS - SEM CARREGAR SENDER VIA WITH
     */
    public function getMessages(int $conversationId, array $params): array
    {
        $conversation = Conversation::findOrFail($conversationId);
        $this->validateAccess($conversation);

        $perPage = min((int) ($params['per_page'] ?? 50), 100);

        $query = Message::where('conversation_id', $conversationId)
            ->with([
                'attachments:id,message_id,file_name,file_path,file_type,file_size,mime_type,download_status'
            ])
            ->select([
                'id',
                'conversation_id',
                'sender_id',
                'sender_type',
                'content',
                'message_type',
                'is_from_contact',
                'sent_at',
                'read_at',
                'status',
                'delivery_status',
                'metadata',
                'whatsapp_message_id'
            ]);

        if ($beforeId = $params['before_id'] ?? null) {
            $query->where('id', '<', $beforeId);
        }

        $messages = $query->orderByDesc('sent_at')->limit($perPage + 1)->get();
        $hasMore = $messages->count() > $perPage;

        if ($hasMore) $messages = $messages->take($perPage);
        $messages = $messages->reverse()->values();

        return [
            'data' => $messages->map(fn($m) => $this->formatMessage($m))->toArray(),
            'pagination' => [
                'has_more' => $hasMore,
                'before_id' => $messages->first()?->id
            ]
        ];
    }

    /**
     * ✅ ENVIAR MENSAGEM - SIMPLIFICADO
     */
    public function sendMessage(int $conversationId, array $data, array $files = []): array
    {
        $conversation = Conversation::findOrFail($conversationId);
        $this->validateAccess($conversation);

        if (empty($data['content']) && empty($files['attachments'])) {
            throw new Exception('Conteúdo ou anexos são obrigatórios');
        }

        return DB::transaction(function () use ($conversation, $data, $files) {
            $attachment = null;
            if (!empty($files['attachments'])) {
                $attachment = $this->fileService->processUpload($files['attachments'][0]);
            }

            $message = $this->createMessage($conversation, $data, $attachment);

            if ($conversation->isWhatsApp()) {
                $this->sendToWhatsApp($conversation, $message, $attachment);
            }

            $this->redis->notifyNewMessage($message, $conversation);

            return [
                'success' => true,
                'message' => $this->formatMessage($message->load('attachments')),
                'conversation_id' => $conversation->id
            ];
        });
    }

    /**
     * ✅ CRIAR MENSAGEM - SIMPLIFICADO
     */
    private function createMessage(Conversation $conversation, array $data, ?array $attachment): Message
    {
        $message = $conversation->messages()->create([
            'content' => $data['content'] ?? ($attachment ? "📎 {$attachment['original_name']}" : ''),
            'sender_id' => Auth::id(),
            'sender_type' => Message::SENDER_TYPE_USER,
            'message_type' => $attachment['type'] ?? 'text',
            'is_from_contact' => false,
            'status' => 'sent',
            'delivery_status' => 'sent',
            'metadata' => [
                'source' => 'api',
                'sender_name' => Auth::user()->name
            ]
        ]);

        if ($attachment) {
            $message->attachments()->create($attachment['data']);
        }

        return $message;
    }

    /**
     * Criar mensagem de sistema
     */
    public function createSystemMessage(int $conversationId, string $content, string $systemType = 'system', array $metadata = []): array
    {
        $conversation = Conversation::findOrFail($conversationId);

        $message = Message::create([
            'conversation_id' => $conversationId,
            'sender_type' => 'system',
            'sender_id' => null,
            'content' => $content,
            'message_type' => 'system',
            'is_from_contact' => false,
            'status' => 'sent',
            'sent_at' => now(),
            'metadata' => array_merge([
                'is_system_message' => true,
                'system_type' => $systemType,
            ], $metadata)
        ]);

        $conversation->update(['last_message_at' => now()]);

        // ✅ NOTIFICAR VIA REDIS
        $this->redis->notifyNewMessage($message, $conversation);

        return [
            'success' => true,
            'message' => $this->formatMessage($message),
            'conversation_id' => $conversationId
        ];
    }

    /**
     * ✅ ENVIAR PARA WHATSAPP - SIMPLIFICADO
     */
    private function sendToWhatsApp(Conversation $conversation, Message $message, ?array $attachment): void
    {
        if (!$conversation->isWhatsApp()) return;

        try {
            SendWhatsAppMessageJob::dispatch(
                $message,
                $conversation,
                $attachment['full_path'] ?? null
            );

            $message->update([
                'delivery_status' => 'pending',
                'status' => 'sending'
            ]);

            Log::info('📋 Job de WhatsApp despachado com sucesso', [
                'message_id' => $message->id,
                'conversation_id' => $conversation->id,
                'job_class' => SendWhatsAppMessageJob::class
            ]);
        } catch (Exception $e) {
            Log::error('❌ Erro ao despachar job de WhatsApp', [
                'message_id' => $message->id,
                'error' => $e->getMessage()
            ]);

            $message->update([
                'delivery_status' => 'failed',
                'status' => 'failed'
            ]);

            throw $e;
        }
    }

    /**
     * ✅ MARCAR COMO LIDA
     */
    public function markAsRead(int $conversationId): array
    {
        $conversation = Conversation::findOrFail($conversationId);
        $this->validateAccess($conversation);

        $count = $conversation->messages()
            ->where('sender_id', '!=', Auth::id())
            ->whereNull('read_at')
            ->update(['read_at' => now()]);

        $conversation->update(['unread_count' => 0]);
        $this->redis->notifyConversationRead($conversationId, Auth::id());

        return ['success' => true, 'messages_marked' => $count];
    }

    /**
     * ✅ INDICADOR DE DIGITAÇÃO
     */
    public function handleTyping(int $conversationId, bool $isTyping): array
    {
        $conversation = Conversation::findOrFail($conversationId);
        $this->validateAccess($conversation);

        $this->redis->notifyUserTyping(Auth::id(), Auth::user()->name, $conversationId, $isTyping);

        return ['success' => true, 'is_typing' => $isTyping];
    }

    /**
     * ✅ VALIDAR ACESSO - SIMPLIFICADO
     */
    private function validateAccess(Conversation $conversation): void
    {
        if ($conversation->isInternal()) {
            if (!$conversation->participants()->where('user_id', Auth::id())->where('is_active', true)->exists()) {
                throw new Exception('Acesso negado à conversa');
            }
        }
    }

    /**
     * ✅ FORMATAR MENSAGEM - CORRIGIDO PARA EVITAR ERRO DO ARRAY_MERGE
     */
    private function formatMessage(Message $message): array
    {
        $senderData = $this->getSenderData($message);

        // ✅ GARANTIR QUE METADATA É SEMPRE UM ARRAY
        $metadata = $message->metadata;
        if (is_string($metadata)) {
            try {
                $metadata = json_decode($metadata, true) ?? [];
            } catch (\Exception $e) {
                $metadata = [];
            }
        } elseif (!is_array($metadata)) {
            $metadata = [];
        }

        return [
            'id' => $message->id,
            'conversation_id' => $message->conversation_id,
            'content' => $message->content,
            'message_type' => $message->message_type,
            'is_from_contact' => $message->is_from_contact,
            'sender_id' => $message->sender_id,
            'sender_name' => $senderData['name'],
            'sender_avatar' => $senderData['avatar'], // ✅ INCLUIR AVATAR
            'sent_at' => $message->sent_at?->toISOString(),
            'read_at' => $message->read_at?->toISOString(),
            'status' => $message->delivery_status ?? $message->status,
            'whatsapp_message_id' => $message->whatsapp_message_id,
            'attachments' => $message->attachments->map(fn($a) => [
                'id' => $a->id,
                'file_name' => $a->file_name,
                'file_url' => $a->file_url,
                'file_type' => $a->file_type,
                'file_size' => $a->file_size,
                'download_status' => $a->download_status
            ])->toArray(),
            'metadata' => array_merge($metadata, [ // ✅ AGORA É SEGURO USAR ARRAY_MERGE
                'sender_avatar' => $senderData['avatar']
            ])
        ];
    }

    /**
     * ✅ OBTER DADOS DO REMETENTE - CORRIGIDO
     */
    private function getSenderData(Message $message): array
    {
        // Se for usuário do sistema
        if (!$message->is_from_contact && $message->sender_id) {
            try {
                $user = User::find($message->sender_id);
                if ($user) {
                    return [
                        'name' => $user->name,
                        'avatar' => $user->avatar_url // ✅ USA O ACCESSOR QUE INCLUI GRAVATAR
                    ];
                }
            } catch (\Exception $e) {
                Log::warning('Erro ao buscar dados do usuário', [
                    'user_id' => $message->sender_id,
                    'error' => $e->getMessage()
                ]);
            }
        }

        // Se for contato
        if ($message->is_from_contact) {
            $conversation = $message->conversation;

            // Garantir que metadata é array
            $metadata = $message->metadata;
            if (is_string($metadata)) {
                try {
                    $metadata = json_decode($metadata, true) ?? [];
                } catch (\Exception $e) {
                    $metadata = [];
                }
            } elseif (!is_array($metadata)) {
                $metadata = [];
            }

            return [
                'name' => $metadata['sender_name'] ?? $conversation->contact_name ?? 'Contato',
                'avatar' => $conversation->processed_avatar ?? $conversation->avatar_url
            ];
        }

        // Fallback
        $metadata = $message->metadata;
        if (is_string($metadata)) {
            try {
                $metadata = json_decode($metadata, true) ?? [];
            } catch (\Exception $e) {
                $metadata = [];
            }
        } elseif (!is_array($metadata)) {
            $metadata = [];
        }

        return [
            'name' => $metadata['sender_name'] ?? 'Usuário',
            'avatar' => null
        ];
    }
}