<?php
// modules/Chat/Services/WhatsAppContactService.php

namespace Modules\Chat\Services;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Modules\Contacts\Models\Contact;
use Modules\Contacts\Models\Phone;
use Modules\Contacts\Enums\PhoneType;
use Exception;

class WhatsAppContactService
{
    /**
     * ✅ BUSCAR OU CRIAR CONTATO POR NÚMERO WHATSAPP
     */
    public function findOrCreateContact(string $whatsappNumber, ?string $pushName = null): ?Contact
    {
        try {
            $normalizedNumber = $this->normalizeNumber($whatsappNumber);
            
            if (!$normalizedNumber) {
                Log::warning('❌ Número WhatsApp inválido', [
                    'original' => $this->maskNumber($whatsappNumber)
                ]);
                return null;
            }

            Log::debug('🔍 Buscando contato', [
                'normalized_number' => $this->maskNumber($normalizedNumber),
                'push_name' => $pushName
            ]);

            // ✅ BUSCAR CONTATO EXISTENTE
            $contact = $this->findExistingContact($normalizedNumber);
            
            if ($contact) {
                Log::info('✅ Contato existente encontrado', [
                    'contact_id' => $contact->id,
                    'name' => $contact->display_name ?? $contact->full_name,
                    'number' => $this->maskNumber($normalizedNumber)
                ]);
                
                $this->updateWhatsAppFields($contact, $normalizedNumber);
                return $contact;
            }

            // ✅ CRIAR NOVO CONTATO COM RELACIONAMENTO
            return $this->createNewContactWithRelationship($normalizedNumber, $pushName);

        } catch (Exception $e) {
            Log::error('❌ Erro ao buscar/criar contato', [
                'number' => $this->maskNumber($whatsappNumber),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return null;
        }
    }

    /**
     * ✅ CRIAR NOVO CONTATO USANDO RELACIONAMENTO (SOLUÇÃO DEFINITIVA)
     */
    private function createNewContactWithRelationship(string $normalizedNumber, ?string $pushName): Contact
    {
        return DB::transaction(function () use ($normalizedNumber, $pushName) {
            try {
                Log::info('➕ Criando novo contato WhatsApp (relacionamento)', [
                    'number' => $this->maskNumber($normalizedNumber),
                    'push_name' => $pushName
                ]);

                // ✅ PREPARAR NOME
                if ($pushName && trim($pushName)) {
                    $nameParts = explode(' ', trim($pushName), 2);
                    $firstName = trim($nameParts[0]);
                    $lastName = isset($nameParts[1]) ? trim($nameParts[1]) : '';
                } else {
                    $firstName = 'Contato WhatsApp';
                    $lastName = '';
                }

                // ✅ CRIAR CONTATO PRIMEIRO
                $contact = Contact::create([
                    'first_name' => $firstName,
                    'last_name' => $lastName,
                    'whatsapp_number' => $normalizedNumber,
                    'whatsapp_opt_in' => true,
                    'last_whatsapp_interaction' => now(),
                    'created_by' => $this->getCurrentUserId(),
                    'user_id' => $this->getDefaultOwnerId(),
                    'job_title' => null,
                    'email' => null,
                    'source_id' => $this->getWhatsAppSourceId(),
                ]);

                Log::debug('✅ Contato criado', [
                    'contact_id' => $contact->id,
                    'name' => $contact->full_name
                ]);

                // ✅ CRIAR TELEFONE USANDO RELACIONAMENTO (SOLUÇÃO DEFINITIVA)
                $phone = $contact->phones()->create([
                    'type' => PhoneType::mobile,  // ✅ Usar o enum diretamente
                    'number' => $normalizedNumber,
                ]);

                Log::debug('✅ Telefone criado via relacionamento', [
                    'phone_id' => $phone->id,
                    'contact_id' => $contact->id,
                    'number' => $this->maskNumber($normalizedNumber),
                    'type' => $phone->type->name ?? 'mobile',
                    'phoneable_type' => $phone->phoneable_type,
                    'phoneable_id' => $phone->phoneable_id
                ]);

                Log::info('✅ Novo contato WhatsApp criado com sucesso', [
                    'contact_id' => $contact->id,
                    'name' => $contact->full_name,
                    'number' => $this->maskNumber($normalizedNumber),
                    'phone_id' => $phone->id,
                    'source' => 'whatsapp_webhook'
                ]);

                return $contact;

            } catch (Exception $e) {
                Log::error('❌ Erro na transação de criação de contato', [
                    'number' => $this->maskNumber($normalizedNumber),
                    'name' => $pushName,
                    'error' => $e->getMessage(),
                    'sql_state' => $e->getCode(),
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                ]);
                throw $e;
            }
        });
    }

    /**
     * ✅ BUSCAR CONTATO EXISTENTE
     */
    private function findExistingContact(string $normalizedNumber): ?Contact
    {
        // 1. Buscar por whatsapp_number
        $contact = Contact::where('whatsapp_number', $normalizedNumber)->first();
        if ($contact) {
            Log::debug('✅ Contato encontrado por whatsapp_number', ['contact_id' => $contact->id]);
            return $contact;
        }

        // 2. Buscar por telefones relacionados
        $variations = $this->getNumberVariations($normalizedNumber);
        
        Log::debug('🔍 Buscando por variações de número', [
            'variations_count' => count($variations),
            'variations_preview' => array_map([$this, 'maskNumber'], array_slice($variations, 0, 3))
        ]);
        
        $phone = Phone::whereIn('number', $variations)
            ->where('phoneable_type', Contact::class)
            ->with('phoneable')
            ->first();
            
        if ($phone && $phone->phoneable instanceof Contact) {
            Log::debug('✅ Contato encontrado por telefone relacionado', [
                'contact_id' => $phone->phoneable->id,
                'phone_number' => $this->maskNumber($phone->number),
                'phone_type' => $phone->type->name ?? 'unknown'
            ]);
            return $phone->phoneable;
        }

        // 3. Buscar por número limpo (fallback)
        $cleanNumber = preg_replace('/[^0-9]/', '', $normalizedNumber);
        
        $contact = Contact::where('whatsapp_number', 'LIKE', "%{$cleanNumber}%")
            ->orWhere(function($query) use ($cleanNumber) {
                $query->whereHas('phones', function($phoneQuery) use ($cleanNumber) {
                    $phoneQuery->where('number', 'LIKE', "%{$cleanNumber}%");
                });
            })
            ->first();
            
        if ($contact) {
            Log::debug('✅ Contato encontrado por busca LIKE', ['contact_id' => $contact->id]);
            return $contact;
        }

        Log::debug('❌ Nenhum contato encontrado para o número');
        return null;
    }

    /**
     * ✅ ATUALIZAR CAMPOS WHATSAPP
     */
    private function updateWhatsAppFields(Contact $contact, string $normalizedNumber): void
    {
        try {
            $updates = [];
            
            if (empty($contact->whatsapp_number)) {
                $updates['whatsapp_number'] = $normalizedNumber;
            }
            
            if (!$contact->last_whatsapp_interaction || 
                $contact->last_whatsapp_interaction->lt(now()->subHour())) {
                $updates['last_whatsapp_interaction'] = now();
            }
            
            if (!$contact->whatsapp_opt_in) {
                $updates['whatsapp_opt_in'] = true;
            }
            
            if (!empty($updates)) {
                $contact->update($updates);
                Log::debug('✅ Campos WhatsApp atualizados', [
                    'contact_id' => $contact->id,
                    'updates' => array_keys($updates)
                ]);
            }
            
        } catch (Exception $e) {
            Log::warning('⚠️ Erro ao atualizar campos WhatsApp', [
                'contact_id' => $contact->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * ✅ MÉTODOS AUXILIARES
     */
    private function getCurrentUserId(): int
    {
        try {
            if (auth()->check()) {
                return auth()->id();
            }
            return 1;
        } catch (Exception $e) {
            Log::warning('⚠️ Erro ao obter usuário atual, usando padrão', ['error' => $e->getMessage()]);
            return 1;
        }
    }

    private function getDefaultOwnerId(): ?int
    {
        try {
            return $this->getCurrentUserId();
        } catch (Exception $e) {
            Log::warning('⚠️ Erro ao obter proprietário padrão', ['error' => $e->getMessage()]);
            return null;
        }
    }

    private function getWhatsAppSourceId(): ?int
    {
        try {
            $source = \Modules\Contacts\Models\Source::where('name', 'LIKE', '%WhatsApp%')
                ->orWhere('name', 'LIKE', '%whatsapp%')
                ->first();
                
            return $source?->id;
        } catch (Exception $e) {
            Log::warning('⚠️ Source WhatsApp não encontrado', ['error' => $e->getMessage()]);
            return null;
        }
    }

    private function normalizeNumber(string $number): ?string
    {
        $clean = preg_replace('/[^0-9]/', '', $number);

        Log::debug('🔧 Normalizando número', [
            'original' => $this->maskNumber($number),
            'clean' => $this->maskNumber($clean),
            'length' => strlen($clean)
        ]);

        if (strlen($clean) < 10) {
            return null;
        }

        if (strlen($clean) >= 13 && str_starts_with($clean, '55')) {
            $clean = substr($clean, 2);
        }

        if (strlen($clean) == 10 || strlen($clean) == 11) {
            $clean = '55' . $clean;
        }

        return $clean;
    }

    private function getNumberVariations(string $number): array
    {
        $clean = preg_replace('/[^0-9]/', '', $number);
        $variations = [$clean];

        if (str_starts_with($clean, '55') && strlen($clean) >= 12) {
            $withoutCountry = substr($clean, 2);
            $variations[] = $withoutCountry;
        } elseif (!str_starts_with($clean, '55') && strlen($clean) >= 10) {
            $withCountry = '55' . $clean;
            $variations[] = $withCountry;
        }

        if (strlen($clean) == 12 && str_starts_with($clean, '55')) {
            $area = substr($clean, 2, 2);
            $num = substr($clean, 4);
            
            if (in_array($num[0], ['6', '7', '8', '9']) && strlen($num) == 8) {
                $variations[] = '55' . $area . '9' . $num;
            }
        }

        if (strlen($clean) == 13 && str_starts_with($clean, '55')) {
            $area = substr($clean, 2, 2);
            $num = substr($clean, 5);
            
            if ($clean[4] == '9' && strlen($num) == 8) {
                $variations[] = '55' . $area . $num;
            }
        }

        return array_unique($variations);
    }

    private function maskNumber(string $number): string
    {
        if (strlen($number) <= 4) {
            return '****';
        }
        return substr($number, 0, 4) . '****' . substr($number, -2);
    }

    // ✅ MÉTODOS PÚBLICOS
    public function isValidWhatsAppNumber(string $number): bool
    {
        $normalized = $this->normalizeNumber($number);
        return $normalized !== null && strlen($normalized) >= 12;
    }

    public function getNormalizedNumber(string $number): ?string
    {
        return $this->normalizeNumber($number);
    }

    public function findContactByNumber(string $number): ?Contact
    {
        $normalized = $this->normalizeNumber($number);
        if (!$normalized) return null;
        
        return $this->findExistingContact($normalized);
    }
}