<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::table('messages', function (Blueprint $table) {
            // ✅ IDENTIFICADORES WHATSAPP/EVOLUTION
            if (!Schema::hasColumn('messages', 'whatsapp_message_id')) {
                $table->string('whatsapp_message_id')->nullable()->after('external_id')
                    ->comment('ID único da mensagem no WhatsApp');
            }
            
            if (!Schema::hasColumn('messages', 'whatsapp_key_id')) {
                $table->string('whatsapp_key_id')->nullable()->after('whatsapp_message_id')
                    ->comment('key.id retornado pela Evolution API');
            }
            
            if (!Schema::hasColumn('messages', 'whatsapp_key_remote_jid')) {
                $table->string('whatsapp_key_remote_jid')->nullable()->after('whatsapp_key_id')
                    ->comment('key.remoteJid da mensagem');
            }
            
            if (!Schema::hasColumn('messages', 'whatsapp_key_from_me')) {
                $table->boolean('whatsapp_key_from_me')->default(false)->after('whatsapp_key_remote_jid')
                    ->comment('key.fromMe - indica se a mensagem foi enviada por nós');
            }
            
            // ✅ TIPOS E STATUS WHATSAPP
            if (!Schema::hasColumn('messages', 'whatsapp_message_type')) {
                $table->enum('whatsapp_message_type', [
                    'text', 'image', 'audio', 'video', 'document', 'sticker', 
                    'location', 'contact', 'poll', 'reaction', 'ephemeral'
                ])->nullable()->after('whatsapp_key_from_me')
                    ->comment('Tipo da mensagem no WhatsApp');
            }
            
            if (!Schema::hasColumn('messages', 'delivery_status')) {
                $table->enum('delivery_status', [
                    'sending', 'sent', 'delivered', 'read', 'failed', 'pending'
                ])->default('pending')->after('whatsapp_message_type')
                    ->comment('Status de entrega da mensagem');
            }
            
            // ✅ TIMESTAMPS WHATSAPP
            if (!Schema::hasColumn('messages', 'whatsapp_timestamp')) {
                $table->timestamp('whatsapp_timestamp')->nullable()->after('delivery_status')
                    ->comment('Timestamp original da mensagem no WhatsApp');
            }
            
            // ✅ DADOS ESPECÍFICOS WHATSAPP
            if (!Schema::hasColumn('messages', 'whatsapp_participant')) {
                $table->string('whatsapp_participant')->nullable()->after('read_at')
                    ->comment('Participante em grupos (para mensagens de grupo)');
            }
            
            if (!Schema::hasColumn('messages', 'whatsapp_quoted_message')) {
                $table->json('whatsapp_quoted_message')->nullable()->after('whatsapp_participant')
                    ->comment('Dados da mensagem citada/respondida');
            }
            
            if (!Schema::hasColumn('messages', 'whatsapp_reactions')) {
                $table->json('whatsapp_reactions')->nullable()->after('whatsapp_quoted_message')
                    ->comment('Reações à mensagem');
            }
            
            if (!Schema::hasColumn('messages', 'is_forwarded')) {
                $table->boolean('is_forwarded')->default(false)->after('whatsapp_reactions')
                    ->comment('Indica se a mensagem foi encaminhada');
            }
            
            if (!Schema::hasColumn('messages', 'forward_score')) {
                $table->integer('forward_score')->nullable()->after('is_forwarded')
                    ->comment('Número de vezes que foi encaminhada');
            }
            
            if (!Schema::hasColumn('messages', 'is_ephemeral')) {
                $table->boolean('is_ephemeral')->default(false)->after('forward_score')
                    ->comment('Indica se é mensagem temporária');
            }
            
            if (!Schema::hasColumn('messages', 'ephemeral_expires_at')) {
                $table->timestamp('ephemeral_expires_at')->nullable()->after('is_ephemeral')
                    ->comment('Quando a mensagem temporária expira');
            }

            if (!Schema::hasColumn('messages', 'reply_to_id')) {
                $table->foreignId('reply_to_id')->nullable()->after('ephemeral_expires_at')
                ->constrained('messages')->onDelete('set null')
                ->comment('ID da mensagem sendo respondida');
            }
            
            // ✅ ÍNDICES PARA PERFORMANCE
            $table->index(['whatsapp_message_id'], 'idx_whatsapp_message_id');
            $table->index(['whatsapp_key_id'], 'idx_whatsapp_key_id');
            $table->index(['delivery_status', 'whatsapp_timestamp'], 'idx_delivery_status_timestamp');
            $table->index(['conversation_id', 'whatsapp_timestamp'], 'idx_conversation_whatsapp_time');
            $table->index(['whatsapp_key_from_me', 'delivery_status'], 'idx_from_me_delivery');
            $table->index(['reply_to_id'], 'idx_reply_to');

        });
    }

    public function down(): void
    {
        Schema::table('messages', function (Blueprint $table) {
            // ✅ REMOVER CHAVES E ÍNDICES PRIMEIRO
            if (Schema::hasColumn('messages', 'reply_to_id')) {
                $table->dropForeign(['reply_to_id']);
            }
            if (Schema::hasIndex('messages', 'idx_reply_to')) {
                $table->dropIndex('idx_reply_to');
            }
            if (Schema::hasIndex('messages', 'idx_whatsapp_message_id')) {
                $table->dropIndex('idx_whatsapp_message_id');
            }
            if (Schema::hasIndex('messages', 'idx_whatsapp_key_id')) {
                $table->dropIndex('idx_whatsapp_key_id');
            }
            if (Schema::hasIndex('messages', 'idx_delivery_status_timestamp')) {
                $table->dropIndex('idx_delivery_status_timestamp');
            }
            if (Schema::hasIndex('messages', 'idx_conversation_whatsapp_time')) {
                $table->dropIndex('idx_conversation_whatsapp_time');
            }
            if (Schema::hasIndex('messages', 'idx_from_me_delivery')) {
                $table->dropIndex('idx_from_me_delivery');
            }
            
            // ✅ REMOVER COLUNAS
            $table->dropColumn([
                'whatsapp_message_id',
                'whatsapp_key_id',
                'whatsapp_key_remote_jid',
                'whatsapp_key_from_me',
                'whatsapp_message_type',
                'delivery_status',
                'whatsapp_timestamp',
                'delivered_at',
                'read_at',
                'whatsapp_participant',
                'whatsapp_quoted_message',
                'whatsapp_reactions',
                'is_forwarded',
                'forward_score',
                'is_ephemeral',
                'ephemeral_expires_at',
                'reply_to_id'
            ]);
        });
    }
};