<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('evolution_instances', function (Blueprint $table) {
            $table->id();
            
            // ✅ IDENTIFICAÇÃO DA INSTÂNCIA
            $table->string('instance_name')->unique()
                ->comment('Nome único da instância (ex: empresa_principal)');
            
            $table->string('display_name')
                ->comment('Nome amigável para exibição');
            
            $table->text('description')->nullable()
                ->comment('Descrição da instância');
            
            // ✅ CONFIGURAÇÃO DA API
            $table->string('api_url')
                ->comment('URL base da API Evolution');
            
            $table->text('api_key')
                ->comment('Chave de API (criptografada)');
            
            $table->string('api_version')->default('v1')
                ->comment('Versão da API Evolution');
            
            // ✅ STATUS E CONEXÃO
            $table->enum('status', [
                'active', 'inactive', 'connecting', 'disconnected', 
                'error', 'qr_pending', 'authenticating', 'maintenance'
            ])->default('inactive')
                ->comment('Status atual da instância');
            
            $table->enum('connection_state', [
                'close', 'connecting', 'open', 'qrcode', 'error', 'disconnected', 'unknown'
            ])->nullable()
                ->comment('Estado da conexão WebSocket');
            
            $table->timestamp('last_ping')->nullable()
                ->comment('Último ping bem-sucedido');
            
            $table->timestamp('connected_at')->nullable()
                ->comment('Quando a instância foi conectada');
            
            $table->timestamp('disconnected_at')->nullable()
                ->comment('Quando a instância foi desconectada');
            
            // ✅ QR CODE E AUTENTICAÇÃO
            $table->json('qr_code')->nullable()
                ->comment('Dados do QR Code atual');
            
            $table->timestamp('qr_code_generated_at')->nullable()
                ->comment('Quando o QR Code foi gerado');
            
            $table->timestamp('qr_code_expires_at')->nullable()
                ->comment('Quando o QR Code expira');
            
            // ✅ DADOS DA INSTÂNCIA WHATSAPP
            $table->json('instance_data')->nullable()
                ->comment('Dados do perfil WhatsApp conectado');
            
            $table->string('whatsapp_number')->nullable()
                ->comment('Número do WhatsApp conectado');
            
            $table->string('whatsapp_name')->nullable()
                ->comment('Nome do perfil WhatsApp');
            
            $table->string('whatsapp_profile_picture')->nullable()
                ->comment('URL da foto do perfil');
            
            $table->boolean('is_business')->default(false)
                ->comment('Indica se é WhatsApp Business');
            
            // ✅ CONFIGURAÇÕES E WEBHOOKS
            $table->json('settings')->nullable()
                ->comment('Configurações específicas da instância');
            
            $table->string('webhook_url')->nullable()
                ->comment('URL do webhook para esta instância');
            
            $table->json('webhook_events')->nullable()
                ->comment('Eventos que o webhook deve receber');
            
            $table->boolean('webhook_active')->default(true)
                ->comment('Se o webhook está ativo');
            
            // ✅ ESTATÍSTICAS E MONITORAMENTO
            $table->integer('messages_sent_today')->default(0)
                ->comment('Mensagens enviadas hoje');
            
            $table->integer('messages_received_today')->default(0)
                ->comment('Mensagens recebidas hoje');
            
            $table->timestamp('last_message_at')->nullable()
                ->comment('Última mensagem processada');
            
            $table->json('error_log')->nullable()
                ->comment('Log dos últimos erros');
            
            $table->integer('error_count')->default(0)
                ->comment('Contador de erros consecutivos');
            
            $table->timestamp('last_error_at')->nullable()
                ->comment('Último erro registrado');
            
            // ✅ CONTROLE DE ACESSO
            $table->foreignId('created_by')->nullable()->constrained('users')->onDelete('set null')
                ->comment('Usuário que criou a instância');
            
            $table->foreignId('managed_by')->nullable()->constrained('users')->onDelete('set null')
                ->comment('Usuário responsável pela instância');
            
            $table->json('allowed_users')->nullable()
                ->comment('IDs dos usuários autorizados a usar esta instância');
            
            $table->boolean('is_default')->default(false)
                ->comment('Se é a instância padrão');
            
            $table->integer('priority')->default(1)
                ->comment('Prioridade da instância (1 = mais alta)');
            
            $table->timestamps();
            $table->softDeletes();
            
            // ✅ ÍNDICES PARA PERFORMANCE
            $table->index(['status', 'is_default'], 'idx_status_default');
            $table->index(['whatsapp_number'], 'idx_whatsapp_number');
            $table->index(['last_ping', 'status'], 'idx_ping_status');
            $table->index(['created_by', 'managed_by'], 'idx_users');
            $table->index(['priority', 'status'], 'idx_priority_status');
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('evolution_instances');
    }
};