<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::table('message_attachments', function (Blueprint $table) {
            // ✅ DADOS WHATSAPP/EVOLUTION
            $table->string('whatsapp_media_key')->nullable()->after('external_url')
                ->comment('Chave da mídia no WhatsApp');
            
            $table->string('whatsapp_direct_path')->nullable()->after('whatsapp_media_key')
                ->comment('Caminho direto da mídia no WhatsApp');
            
            $table->string('whatsapp_media_url')->nullable()->after('whatsapp_direct_path')
                ->comment('URL original da mídia no WhatsApp');
            
            $table->json('whatsapp_metadata')->nullable()->after('whatsapp_media_url')
                ->comment('Metadados da mídia do WhatsApp');
            
            // ✅ PROCESSAMENTO E DOWNLOAD
            $table->enum('download_status', [
                'pending', 'downloading', 'completed', 'failed', 'not_needed'
            ])->default('pending')->after('whatsapp_metadata')
                ->comment('Status do download da mídia');
            
            $table->timestamp('downloaded_at')->nullable()->after('download_status')
                ->comment('Quando a mídia foi baixada');
            
            $table->integer('download_attempts')->default(0)->after('downloaded_at')
                ->comment('Tentativas de download');
            
            $table->text('download_error')->nullable()->after('download_attempts')
                ->comment('Erro no download se houver');
            
            // ✅ INFORMAÇÕES ADICIONAIS
            $table->integer('width')->nullable()->after('download_error')
                ->comment('Largura da imagem/vídeo');
            
            $table->integer('height')->nullable()->after('width')
                ->comment('Altura da imagem/vídeo');
            
            $table->integer('duration')->nullable()->after('height')
                ->comment('Duração do áudio/vídeo em segundos');
            
            $table->string('thumbnail_path')->nullable()->after('duration')
                ->comment('Caminho da thumbnail gerada');
            
            $table->boolean('is_animated')->default(false)->after('thumbnail_path')
                ->comment('Se é GIF ou vídeo animado');
            
            // ✅ ÍNDICES
            $table->index(['whatsapp_media_key'], 'idx_whatsapp_media_key');
            $table->index(['download_status', 'downloaded_at'], 'idx_download_status_time');
        });
    }

    public function down(): void
    {
        Schema::table('message_attachments', function (Blueprint $table) {
            // ✅ REMOVER ÍNDICES
            $table->dropIndex('idx_whatsapp_media_key');
            $table->dropIndex('idx_download_status_time');
            
            // ✅ REMOVER COLUNAS
            $table->dropColumn([
                'whatsapp_media_key',
                'whatsapp_direct_path',
                'whatsapp_media_url',
                'whatsapp_metadata',
                'download_status',
                'downloaded_at',
                'download_attempts',
                'download_error',
                'width',
                'height',
                'duration',
                'thumbnail_path',
                'is_animated'
            ]);
        });
    }
};