// modules/Chat/resources/js/composables/useChatConversations.js
import { ref, computed, shallowRef, markRaw, nextTick } from 'vue'
import servicoApi from '../servicos/api.js'

export function useChatConversations(currentUser, router, route) {
  const carregandoConversas = ref(false)
  const showAllConversations = ref(false)
  const avatarCache = new Map()
  const requestCache = new Map()

  // ✅ Validação de serviços com fallback
  const getChatStore = () => {
    const store = window.chatStore
    if (!store) {
      console.warn('ChatStore not available')
      return null
    }
    return store
  }

  const getWebSocketService = () => {
    const service = window.websocketService
    if (!service) {
      console.warn('WebSocketService not available')
      return null
    }
    return service
  }

  // ✅ Refs reativas com validação
  const conversas = computed(() => {
    const chatStore = getChatStore()
    return chatStore?.conversas?.value || []
  })

  const conversaSelecionada = computed(() => {
    const chatStore = getChatStore()
    return chatStore?.conversaSelecionada?.value || null
  })

  const isAdmin = computed(() => {
    const ws = getWebSocketService()
    return ws?.getCurrentUser()?.super_admin || false
  })

  // ✅ Validação de currentUser
  const getCurrentUserId = () => {
    return currentUser?.value?.id || getWebSocketService()?.getCurrentUser()?.id || null
  }

  // ✅ Processamento de avatar com validação
  function processarAvatarConversa(conversa) {
    if (!conversa || !conversa.id) {
      console.warn('Invalid conversation for avatar processing')
      return {
        avatar_url: null,
        display_name: 'Conversa Inválida',
        participant: null
      }
    }

    const cacheKey = `${conversa.id}-${conversa.type}-${conversa.updated_at}`
    if (avatarCache.has(cacheKey)) {
      return avatarCache.get(cacheKey)
    }

    let result
    
    if (conversa.type === 'internal') {
      const currentUserId = getCurrentUserId()
      const otherParticipant = conversa.participants?.find(p => 
        parseInt(p.id) !== parseInt(currentUserId)
      )
      
      if (otherParticipant) {
        result = {
          avatar_url: otherParticipant.avatar || getGravatarUrl(otherParticipant.email || otherParticipant.name),
          display_name: otherParticipant.name || `Usuário #${otherParticipant.id}`,
          participant: otherParticipant
        }
      } else {
        result = {
          avatar_url: null,
          display_name: conversa.subject || 'Chat Interno',
          participant: null
        }
      }
    } else {
      result = {
        avatar_url: conversa.avatar_url || conversa.whatsapp_profile_picture || getGravatarUrl(conversa.phone || conversa.contact_name),
        display_name: conversa.contact_name || conversa.phone || `Conversa #${conversa.id}`,
        participant: null
      }
    }
    
    avatarCache.set(cacheKey, result)
    return result
  }

  function getGravatarUrl(identifier, size = 40) {
    if (!identifier) return null
    try {
      const cleanIdentifier = identifier.toString().toLowerCase().trim()
      const hash = simpleHash(cleanIdentifier)
      return `https://www.gravatar.com/avatar/${hash}?s=${size}&d=identicon`
    } catch (error) {
      console.warn('Error generating Gravatar URL:', error)
      return null
    }
  }

  function simpleHash(string) {
    let hash = 0
    if (!string || string.length === 0) return '00000000'
    for (let i = 0; i < string.length; i++) {
      const char = string.charCodeAt(i)
      hash = ((hash << 5) - hash) + char
      hash = hash & hash
    }
    return Math.abs(hash).toString(16).padStart(8, '0')
  }

  // ✅ Carregar conversas com validação completa
  async function carregarConversasIniciais(filterId = 'todas') {
    if (carregandoConversas.value) {
      console.log('Already loading conversations, skipping...')
      return
    }
    
    try {
      carregandoConversas.value = true
      const chatStore = getChatStore()
      
      if (chatStore) {
        chatStore.isLoading.value = true
      }
      
      // Validar servicoApi
      if (!servicoApi || !servicoApi.getConversations) {
        throw new Error('API service not available')
      }
      
      const response = await servicoApi.getConversations({ 
        include_archived: filterId === 'arquivadas',
        show_all: showAllConversations.value
      })

      const todasConversas = normalizeResponse(response)
      
      if (todasConversas.length > 0) {
        const processedConversations = await processConversationsInBatches(todasConversas)
        
        if (chatStore) {
          chatStore.conversas.value = markRaw(processedConversations)
        }
        
        await carregarConversaDaUrl()
      } else if (chatStore) {
        chatStore.conversas.value = markRaw([])
      }
      
    } catch (error) {
      console.error('Erro ao carregar conversas:', error)
      
      const chatStore = getChatStore()
      if (chatStore) {
        chatStore.conversas.value = markRaw([])
      }
      
      // Usar sistema de notificação do Concord CRM com fallback
      if (window.Innoclapps?.error) {
        window.Innoclapps.error('Falha ao carregar conversas: ' + (error.message || 'Erro desconhecido'))
      } else {
        console.error('Notification system not available')
      }
    } finally {
      carregandoConversas.value = false
      const chatStore = getChatStore()
      if (chatStore) {
        chatStore.isLoading.value = false
      }
    }
  }

  // ✅ Normalização com validação
  function normalizeResponse(response) {
    if (!response) {
      console.warn('Empty response received')
      return []
    }
    if (Array.isArray(response)) return response
    if (response.data) {
      if (Array.isArray(response.data)) return response.data
      if (response.data.data && Array.isArray(response.data.data)) return response.data.data
    }
    console.warn('Unexpected response format:', response)
    return []
  }

  // ✅ Processamento em lotes com validação
  async function processConversationsInBatches(conversas, batchSize = 20) {
    if (!Array.isArray(conversas)) {
      console.warn('Invalid conversations array for batch processing')
      return []
    }

    const results = []
    
    for (let i = 0; i < conversas.length; i += batchSize) {
      const batch = conversas.slice(i, i + batchSize)
      const processed = batch.map(conversa => {
        try {
          const avatarInfo = processarAvatarConversa(conversa)
          
          return {
            ...conversa,
            type: conversa.type || (conversa.participants ? 'internal' : 'external'),
            unread_count: conversa.unread_count || 0,
            contact_name: conversa.contact_name || conversa.subject || `Conversa #${conversa.id}`,
            last_message: conversa.last_message || '',
            last_message_at: conversa.last_message_at || conversa.updated_at || conversa.created_at,
            processed_avatar: avatarInfo.avatar_url,
            processed_display_name: avatarInfo.display_name,
            processed_participant: avatarInfo.participant
          }
        } catch (error) {
          console.warn('Error processing conversation:', conversa.id, error)
          return {
            ...conversa,
            type: 'external',
            unread_count: 0,
            contact_name: `Conversa #${conversa.id}`,
            last_message: '',
            processed_avatar: null,
            processed_display_name: `Conversa #${conversa.id}`,
            processed_participant: null
          }
        }
      })
      
      results.push(...processed)
      
      // Yield control para não bloquear UI
      if (i + batchSize < conversas.length) {
        await new Promise(resolve => setTimeout(resolve, 0))
      }
    }
    
    return results
  }

  // ✅ Selecionar conversa com validação
  async function selecionarConversa(conversa) {
    if (!conversa || !conversa.id) {
      console.warn('Invalid conversation for selection')
      return
    }

    try {
      const chatStore = getChatStore()
      const ws = getWebSocketService()
      
      if (conversaSelecionada.value?.id === conversa.id) {
        console.log('Conversation already selected')
        return
      }
      
      // Leave current conversation
      if (conversaSelecionada.value?.id && ws?.isValid()) {
        await ws.leaveConversation(conversaSelecionada.value.id)
      }
      
      // Set new conversation
      if (chatStore) {
        chatStore.conversaSelecionada.value = conversa
      }
      
      // Join new conversation
      if (ws?.isValid()) {
        await ws.joinConversation(conversa.id, conversa.type)
      }
      
      // Execute parallel tasks
      await Promise.allSettled([
        marcarComoLidaViaApi(conversa.id),
        atualizarUrlConversa(conversa.id)
      ])
    } catch (error) {
      console.error('Erro ao selecionar conversa:', error)
      if (window.Innoclapps?.error) {
        window.Innoclapps.error('Falha ao selecionar conversa: ' + (error.message || ''))
      }
    }
  }

  // ✅ Marcar como lida com validação
  async function marcarComoLidaViaApi(conversationId) {
    if (!conversationId) return

    try {
      const chatStore = getChatStore()
      if (chatStore) {
        chatStore.markConversationAsReadGlobal(conversationId)
      }
      
      if (!servicoApi?.markAsRead) {
        console.warn('markAsRead API not available')
        return
      }
      
      const resultado = await servicoApi.markAsRead(conversationId)
      
      if (resultado.success !== false) {
        const ws = getWebSocketService()
        if (ws?.isValid()) {
          ws.sendEvent('mark_as_read', {
            conversation_id: conversationId,
            user_id: ws.getCurrentUser()?.id,
            read_at: new Date().toISOString()
          })
        }
      }
    } catch (error) {
      console.error('Erro ao marcar como lida:', error)
    }
  }

  // ✅ Atualizar URL com validação
  async function atualizarUrlConversa(conversationId) {
    if (!conversationId || !router) return

    try {
      await nextTick()
      const currentQuery = router.currentRoute.value.query
      if (currentQuery.conversation !== conversationId.toString()) {
        await router.replace({ 
          query: { ...currentQuery, conversation: conversationId } 
        })
      }
    } catch (error) {
      console.warn('Erro ao atualizar URL:', error)
    }
  }

  // ✅ Carregar conversa da URL com validação
  async function carregarConversaDaUrl() {
    if (!router || !route) return

    try {
      const conversationId = router.currentRoute.value.query.conversation
      if (conversationId && conversas.value.length > 0) {
        const conversa = conversas.value.find(c => c.id === parseInt(conversationId))
        if (conversa) {
          await selecionarConversa(conversa)
        }
      }
    } catch (error) {
      console.warn('Erro ao carregar conversa da URL:', error)
    }
  }

  // ✅ Tratar mensagem enviada com validação
  async function tratarMensagemEnviada(mensagemData) {
    if (!mensagemData) return

    try {
      if (mensagemData && conversaSelecionada.value?.id) {
        const currentUserId = getWebSocketService()?.getCurrentUser()?.id
        if (currentUserId) {
          const chatStore = getChatStore()
          if (chatStore) {
            chatStore.updateConversationList(conversaSelecionada.value.id, mensagemData, true)
            chatStore.debouncedSort()
          }
        }
      }
    } catch (error) {
      console.error('Erro ao processar mensagem enviada:', error)
    }
  }

  // ✅ Tratar nova conversa com validação
  async function tratarNovaConversa(dadosConversa) {
    if (!dadosConversa || !servicoApi) {
      console.warn('Invalid data for new conversation')
      return
    }

    try {
      const response = dadosConversa.type === 'internal' 
        ? await servicoApi.createInternalConversation(dadosConversa)
        : await servicoApi.createConversation(dadosConversa)
      
      if (response?.conversation || response?.data?.conversation) {
        const novaConversa = response.conversation || response.data.conversation
        const avatarInfo = processarAvatarConversa({ ...novaConversa, type: dadosConversa.type })
        
        const conversaProcessada = {
          ...novaConversa,
          type: dadosConversa.type,
          unread_count: 0,
          contact_name: novaConversa.contact_name || dadosConversa.contact_name || 'Nova Conversa',
          last_message: '',
          last_message_at: novaConversa.created_at,
          processed_avatar: avatarInfo.avatar_url,
          processed_display_name: avatarInfo.display_name,
          processed_participant: avatarInfo.participant
        }
        
        const chatStore = getChatStore()
        if (chatStore) {
          chatStore.addConversation(conversaProcessada)
        }
        
        await selecionarConversa(conversaProcessada)
        
        // Notificação de sucesso
        if (window.Innoclapps?.success) {
          const message = `Conversa ${dadosConversa.type === 'internal' ? 'interna' : 'externa'} criada!`
          window.Innoclapps.success(message)
        }
      }
    } catch (error) {
      console.error('Erro ao criar conversa:', error)
      
      if (error.response?.status === 409) {
        const existingConversation = error.response?.data?.conversation
        if (existingConversation) {
          const avatarInfo = processarAvatarConversa(existingConversation)
          await selecionarConversa({ ...existingConversation, ...avatarInfo })
          
          if (window.Innoclapps?.info) {
            window.Innoclapps.info('Conversa já existe')
          }
          return
        }
      }
      
      const mensagem = error.response?.data?.message || error.message || 'Erro ao criar conversa'
      if (window.Innoclapps?.error) {
        window.Innoclapps.error(mensagem)
      }
    }
  }

  // ✅ Tratar conversa atualizada com validação
  function tratarConversaAtualizada(updatedConversation) {
    if (!updatedConversation || !updatedConversation.id) {
      console.warn('Invalid conversation update data')
      return
    }

    try {
      if (!updatedConversation.processed_avatar || !updatedConversation.processed_display_name) {
        const avatarInfo = processarAvatarConversa(updatedConversation)
        updatedConversation.processed_avatar = avatarInfo.avatar_url
        updatedConversation.processed_display_name = avatarInfo.display_name
        updatedConversation.processed_participant = avatarInfo.participant
      }

      const chatStore = getChatStore()
      if (chatStore) {
        chatStore.updateConversationMeta(updatedConversation)
      }

      if (conversaSelecionada.value && conversaSelecionada.value.id === updatedConversation.id) {
        Object.assign(conversaSelecionada.value, updatedConversation)
      }
      
      // Recarregar se conversa foi arquivada/deletada ou mudou atribuição
      const currentUserId = getCurrentUserId()
      if (updatedConversation.status === 'archived' || 
          updatedConversation.status === 'deleted' ||
          (updatedConversation.assigned_user_id !== currentUserId && 
           !isAdmin.value && !showAllConversations.value)) {
        carregarConversasIniciais()
      }
    } catch (error) {
      console.error('Erro ao tratar conversa atualizada:', error)
    }
  }

  // ✅ Toggle mostrar todas as conversas
  async function toggleShowAllConversations() {
    showAllConversations.value = !showAllConversations.value
    requestCache.clear()
    await carregarConversasIniciais()
  }

  return {
    conversas,
    conversaSelecionada,
    carregandoConversas,
    showAllConversations,
    isAdmin,
    carregarConversasIniciais,
    selecionarConversa,
    tratarConversaAtualizada,
    tratarMensagemEnviada,
    tratarNovaConversa,
    toggleShowAllConversations
  }
}