// modules/Chat/resources/js/composables/useChatWebSocket.js
import { ref, shallowRef, computed, onMounted, onUnmounted } from 'vue'

export function useChatWebSocket() {
  const websocketStatus = shallowRef({
    isConnected: false,
    isInitialized: false,
    connectionState: 'disconnected',
    currentUser: null,
    reconnectAttempts: 0,
    maxReconnectAttempts: 5,
    eventCount: 0,
    lastError: null,
    socketId: null
  })
  
  const reconectandoWebSocket = ref(false)
  let statusInterval = null

  const websocketStatusClass = computed(() => {
    const stateMap = {
      'connected': 'status-connected',
      'connecting': 'status-connecting', 
      'error': 'status-error',
      'disconnected': 'status-disconnected'
    }
    return stateMap[websocketStatus.value.connectionState] || 'status-disconnected'
  })

  const websocketStatusText = computed(() => {
    const textMap = {
      'connected': 'Online',
      'connecting': 'Conectando...',
      'error': 'Erro',
      'disconnected': 'Offline'
    }
    return textMap[websocketStatus.value.connectionState] || 'Desconhecido'
  })

  function getWebSocketService() {
    return window.websocketService
  }

  function actualizarStatusWebSocket() {
    try {
      const ws = getWebSocketService()
      if (ws) {
        const status = ws.getStatus()
        websocketStatus.value = {
          isConnected: status.isConnected || false,
          isInitialized: status.isInitialized || false,
          connectionState: status.connectionState || 'disconnected',
          currentUser: status.currentUser || null,
          reconnectAttempts: status.reconnectAttempts || 0,
          maxReconnectAttempts: status.maxReconnectAttempts || 5,
          eventCount: status.eventCount || 0,
          lastError: status.lastError || null,
          socketId: status.socketId || null
        }
      }
    } catch (error) {
      console.warn('Erro ao atualizar status WebSocket:', error)
    }
  }

  async function reconectarWebSocket() {
    if (reconectandoWebSocket.value) return
    
    try {
      reconectandoWebSocket.value = true
      const ws = getWebSocketService()
      
      if (ws) {
        const sucesso = await ws.restart()
        if (sucesso) {
          if (window.Innoclapps) {
            window.Innoclapps.success('WebSocket reconectado com sucesso')
          }
        } else {
          if (window.Innoclapps) {
            window.Innoclapps.error('Falha ao reconectar WebSocket')
          }
        }
      }
    } catch (error) {
      console.error('Erro ao reconectar WebSocket:', error)
      if (window.Innoclapps) {
        window.Innoclapps.error('Erro ao tentar reconectar: ' + (error.message || ''))
      }
    } finally {
      reconectandoWebSocket.value = false
    }
  }

  function tratarErroConexao(error) {
    console.error('Erro de conexão:', error)
    if (window.Innoclapps) {
      window.Innoclapps.error('Problemas na conexão com o servidor')
    }
  }

  function startStatusInterval() {
    if (statusInterval) return
    statusInterval = setInterval(actualizarStatusWebSocket, 10000)
  }

  function stopStatusInterval() {
    if (statusInterval) {
      clearInterval(statusInterval)
      statusInterval = null
    }
  }

  onMounted(() => {
    startStatusInterval()
    actualizarStatusWebSocket()
  })

  onUnmounted(() => {
    stopStatusInterval()
  })

  return {
    websocketStatus,
    websocketStatusClass,
    websocketStatusText,
    reconectandoWebSocket,
    reconectarWebSocket,
    tratarErroConexao,
    startStatusInterval,
    stopStatusInterval
  }
}