// modules/Chat/resources/js/global-chat.js

class GlobalChatManager {
  constructor() {
    this.initialized = false

    this.services = {
      websocket: null,
      chatStore: null,
    }

    this.config = {
      debug: false,
      enabled: true,
      websocket_url: null,
      notifications: {
        enabled: true,
        sound: true,
        desktop: true,
        maxVisible: 3,
        duration: 5000,
        throttleTime: 3000,
      },
    }

    this.state = {
      lastNotification: { type: null, time: 0 },
      eventListeners: [],
      timers: [],
      healthCheckTimer: null,
    }
  }

  /** LOG CONTROLADO POR debug */
  log(level, ...args) {
    if (!this.config.debug && level !== 'error' && level !== 'warn') return
    const prefix = `[Chat Global] - ${level.toUpperCase()}:`
    // eslint-disable-next-line no-console
    console[level] ? console[level](prefix, ...args) : console.log(prefix, ...args)
  }

  /** ====== BOOT PRINCIPAL ====== */
  async init() {
    if (this.initialized) return true

    try {
      this.log('info', '🌐 Inicializando sistema de Chat Global...')

      this.loadConfig()
      if (!this.config.enabled) {
        this.log('warn', 'Chat global desabilitado em configuração')
        return false
      }

      await this.requestNotificationPermission()
      await this.loadServices()

      const connected = await this.services.websocket.init({
        url: this.config.websocket_url,
        debug: this.config.debug,
      })

      if (!connected) {
        throw new Error('Falha ao conectar WebSocket')
      }

      this.setupEventListeners()
      await this.loadInitialData()
      this.integrateWithNotifications()
      this.startHealthMonitoring()

      this.initialized = true
      window.chatGlobalInitialized = true

      this.log('info', '✅ Chat Global inicializado com sucesso')
      return true
    } catch (error) {
      this.log('error', 'Falha na inicialização do Chat Global:', error)
      this.cleanup()
      return false
    }
  }

  /** ====== CONFIG, SERVICES & DATA ====== */

  loadConfig() {
    const chatConfig = window.Innoclapps?.scriptConfig?.('chat')

    if (!chatConfig) {
      this.log('warn', 'Nenhuma configuração de Chat encontrada em scriptData("chat")')
      return
    }

    this.config.debug = !!chatConfig.debug
    this.config.enabled = chatConfig.enabled !== false
    this.config.websocket_url = chatConfig.websocket_url

    this.config.notifications = {
      ...this.config.notifications,
      enabled: chatConfig.notifications_enabled !== false,
      sound: chatConfig.sound_enabled !== false,
    }

    this.log('info', 'Configuração de Chat carregada:', this.config)
  }

  async loadServices() {
    const [{ default: websocketService }, { default: chatStore }] = await Promise.all([
      import('./servicos/websocket.js'),
      import('./servicos/chatStore.js'),
    ])

    this.services.websocket = websocketService
    this.services.chatStore = chatStore

    // Expor para outras partes da aplicação (principalmente PaginaChat.vue)
    window.websocketService = websocketService
    window.chatStore = chatStore

    this.log('info', 'Serviços de WebSocket e ChatStore carregados')
  }

  async loadInitialData() {
    try {
      if (window.Innoclapps?.request) {
        const response = await window.Innoclapps.request('/chat/conversations/summary')
        const unread = response?.data?.unread_count || 0
        this.updateNotificationBadge(unread)
      }
    } catch (error) {
      this.log('warn', 'Falha ao carregar dados iniciais do Chat:', error)
    }
  }

  /** ====== EVENTOS E INTEGRAÇÕES GLOBAIS ====== */

  setupEventListeners() {
    const listeners = [
      // Eventos vindos do WebSocket (chat-message-received)
      ['chat-message-received', this.handleNewMessage.bind(this)],

      // Compat: eventos disparados pelo chatStore (se usados)
      ['chat-store-new-unread-message', this.handleNewMessage.bind(this)],
      ['chat-store-updated', this.updateNotificationBadge.bind(this)],
      ['chat-outbound-message-received', this.handleOutboundMessage.bind(this)],

      // Eventos do WebSocket
      [
        'websocket-connected',
        () => this.showToast('Conectado ao servidor de chat', 'success', 2000),
      ],
      [
        'websocket-disconnected',
        () =>
          this.showToast(
            'Desconectado, tentando reconectar...',
            'warning',
            3000
          ),
      ],

      // Lifecycle da aba
      ['beforeunload', this.cleanup.bind(this)],
      ['pagehide', this.cleanup.bind(this)],
    ]

    listeners.forEach(([event, handler]) => {
      window.addEventListener(event, handler)
      this.state.eventListeners.push({ element: window, event, handler })
    })

    this.log('info', 'Listeners globais de Chat registrados')
  }

  async requestNotificationPermission() {
    if (!('Notification' in window)) return

    if (Notification.permission === 'default') {
      try {
        const permission = await Notification.requestPermission()
        this.log('info', 'Permissão de notificação:', permission)
      } catch (error) {
        this.log('warn', 'Falha ao solicitar permissão de notificação:', error)
      }
    }
  }

  /** ====== NOTIFICAÇÕES (NOVAS MENSAGENS) ====== */

  handleNewMessage(event) {
    const detail = event?.detail || event || {}

    let conversation = detail.conversation || null
    let message = detail.message || null
    let conversationId =
      detail.conversation_id ||
      detail.conversationId ||
      detail.conversation_id?.id ||
      (message && (message.conversation_id || message.conversationId))

    // Formato vindo do websocketService (chat-message-received)
    if (!conversation && conversationId && this.services.chatStore?.conversas) {
      const convIdNum = parseInt(conversationId)
      const found = this.services.chatStore.conversas.value?.find(
        c => parseInt(c.id) === convIdNum
      )
      conversation = found || { id: convIdNum }
    }

    // Caso o payload venha direto como data/message
    if (!message && detail.data?.message) {
      message = detail.data.message
    } else if (!message && detail.content) {
      message = detail
    }

    if (!conversation || !message) {
      this.log('warn', 'Evento de nova mensagem sem dados válidos', detail)
      return
    }

    const convId = parseInt(conversation.id || conversationId)
    if (Number.isNaN(convId)) {
      this.log('warn', 'ID de conversa inválido em handleNewMessage', {
        conversation,
        conversationId,
      })
      return
    }

    // 🚫 Não notificar mensagens do próprio usuário
    try {
      const currentUserId = this.services.websocket?.getCurrentUser?.()?.id
      const senderId = message.sender_id || message.sender?.id
      if (
        currentUserId &&
        senderId &&
        parseInt(currentUserId) === parseInt(senderId)
      ) {
        return
      }
    } catch (e) {
      this.log('warn', 'Erro ao comparar usuário atual com sender_id', e)
    }

    // Se já estou na conversa ativa no /chat, não preciso de notificação global
    if (this.isOnActiveConversation(convId)) {
      return
    }

    // 🧠 Montar nome do contato pra usar como título
    const sender = message.sender || {}
    const metadata = message.metadata || {}

    const fullName = [sender.first_name, sender.last_name].filter(Boolean).join(' ')
    const profileName =
      conversation.whatsapp_profile_name ||
      metadata.push_name

    const phone =
      sender.whatsapp_number ||
      conversation.contact_phone ||
      conversation.whatsapp_chat_id ||
      conversation.whatsapp_remote_jid ||
      metadata?.whatsapp_key?.remoteJid

    const contactName =
      conversation.contact_name ||
      fullName ||
      profileName ||
      phone ||
      `Conversa #${convId}`

    const rawContent = message.content || message.message || ''
    const normalizedContent = this.normalizeMessageContent(rawContent)

    this.showGlobalNotification({
      title: `${contactName} • Nova mensagem`,
      message: normalizedContent,
      conversation_id: convId,
    })

    // Atualizar badge baseado em unreadTotal do chatStore
    this.updateNotificationBadge()
  }

  showGlobalNotification({ title, message, conversation_id }) {
    if (!this.config.notifications.enabled) return

    const truncated = this.truncateMessage(message)

    // Notificação desktop
    if (
      this.config.notifications.desktop &&
      'Notification' in window &&
      Notification.permission === 'granted'
    ) {
      try {
        const notification = new Notification(title, {
          body: truncated,
          icon: '/favicon.ico',
          tag: `chat-${conversation_id}`,
          requireInteraction: false,
          silent: !this.config.notifications.sound,
        })

        notification.onclick = () => {
          window.focus()
          this.openChat(conversation_id)
          notification.close()
        }

        this.addTimer(
          setTimeout(
            () => notification.close(),
            this.config.notifications.duration
          )
        )
      } catch (error) {
        this.log('warn', 'Falha ao exibir notificação desktop:', error)
      }
    }

    // Toast visual
    this.showToast(`${title}: ${truncated}`, 'info')

    // Som
    if (this.config.notifications.sound) {
      this.playNotificationSound()
    }

    this.log('info', '🔔 Notificação global exibida:', {
      title,
      conversation_id,
    })
  }

  playNotificationSound() {
    try {
      const audio = new Audio(
        'data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmMeAjuZ2e/AcSEELYPP8dWLOAcTZLnp6Z1NEAxOpd/yxnkpBSl+zPLaizsIGGS57OihUgwLUarm7K5fFgU9k9n0unAhBC2Iz/DUizgHEWG56+OZURE='
      )
      audio.volume = 0.3
      audio.play().catch(() => {})
    } catch (e) {
      this.log('warn', 'Falha ao tocar som de notificação', e)
    }
  }

  /** ====== TOASTS VISUAIS ====== */

  showToast(message, type = 'info', duration = 4000) {
    const now = Date.now()

    // Anti-spam por tipo
    if (
      this.state.lastNotification.type === type &&
      now - this.state.lastNotification.time <
        this.config.notifications.throttleTime
    ) {
      return
    }

    this.state.lastNotification = { type, time: now }

    const container = document.getElementById('chat-notifications-container')
    if (!container) {
      this.log(
        'warn',
        'Container de notificações (#chat-notifications-container) não encontrado'
      )
      return
    }

    // Respeitar limite de toasts visíveis
    while (container.children.length >= this.config.notifications.maxVisible) {
      container.removeChild(container.firstElementChild)
    }

    const toast = document.createElement('div')
    toast.className = `chat-toast chat-toast-${type}`

    toast.innerHTML = `
      <div class="chat-toast-content">
        <div class="chat-toast-message">
          <div class="chat-toast-title">${this.getToastTitle(type)}</div>
          <div>${message}</div>
        </div>
        <button type="button" class="chat-toast-close" aria-label="Fechar notificação">&times;</button>
      </div>
    `

    const closeBtn = toast.querySelector('.chat-toast-close')
    const removeToast = () => {
      toast.classList.remove('chat-toast-show')
      this.addTimer(
        setTimeout(() => {
          if (toast.parentNode === container) {
            container.removeChild(toast)
          }
        }, 200)
      )
    }

    closeBtn.addEventListener('click', removeToast)

    container.appendChild(toast)

    // Animação
    requestAnimationFrame(() => {
      toast.classList.add('chat-toast-show')
    })

    // Remoção automática
    this.addTimer(setTimeout(removeToast, duration))
  }

  getToastTitle(type) {
    switch (type) {
      case 'success':
        return 'Sucesso'
      case 'error':
        return 'Erro'
      case 'warning':
        return 'Atenção'
      default:
        return 'Informação'
    }
  }

  /** ====== BADGE E TÍTULO ====== */

  // ✅ NOVO MÉTODO: Processar mensagens externas sem notificação
handleOutboundMessage(event) {
  const detail = event?.detail || event || {}
  
  // Apenas atualizar badge se necessário, sem notificação visual/sonora
  this.updateNotificationBadge()
  
  this.log('debug', '📤 Mensagem WhatsApp externa processada (sem notificação)', {
    conversation_id: detail.conversation_id,
    message_id: detail.message?.id
  })
}

  updateNotificationBadge(countOrEvent = null) {
    let count = 0

    if (typeof countOrEvent === 'number') {
      count = countOrEvent
    } else if (
      countOrEvent &&
      typeof countOrEvent === 'object' &&
      countOrEvent.detail?.unreadTotal != null
    ) {
      count = countOrEvent.detail.unreadTotal
    } else if (this.services.chatStore?.unreadTotal?.value != null) {
      count = this.services.chatStore.unreadTotal.value
    }

    const badges = document.querySelectorAll('.chat-notification-badge')
    badges.forEach(badge => {
      badge.textContent = count > 99 ? '99+' : String(count)
      badge.classList.toggle('show', count > 0)
    })

    const originalTitle = document.title.replace(/^\(\d+\)\s*/, '')
    document.title = count > 0 ? `(${count}) ${originalTitle}` : originalTitle

    if ('setAppBadge' in navigator) {
      if (count > 0) {
        navigator.setAppBadge(count).catch(() => {})
      } else {
        navigator.clearAppBadge().catch(() => {})
      }
    }
  }

  integrateWithNotifications() {
    const selectors = [
      '[data-notifications-button]',
      '.notifications-button',
      '[data-testid="notifications"]',
      '.notification-bell',
      '.nav-notifications',
    ]

    for (const selector of selectors) {
      const button = document.querySelector(selector)
      if (button && !button.querySelector('.chat-notification-badge')) {
        const badge = document.createElement('span')
        badge.className = 'chat-notification-badge'

        if (
          !['relative', 'absolute', 'fixed', 'sticky'].includes(
            button.style.position
          )
        ) {
          button.style.position = 'relative'
        }

        button.appendChild(badge)
        this.log(
          'info',
          '✅ Badge de notificações integrado no seletor:',
          selector
        )
        break
      }
    }
  }

  /** ====== HEALTH CHECK ====== */

  startHealthMonitoring() {
    if (this.state.healthCheckTimer) return

    const timer = setInterval(() => {
      const ok = this.performHealthCheck()
      if (!ok) {
        this.log(
          'info',
          '🔄 Health check falhou, tentando recuperar conexão...'
        )
        if (this.services.websocket && !this.services.websocket.isConnected) {
          this.services.websocket
            .restart()
            .catch(err =>
              this.log('error', 'Restart WebSocket falhou:', err)
            )
        }
      }

      // Atualizar badge periodicamente com base no chatStore
      this.updateNotificationBadge()
    }, 30000)

    this.state.healthCheckTimer = timer
    this.addTimer(timer)
  }

  performHealthCheck() {
    const wsOk = !!this.services.websocket?.isConnected
    const storeOk = !!this.services.chatStore
    const initOk = !!this.initialized
    return wsOk && storeOk && initOk
  }

  /** ====== HELPERS ====== */

  normalizeMessageContent(raw) {
    if (!raw || typeof raw !== 'string') return ''
    let text = raw

    // remover markdown simples do WhatsApp (*negrito*, _itálico_, etc. se quiser evoluir)
    text = text.replace(/\*/g, '')

    // normalizar quebras de linha em espaços
    text = text.replace(/\s+/g, ' ').trim()

    return text
  }

  truncateMessage(message) {
    if (!message || typeof message !== 'string') return ''
    return message.length > 100 ? message.substring(0, 97) + '...' : message
  }

  isOnActiveConversation(conversationId) {
    const isOnChatPage = window.location.pathname.includes('/chat')
    const selected = this.services.chatStore?.conversaSelecionada?.value
    return (
      isOnChatPage &&
      selected &&
      parseInt(selected.id) === parseInt(conversationId)
    )
  }

  openChat(conversationId = null) {
    const url = conversationId ? `/chat?conversation=${conversationId}` : '/chat'

    if (
      window.location.pathname === '/chat' &&
      conversationId &&
      this.services.chatStore?.selectConversation
    ) {
      this.services.chatStore.selectConversation(conversationId)
    } else {
      window.location.href = url
    }
  }

  addTimer(timerId) {
    if (timerId) this.state.timers.push(timerId)
    return timerId
  }

  /** ====== CLEANUP GLOBAL ====== */

  cleanup() {
    if (!this.initialized) return

    this.log('info', '🧹 Limpando Chat Global...')

    // Remover listeners
    this.state.eventListeners.forEach(({ element, event, handler }) => {
      try {
        element.removeEventListener(event, handler)
      } catch (e) {
        this.log('warn', 'Erro ao remover listener:', e)
      }
    })
    this.state.eventListeners = []

    // Timers
    this.state.timers.forEach(timerId => {
      try {
        clearTimeout(timerId)
      } catch (e) {
        this.log('warn', 'Erro ao limpar timer:', e)
      }
    })
    this.state.timers = []
    this.state.healthCheckTimer = null

    // Limpar container de notificações
    const container = document.getElementById('chat-notifications-container')
    if (container) container.innerHTML = ''

    // WebSocket
    if (this.services.websocket?.destroy) {
      try {
        this.services.websocket.destroy()
      } catch (e) {
        this.log('warn', 'Erro ao destruir WebSocket:', e)
      }
    }

    // Não destruímos o chatStore aqui, pois ele é global e reusado

    this.state.lastNotification = { type: null, time: 0 }
    this.initialized = false
    window.chatGlobalInitialized = false

    this.log('info', '✅ Chat Global limpo')
  }

  /** ====== DEBUG / STATUS ====== */

  getStatusSnapshot() {
    return {
      initialized: this.initialized,
      config: this.config,
      websocket: this.services.websocket?.getStatus?.() ?? null,
      chatStore: this.services.chatStore?.getStats?.() ?? null,
    }
  }
}

const globalChatManager = new GlobalChatManager()

/** API global amigável pra outros módulos/console */
window.chatRuntime = {
  init: () => globalChatManager.init(),
  cleanup: () => globalChatManager.cleanup(),
  status: () => globalChatManager.getStatusSnapshot(),
  debug: () => {
    // eslint-disable-next-line no-console
    console.log(
      '=== CHAT RUNTIME STATUS ===',
      globalChatManager.getStatusSnapshot()
    )
  },
}

/** Helper global de toast p/ outros pontos do sistema */
window.showGlobalToast = function (message, type = 'info', duration = 4000) {
  globalChatManager.showToast(message, type, duration)
}

/** Compat para código antigo, se você ainda chamar em algum lugar */
window.cleanupGlobalChat = function () {
  globalChatManager.cleanup()
}

/** Boot automático quando DOM estiver pronto */
if (typeof window !== 'undefined') {
  const initWhenReady = () => {
    if (!globalChatManager.initialized) {
      globalChatManager.init()
    }
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
      setTimeout(initWhenReady, 500)
    })
  } else {
    setTimeout(initWhenReady, 500)
  }
}

export default globalChatManager
