// modules/Chat/resources/js/services/globalChatService.js
class GlobalChatService {
  constructor() {
    this.initialized = false
    this.config = {}
    this.eventListeners = []
    this.timers = []
    this.state = {
      lastNotification: { type: null, time: 0 },
      notificationsPaused: false
    }
  }

  async init(config = {}) {
    if (this.initialized) return true

    try {
      this.config = {
        enabled: config.enabled !== false,
        debug: config.debug || false,
        notifications: {
          enabled: config.notifications_enabled !== false,
          sound: config.sound_enabled !== false,
          desktop: config.desktop_enabled !== false,
          throttleTime: config.throttle_time || 3000
        },
        ...config
      }

      if (!this.config.enabled) {
        console.log('GlobalChatService: Disabled in configuration')
        return false
      }

      await this.waitForServices()
      this.setupEventListeners()
      await this.loadInitialData()
      this.integrateWithUI()
      this.startHealthMonitoring()
      
      this.initialized = true
      window.chatGlobalInitialized = true
      
      console.log('✅ GlobalChatService: Initialized')
      return true

    } catch (error) {
      console.error('❌ GlobalChatService: Initialization failed:', error)
      return false
    }
  }

  log(level, ...args) {
    if (!this.config.debug && level !== 'error') return
    console[level](`[GlobalChat] - ${level.toUpperCase()}:`, ...args)
  }

  async waitForServices() {
    let attempts = 0
    while ((!window.websocketService || !window.chatStore || !window.notificationService) && attempts < 50) {
      await new Promise(resolve => setTimeout(resolve, 100))
      attempts++
    }

    if (!window.websocketService || !window.chatStore || !window.notificationService) {
      throw new Error('Required services not available')
    }
  }

  setupEventListeners() {
    const listeners = [
      ['chat-store-new-unread-message', this.handleNewMessage.bind(this)],
      ['chat-store-updated', this.updateNotificationBadge.bind(this)],
      ['websocket-connected', () => this.showToast('Conectado', 'success', 2000)],
      ['websocket-disconnected', () => this.showToast('Desconectado, reconectando...', 'warning', 3000)],
      ['beforeunload', this.cleanup.bind(this)],
      ['pagehide', this.cleanup.bind(this)]
    ]

    listeners.forEach(([event, handler]) => {
      window.addEventListener(event, handler)
      this.eventListeners.push({ event, handler })
    })

    this.log('info', 'Event listeners configurados')
  }

  handleNewMessage(event) {
    if (this.state.notificationsPaused) return

    const { conversation, message } = event.detail
    if (!this.isOnActiveConversation(conversation.id)) {
      this.showGlobalNotification({
        title: message.sender_name || 'Nova Mensagem',
        message: message.content,
        conversation_id: conversation.id
      })
    }
  }

  showGlobalNotification({ title, message, conversation_id }) {
    if (!this.config.notifications.enabled || this.state.notificationsPaused) return

    // Throttle de notificações
    const now = Date.now()
    if (this.state.lastNotification.type === 'message' && 
        (now - this.state.lastNotification.time) < this.config.notifications.throttleTime) {
      return
    }

    this.state.lastNotification = { type: 'message', time: now }

    // Notificação desktop
    if (this.config.notifications.desktop && 'Notification' in window && Notification.permission === 'granted') {
      const notification = new Notification(title, {
        body: this.truncateMessage(message),
        icon: '/favicon.ico',
        tag: `chat-${conversation_id}`,
        requireInteraction: false,
        silent: !this.config.notifications.sound
      })

      notification.onclick = () => {
        window.focus()
        this.openChat(conversation_id)
        notification.close()
      }

      this.addTimer(setTimeout(() => notification.close(), 5000))
    }

    // Toast notification
    this.showToast(`${title}: ${this.truncateMessage(message)}`, 'info')
    
    // Som de notificação
    if (this.config.notifications.sound) {
      this.playNotificationSound()
    }

    this.log('info', '🔔 Global notification shown:', { title, conversation_id })
  }

  showToast(message, type = 'info', duration = 4000) {
    // Usar sistema de notificação do Concord CRM
    if (window.Innoclapps) {
      if (type === 'success') window.Innoclapps.success(message, duration)
      else if (type === 'error') window.Innoclapps.error(message, duration)
      else if (type === 'warning') window.Innoclapps.warning(message, duration)
      else window.Innoclapps.info(message, duration)
    }

    // Também usar serviço interno se disponível
    const notificationService = window.notificationService
    if (notificationService && notificationService.show) {
      notificationService.show({
        type,
        message,
        duration,
        showDesktop: false
      })
    }
  }

  playNotificationSound() {
    try {
      const notificationService = window.notificationService
      if (notificationService && notificationService.playSound) {
        notificationService.playSound('message')
      } else {
        // Fallback para áudio básico
        const audio = new Audio('data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmMeAjuZ2e/AcSEELYPP8dWLOAcTZLnp6Z1NEAxOpd/yxnkpBSl+zPLaizsIGGS57OihUgwLUarm7K5fFgU9k9n0unAhBC2Iz/DUizgHEWG56+OZURE=')
        audio.volume = 0.3
        audio.play().catch(() => {})
      }
    } catch (e) {
      this.log('warn', 'Sound notification failed', e)
    }
  }

  updateNotificationBadge(event) {
    const count = event.detail?.unreadTotal || window.chatStore?.unreadTotal?.value || 0
    
    // Atualizar badges na UI
    document.querySelectorAll('.chat-notification-badge').forEach(badge => {
      badge.textContent = count > 99 ? '99+' : count
      badge.style.display = count > 0 ? 'flex' : 'none'
    })

    // Atualizar título da página
    const originalTitle = document.title.replace(/^\(\d+\)\s*/, '')
    document.title = count > 0 ? `(${count}) ${originalTitle}` : originalTitle

    // Badge do navegador
    if ('setAppBadge' in navigator) {
      count > 0 ? 
        navigator.setAppBadge(count).catch(() => {}) : 
        navigator.clearAppBadge().catch(() => {})
    }

    this.log('info', 'Badge atualizado:', count)
  }

  async requestNotificationPermission() {
    if ('Notification' in window && Notification.permission === 'default') {
      try {
        const permission = await Notification.requestPermission()
        this.log('info', 'Notification permission:', permission)
        return permission === 'granted'
      } catch (error) {
        this.log('warn', 'Notification permission request failed:', error)
        return false
      }
    }
    return Notification.permission === 'granted'
  }

  async loadInitialData() {
    try {
      if (window.Innoclapps?.request) {
        const response = await window.Innoclapps.request('/chat/conversations/summary')
        if (response.data?.unread_count) {
          this.updateNotificationBadge({ detail: { unreadTotal: response.data.unread_count } })
        }
      }
    } catch (error) {
      this.log('warn', 'Failed to load initial data:', error)
    }
  }

  integrateWithUI() {
    const selectors = [
      '[data-notifications-button]',
      '.notifications-button',
      '[data-testid="notifications"]',
      '.notification-bell',
      '.nav-notifications'
    ]

    for (const selector of selectors) {
      const button = document.querySelector(selector)
      if (button && !button.querySelector('.chat-notification-badge')) {
        const badge = document.createElement('span')
        badge.className = 'chat-notification-badge'
        
        if (button.style.position !== 'relative') {
          button.style.position = 'relative'
        }
        
        button.appendChild(badge)
        this.log('info', '✅ Badge integrated:', selector)
        break
      }
    }
  }

  startHealthMonitoring() {
    const healthTimer = setInterval(() => {
      if (!this.performHealthCheck()) {
        this.log('info', '🔄 Attempting recovery...')
        const websocketService = window.websocketService
        if (websocketService && !websocketService.isValid()) {
          websocketService.restart().catch(error => {
            this.log('error', 'Recovery failed:', error)
          })
        }
      }
    }, 30000)
    
    this.addTimer(healthTimer)
  }

  performHealthCheck() {
    const websocketService = window.websocketService
    const chatStore = window.chatStore
    const notificationService = window.notificationService
    
    return websocketService?.isValid() && 
           chatStore?.isInitialized && 
           notificationService?.isInitialized &&
           this.initialized
  }

  truncateMessage(message) {
    return message && message.length > 100 ? message.substring(0, 97) + '...' : message
  }

  isOnActiveConversation(conversationId) {
    const isOnChatPage = window.location.pathname.includes('/chat')
    const chatStore = window.chatStore
    return isOnChatPage && 
           chatStore?.conversaSelecionada?.value?.id === parseInt(conversationId)
  }

  openChat(conversationId = null) {
    const url = conversationId ? `/chat?conversation=${conversationId}` : '/chat'
    
    if (window.location.pathname === '/chat' && conversationId) {
      const chatStore = window.chatStore
      if (chatStore && chatStore.selectConversation) {
        chatStore.selectConversation(conversationId)
      }
    } else {
      window.location.href = url
    }
  }

  // Métodos para controle de notificações
  pauseNotifications() {
    this.state.notificationsPaused = true
    this.log('info', 'Notifications paused')
  }

  resumeNotifications() {
    this.state.notificationsPaused = false
    this.log('info', 'Notifications resumed')
  }

  addTimer(timerId) {
    if (timerId) this.timers.push(timerId)
    return timerId
  }

  cleanup() {
    if (!this.initialized) return

    this.log('info', '🧹 Cleaning up global chat...')

    // Remover event listeners
    this.eventListeners.forEach(({ event, handler }) => {
      try {
        window.removeEventListener(event, handler)
      } catch (e) {
        this.log('warn', 'Error removing listener:', e)
      }
    })

    // Limpar timers
    this.timers.forEach(timerId => {
      try {
        clearTimeout(timerId)
      } catch (e) {
        this.log('warn', 'Error clearing timer:', e)
      }
    })

    // Limpar container de notificações
    const container = document.getElementById('chat-notifications-container')
    if (container) container.innerHTML = ''

    // Reset state
    this.eventListeners = []
    this.timers = []
    this.state = {
      lastNotification: { type: null, time: 0 },
      notificationsPaused: false
    }

    this.initialized = false
    window.chatGlobalInitialized = false

    this.log('info', '✅ Global chat cleaned up')
  }

  destroy() {
    this.cleanup()
    console.log('✅ GlobalChatService: Destroyed')
  }
}

export default new GlobalChatService()