// modules/Chat/resources/js/services/notificationService.js
class NotificationService {
  constructor() {
    this.notifications = new Map()
    this.config = {}
    this.container = null
    this.soundCache = new Map()
    this.timers = []
    this.isInitialized = false
  }

  async init(config = {}) {
    if (this.isInitialized) return true

    this.config = {
      maxVisible: config.maxVisible || 3,
      defaultDuration: config.defaultDuration || 5000,
      position: config.position || 'top-right',
      enableSound: config.enableSound !== false,
      enableDesktop: config.enableDesktop !== false,
      sounds: {
        message: config.sounds?.message || 'data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmMeAjuZ2e/AcSEELYPP8dWLOAcTZLnp6Z1NEAxOpd/yxnkpBSl+zPLaizsIGGS57OihUgwLUarm7K5fFgU9k9n0unAhBC2Iz/DUizgHEWG56+OZURE=',
        success: config.sounds?.success || 'data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmMeAjuZ2e/AcSEELYPP8dWLOAcTZLnp6Z1NEAxOpd/yxnkpBSl+zPLaizsIGGS57OihUgwLUarm7K5fFgU9k9n0unAhBC2Iz/DUizgHEWG56+OZURE=',
        error: config.sounds?.error || 'data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmMeAjuZ2e/AcSEELYPP8dWLOAcTZLnp6Z1NEAxOpd/yxnkpBSl+zPLaizsIGGS57OihUgwLUarm7K5fFgU9k9n0unAhBC2Iz/DUizgHEWG56+OZURE='
      },
      ...config
    }

    try {
      this.createContainer()
      await this.preloadSounds()
      await this.requestPermissions()
      
      this.isInitialized = true
      console.log('✅ NotificationService: Initialized')
      return true
    } catch (error) {
      console.error('❌ NotificationService: Init failed:', error)
      return false
    }
  }

  createContainer() {
    this.container = document.getElementById('chat-notifications-container')
    if (!this.container) {
      this.container = document.createElement('div')
      this.container.id = 'chat-notifications-container'
      this.container.className = `chat-notifications-container position-${this.config.position}`
      document.body.appendChild(this.container)
    }
  }

  async preloadSounds() {
    for (const [type, src] of Object.entries(this.config.sounds)) {
      try {
        const audio = new Audio(src)
        audio.volume = 0.3
        audio.preload = 'auto'
        await new Promise((resolve, reject) => {
          audio.oncanplaythrough = resolve
          audio.onerror = reject
          audio.load()
        })
        this.soundCache.set(type, audio)
      } catch (error) {
        console.warn(`Failed to preload sound: ${type}`, error)
      }
    }
  }

  async requestPermissions() {
    if ('Notification' in window && Notification.permission === 'default') {
      try {
        const permission = await Notification.requestPermission()
        console.log('Notification permission:', permission)
      } catch (error) {
        console.warn('Notification permission request failed:', error)
      }
    }
  }

  show(options) {
    if (!this.isInitialized) {
      console.warn('NotificationService not initialized')
      return null
    }

    const notification = this.createNotification(options)
    
    this.manageVisibleNotifications()
    this.renderNotification(notification)
    this.notifications.set(notification.id, notification)
    
    if (this.config.enableSound && options.playSound !== false) {
      this.playSound(notification.type)
    }
    
    if (this.config.enableDesktop && document.hidden && options.showDesktop !== false) {
      this.showDesktopNotification(notification)
    }
    
    if (notification.duration > 0) {
      const timer = setTimeout(() => this.hide(notification.id), notification.duration)
      this.timers.push(timer)
    }
    
    return notification.id
  }

  createNotification(options) {
    const {
      type = 'info',
      title = '',
      message = '',
      duration = this.config.defaultDuration,
      actions = [],
      data = {},
      priority = 'normal'
    } = options

    return {
      id: this.generateId(),
      type,
      title,
      message,
      duration,
      actions,
      data,
      priority,
      createdAt: Date.now(),
      element: null
    }
  }

  renderNotification(notification) {
    const element = document.createElement('div')
    element.className = `chat-toast chat-toast-${notification.type}`
    element.setAttribute('data-notification-id', notification.id)
    
    element.innerHTML = `
      <div class="chat-toast-content">
        <div class="chat-toast-message">
          ${notification.title ? `<div class="chat-toast-title">${this.escapeHtml(notification.title)}</div>` : ''}
          <div class="chat-toast-body">${this.escapeHtml(notification.message)}</div>
        </div>
        <button class="chat-toast-close" type="button" aria-label="Fechar notificação">
          ×
        </button>
      </div>
    `

    // Event listeners
    const closeButton = element.querySelector('.chat-toast-close')
    closeButton.addEventListener('click', () => this.hide(notification.id))

    element.addEventListener('click', (e) => {
      if (e.target !== closeButton) {
        this.handleNotificationClick(notification)
      }
    })

    notification.element = element
    this.container.appendChild(element)

    // Trigger animation
    requestAnimationFrame(() => {
      element.classList.add('chat-toast-show')
    })
  }

  manageVisibleNotifications() {
    const visible = Array.from(this.notifications.values())
      .filter(n => n.element && n.element.parentNode)
      .sort((a, b) => a.createdAt - b.createdAt)

    if (visible.length >= this.config.maxVisible) {
      const toRemove = visible.slice(0, visible.length - this.config.maxVisible + 1)
      toRemove.forEach(notification => this.hide(notification.id))
    }
  }

  hide(id) {
    const notification = this.notifications.get(id)
    if (!notification || !notification.element) return

    notification.element.classList.remove('chat-toast-show')
    
    setTimeout(() => {
      if (notification.element && notification.element.parentNode) {
        notification.element.parentNode.removeChild(notification.element)
      }
      this.notifications.delete(id)
    }, 300)
  }

  playSound(type) {
    if (!this.config.enableSound) return
    
    const audio = this.soundCache.get(type) || this.soundCache.get('message')
    if (audio) {
      try {
        audio.currentTime = 0
        audio.play().catch(() => {})
      } catch (error) {
        console.warn('Failed to play notification sound:', error)
      }
    }
  }

  showDesktopNotification(notification) {
    if (!this.config.enableDesktop) return
    
    if ('Notification' in window && Notification.permission === 'granted') {
      const desktopNotification = new Notification(notification.title || 'Nova mensagem', {
        body: notification.message,
        icon: '/favicon.ico',
        tag: `chat-${notification.data.conversation_id || notification.id}`,
        requireInteraction: false,
        silent: true
      })

      desktopNotification.onclick = () => {
        window.focus()
        this.handleNotificationClick(notification)
        desktopNotification.close()
      }

      const timer = setTimeout(() => desktopNotification.close(), notification.duration)
      this.timers.push(timer)
    }
  }

  handleNotificationClick(notification) {
    if (notification.data.conversation_id) {
      // Navegar para a conversa
      const chatStore = window.chatStore
      if (chatStore && chatStore.selectConversation) {
        chatStore.selectConversation(notification.data.conversation_id)
      }
      
      if (window.location.pathname !== '/chat') {
        window.location.href = `/chat?conversation=${notification.data.conversation_id}`
      }
    }

    this.hide(notification.id)
  }

  // Métodos de conveniência
  success(title, message, options = {}) {
    return this.show({ ...options, type: 'success', title, message })
  }

  error(title, message, options = {}) {
    return this.show({ ...options, type: 'error', title, message })
  }

  warning(title, message, options = {}) {
    return this.show({ ...options, type: 'warning', title, message })
  }

  info(title, message, options = {}) {
    return this.show({ ...options, type: 'info', title, message })
  }

  clear() {
    this.notifications.forEach(notification => this.hide(notification.id))
  }

  updateConfig(newConfig) {
    Object.assign(this.config, newConfig)
  }

  getStatus() {
    return {
      isInitialized: this.isInitialized,
      activeNotifications: this.notifications.size,
      config: { ...this.config }
    }
  }

  escapeHtml(text) {
    const div = document.createElement('div')
    div.textContent = text
    return div.innerHTML
  }

  generateId() {
    return `notification_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`
  }

  destroy() {
    console.log('🧹 NotificationService: Destroying...')
    
    this.clear()
    
    this.timers.forEach(timer => clearTimeout(timer))
    this.timers = []
    
    if (this.container && this.container.parentNode) {
      this.container.parentNode.removeChild(this.container)
    }
    
    this.soundCache.clear()
    this.notifications.clear()
    this.isInitialized = false
    
    console.log('✅ NotificationService: Destroyed')
  }
}

export default new NotificationService()