// modules/Chat/resources/js/services/websocketService.js
class WebSocketService {
  constructor() {
    this.connection = null
    this.config = {}
    this.state = {
      isConnected: false,
      isInitialized: false,
      connectionState: 'disconnected',
      reconnectAttempts: 0,
      eventCount: 0,
      lastError: null,
      socketId: null,
      currentUser: null
    }
    
    this.messageQueue = []
    this.eventHandlers = new Map()
    this.rateLimiter = new Map()
    this.timers = []
    
    this.setupDefaultHandlers()
  }

  async init(config = {}) {
    this.config = {
      url: config.url || 'ws://localhost:3000',
      reconnectAttempts: config.reconnectAttempts || 5,
      reconnectDelay: config.reconnectDelay || 1000,
      heartbeatInterval: config.heartbeatInterval || 30000,
      rateLimit: {
        maxEvents: config.maxEvents || 10,
        window: config.window || 1000
      },
      ...config
    }

    try {
      const success = await this.connect()
      if (success) {
        this.state.isInitialized = true
        this.startHeartbeat()
        this.emit('websocket-initialized')
        console.log('✅ WebSocketService: Initialized')
      }
      return success
    } catch (error) {
      this.state.lastError = error.message
      this.emit('websocket-error', { error })
      console.error('❌ WebSocketService: Init failed:', error)
      return false
    }
  }

  async connect() {
    if (this.connection?.readyState === WebSocket.OPEN) {
      return true
    }

    this.state.connectionState = 'connecting'
    this.emit('websocket-connecting')

    return new Promise((resolve, reject) => {
      try {
        this.connection = new WebSocket(this.config.url)
        
        const timeout = setTimeout(() => {
          this.connection.close()
          reject(new Error('Connection timeout'))
        }, 10000)

        this.connection.onopen = () => {
          clearTimeout(timeout)
          this.handleConnectionOpen()
          resolve(true)
        }

        this.connection.onerror = (error) => {
          clearTimeout(timeout)
          this.handleConnectionError(error)
          reject(error)
        }

        this.connection.onmessage = (event) => {
          this.handleMessage(event)
        }

        this.connection.onclose = (event) => {
          this.handleConnectionClose(event)
        }

      } catch (error) {
        reject(error)
      }
    })
  }

  handleConnectionOpen() {
    this.state.isConnected = true
    this.state.connectionState = 'connected'
    this.state.reconnectAttempts = 0
    this.state.socketId = this.generateSocketId()
    
    this.processMessageQueue()
    this.emit('websocket-connected')
    
    console.log('✅ WebSocket connected')
  }

  handleConnectionError(error) {
    this.state.lastError = error.message || 'Connection error'
    this.state.connectionState = 'error'
    this.emit('websocket-error', { error })
    
    console.error('❌ WebSocket error:', error)
  }

  handleConnectionClose(event) {
    this.state.isConnected = false
    this.state.connectionState = 'disconnected'
    this.stopHeartbeat()
    
    this.emit('websocket-disconnected', { 
      code: event.code, 
      reason: event.reason 
    })
    
    console.log('🔌 WebSocket disconnected:', event.code, event.reason)
    
    // Auto-reconectar se não foi fechamento intencional
    if (event.code !== 1000 && this.state.reconnectAttempts < this.config.reconnectAttempts) {
      this.scheduleReconnect()
    }
  }

  handleMessage(event) {
    try {
      const data = JSON.parse(event.data)
      this.state.eventCount++
      
      // Processar mensagem baseado no tipo
      switch (data.type) {
        case 'pong':
          // Resposta do heartbeat
          break
        case 'user_authenticated':
          this.state.currentUser = data.user
          this.emit('websocket-authenticated', data)
          break
        case 'message_received':
          this.emit('chat-message-received', data)
          break
        case 'user_typing':
          this.emit('chat-user-typing', data)
          break
        case 'conversation_read':
          this.emit('chat-conversation-read', data)
          break
        case 'conversation_updated':
          this.emit('chat-conversation-updated', data)
          break
        default:
          this.emit('websocket-message', data)
      }
      
    } catch (error) {
      console.error('❌ Failed to parse WebSocket message:', error)
    }
  }

  sendEvent(type, data = {}) {
    if (!this.canSendEvent(type)) {
      console.warn(`Rate limit exceeded for event: ${type}`)
      return false
    }

    const message = {
      type,
      data,
      timestamp: new Date().toISOString(),
      client_id: this.state.socketId
    }

    if (this.state.isConnected && this.connection.readyState === WebSocket.OPEN) {
      try {
        this.connection.send(JSON.stringify(message))
        return true
      } catch (error) {
        console.error('❌ Failed to send message:', error)
        this.queueMessage(message)
        return false
      }
    } else {
      this.queueMessage(message)
      return false
    }
  }

  canSendEvent(eventType) {
    const now = Date.now()
    const windowStart = now - this.config.rateLimit.window
    
    if (!this.rateLimiter.has(eventType)) {
      this.rateLimiter.set(eventType, [])
    }
    
    const events = this.rateLimiter.get(eventType)
    const recentEvents = events.filter(time => time > windowStart)
    
    if (recentEvents.length >= this.config.rateLimit.maxEvents) {
      return false
    }
    
    recentEvents.push(now)
    this.rateLimiter.set(eventType, recentEvents)
    return true
  }

  queueMessage(message) {
    if (this.messageQueue.length >= 100) {
      this.messageQueue.shift()
    }
    this.messageQueue.push(message)
  }

  processMessageQueue() {
    while (this.messageQueue.length > 0 && this.state.isConnected) {
      const message = this.messageQueue.shift()
      
      // Verificar se mensagem não é muito antiga (5 minutos)
      const messageTime = new Date(message.timestamp).getTime()
      if (Date.now() - messageTime < 300000) {
        this.connection.send(JSON.stringify(message))
      }
    }
  }

  startHeartbeat() {
    if (this.heartbeatTimer) return
    
    this.heartbeatTimer = setInterval(() => {
      if (this.state.isConnected) {
        this.sendEvent('ping', { timestamp: Date.now() })
      }
    }, this.config.heartbeatInterval)
    
    this.timers.push(this.heartbeatTimer)
  }

  stopHeartbeat() {
    if (this.heartbeatTimer) {
      clearInterval(this.heartbeatTimer)
      this.heartbeatTimer = null
    }
  }

  scheduleReconnect() {
    this.state.reconnectAttempts++
    const delay = Math.min(
      this.config.reconnectDelay * Math.pow(2, this.state.reconnectAttempts - 1),
      30000
    )
    
    console.log(`🔄 Reconnecting in ${delay}ms (attempt ${this.state.reconnectAttempts}/${this.config.reconnectAttempts})`)
    
    const reconnectTimer = setTimeout(async () => {
      try {
        await this.connect()
      } catch (error) {
        console.error('❌ Reconnection failed:', error)
      }
    }, delay)
    
    this.timers.push(reconnectTimer)
  }

  async restart() {
    console.log('🔄 WebSocket: Restarting...')
    this.destroy()
    return await this.init(this.config)
  }

  // Métodos específicos do chat
  async joinConversation(conversationId, type = 'external') {
    return this.sendEvent('join_conversation', {
      conversation_id: conversationId,
      type
    })
  }

  async leaveConversation(conversationId) {
    return this.sendEvent('leave_conversation', {
      conversation_id: conversationId
    })
  }

  async sendTyping(conversationId, isTyping = true) {
    return this.sendEvent('user_typing', {
      conversation_id: conversationId,
      is_typing: isTyping,
      user_id: this.state.currentUser?.id
    })
  }

  async markAsRead(conversationId) {
    return this.sendEvent('mark_as_read', {
      conversation_id: conversationId,
      user_id: this.state.currentUser?.id,
      read_at: new Date().toISOString()
    })
  }

  // Event system
  setupDefaultHandlers() {
    // Handlers padrão podem ser configurados aqui
  }

  on(eventName, handler) {
    if (!this.eventHandlers.has(eventName)) {
      this.eventHandlers.set(eventName, new Set())
    }
    this.eventHandlers.get(eventName).add(handler)
    
    // Também registrar no window para compatibilidade
    window.addEventListener(eventName, handler)
  }

  off(eventName, handler) {
    if (this.eventHandlers.has(eventName)) {
      this.eventHandlers.get(eventName).delete(handler)
    }
    window.removeEventListener(eventName, handler)
  }

  emit(eventName, data = {}) {
    // Emitir via window para compatibilidade
    window.dispatchEvent(new CustomEvent(eventName, { detail: data }))
    
    // Emitir para handlers internos
    if (this.eventHandlers.has(eventName)) {
      this.eventHandlers.get(eventName).forEach(handler => {
        try {
          handler({ detail: data })
        } catch (error) {
          console.error(`Error in event handler for ${eventName}:`, error)
        }
      })
    }
  }

  // Getters
  isValid() {
    return this.state.isInitialized && this.state.isConnected
  }

  getCurrentUser() {
    return this.state.currentUser
  }

  getStatus() {
    return { ...this.state }
  }

  generateSocketId() {
    return `socket_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`
  }

  destroy() {
    console.log('🧹 WebSocketService: Destroying...')
    
    this.stopHeartbeat()
    
    // Limpar timers
    this.timers.forEach(timer => clearTimeout(timer))
    this.timers = []
    
    // Limpar event handlers
    this.eventHandlers.forEach((handlers, eventName) => {
      handlers.forEach(handler => {
        window.removeEventListener(eventName, handler)
      })
    })
    this.eventHandlers.clear()
    
    // Fechar conexão
    if (this.connection) {
      this.connection.onopen = null
      this.connection.onmessage = null
      this.connection.onerror = null
      this.connection.onclose = null
      
      if (this.connection.readyState === WebSocket.OPEN) {
        this.connection.close(1000, 'Service destroyed')
      }
      
      this.connection = null
    }
    
    // Reset state
    this.state = {
      isConnected: false,
      isInitialized: false,
      connectionState: 'disconnected',
      reconnectAttempts: 0,
      eventCount: 0,
      lastError: null,
      socketId: null,
      currentUser: null
    }
    
    this.messageQueue = []
    this.rateLimiter.clear()
    
    console.log('✅ WebSocketService: Destroyed')
  }
}

export default new WebSocketService()