import { ref, watch, shallowRef, markRaw } from 'vue'

class ChatStore {
  constructor() {
    this.conversas = shallowRef([])
    this.conversaSelecionada = ref(null)
    this.usuariosDigitando = shallowRef(new Map())
    this.isLoading = ref(false)
    this.lastUpdate = ref(null)
    this.unreadTotal = ref(0)

    this.notificationsEnabled = ref(true)
    this.soundEnabled = ref(true)
    this.loadInitialConversations = null

    this.cache = {
      sortedConversations: null,
      filteredConversations: new Map(),
      lastSortHash: null,
      lastFilterHash: new Map(),
    }

    this.metrics = {
      sortCount: 0,
      updateCount: 0,
      cacheHits: 0,
      cacheMisses: 0,
      lastResetTime: Date.now(),
    }

    this.config = {
      debounceSort: 100,
      debounceUpdateTotal: 200,
      typingCleanupInterval: 10000,
      maxCacheSize: 1000,
    }

    this.sortTimer = null
    this.updateTotalTimer = null
    this.typingCleanupTimer = null
    this.cacheCleanupTimer = null

    this.init()

    // console.log('ChatStore: Inicializado (versão otimizada, sem marcação automática)')
  }

  // ========= INIT / CLEANUP =========

  init() {
    this.setupWatchers()
    this.setupGlobalListeners()
    this.startTypingCleanup()
    this.startCacheCleanup()
  }

  destroy() {
    if (this.sortTimer) clearTimeout(this.sortTimer)
    if (this.updateTotalTimer) clearTimeout(this.updateTotalTimer)
    if (this.typingCleanupTimer) clearInterval(this.typingCleanupTimer)
    if (this.cacheCleanupTimer) clearInterval(this.cacheCleanupTimer)

    this.reset()
    this.loadInitialConversations = null

    // console.log('ChatStore: Destruído')
  }

  reset() {
    this.conversas.value = markRaw([])
    this.conversaSelecionada.value = null
    this.usuariosDigitando.value = markRaw(new Map())
    this.isLoading.value = false
    this.lastUpdate.value = null
    this.unreadTotal.value = 0
    this.invalidateCache('all')
    this.resetMetrics()
    this.emitStoreUpdated()
    // console.log('ChatStore: Resetado')
  }

  resetMetrics() {
    this.metrics = {
      sortCount: 0,
      updateCount: 0,
      cacheHits: 0,
      cacheMisses: 0,
      lastResetTime: Date.now(),
    }
    // console.log('ChatStore: Métricas resetadas')
  }

  // ========= HELPERS =========

  emitStoreUpdated() {
    try {
      window.dispatchEvent(
        new CustomEvent('chat-store-updated', {
          detail: this.getStats(),
        })
      )
    } catch (e) {
      console.error('ChatStore: Erro ao emitir chat-store-updated:', e)
    }
  }

  emitNewUnreadMessage(conversation, message) {
    try {
      window.dispatchEvent(
        new CustomEvent('chat-store-new-unread-message', {
          detail: { conversation, message },
        })
      )
    } catch (e) {
      console.error('ChatStore: Erro ao emitir chat-store-new-unread-message:', e)
    }
  }

  // ✅ NOVO: Emitir evento global genérico
  emitGlobalEvent(eventName, data) {
    try {
      const event = new CustomEvent(eventName, { detail: data })
      window.dispatchEvent(event)
      console.log(`📡 ChatStore: Evento ${eventName} emitido`, data)
    } catch (error) {
      console.error(`❌ ChatStore: Erro ao emitir evento ${eventName}:`, error)
    }
  }

  generateSortHash(conversations) {
    if (!conversations || conversations.length === 0) return 'empty'

    return conversations
      .map(
        c =>
          `${c.id}:${c.unread_count || 0}:${c.is_pinned || false}:${
            c.last_message_at || ''
          }:${this.usuariosDigitando.value.has(c.id)}`
      )
      .join('|')
  }

  generateFilterHash(filter, conversations) {
    return `${filter}:${conversations.length}:${this.generateSortHash(conversations)}`
  }

  invalidateCache(type = 'all') {
    switch (type) {
      case 'sort':
        this.cache.sortedConversations = null
        this.cache.lastSortHash = null
        break
      case 'filter':
        this.cache.filteredConversations.clear()
        this.cache.lastFilterHash.clear()
        break
      case 'all':
      default:
        this.cache.sortedConversations = null
        this.cache.filteredConversations.clear()
        this.cache.lastSortHash = null
        this.cache.lastFilterHash.clear()
        break
    }
  }

  startCacheCleanup() {
    this.cacheCleanupTimer = setInterval(() => {
      try {
        if (this.cache.filteredConversations.size > this.config.maxCacheSize) {
          const entries = Array.from(this.cache.filteredConversations.entries())
          const toKeep = entries.slice(-Math.floor(this.config.maxCacheSize / 2))
          this.cache.filteredConversations.clear()
          toKeep.forEach(([key, value]) =>
            this.cache.filteredConversations.set(key, value)
          )
        }
      } catch (error) {
        console.error('ChatStore: Erro na limpeza de cache:', error)
      }
    }, 300000)
  }

  // ========= WATCHERS / LISTENERS =========

  setupWatchers() {
    watch(
      this.conversas,
      (novas, antigas) => {
        const novoTamanho = novas?.length || 0
        const tamanhoAnterior = antigas?.length || 0

        if (novoTamanho !== tamanhoAnterior) {
          console.log('ChatStore: Lista atualizada:', tamanhoAnterior, '->', novoTamanho)
          this.lastUpdate.value = new Date().toISOString()
          this.invalidateCache('filter')
        }

        this.debouncedUpdateTotal()
      },
      { deep: false }
    )

    watch(
      this.usuariosDigitando,
      nova => {
        if (nova.size > 0) {
          console.log('ChatStore: Digitação ativa em', nova.size, 'conversas')
        }
        this.invalidateCache('sort')
        this.debouncedSort()
      },
      { deep: false }
    )

    // ✅ WATCHER LIMPO - SEM MARCAÇÃO AUTOMÁTICA
    watch(
      this.conversaSelecionada,
      (nova, antiga) => {
        if (nova?.id !== antiga?.id) {
          console.log('ChatStore: Conversa selecionada:', nova?.id)
          // ✅ REMOVIDO: Não marca mais automaticamente como lida
          // Agora é responsabilidade do componente que usa o store
        }
      },
      { deep: false }
    )
  }

   setupGlobalListeners() {
    // Mensagens vindas do WebSocket (via websocket.js)
    window.addEventListener('chat-message-received', event => {
      const data = event.detail
      const message = data?.message || data
      const conversationId = data?.conversation_id || message?.conversation_id

      if (!conversationId || !message) return

      const convId = parseInt(conversationId)
      const isCurrentConversation =
        this.conversaSelecionada.value?.id === convId

      const currentUserId = window.websocketService?.getCurrentUser?.()?.id
      const isOwnMessage =
        currentUserId &&
        message?.sender_id &&
        parseInt(message.sender_id) === parseInt(currentUserId)

      this.updateConversationList(
        conversationId,
        message,
        !!isOwnMessage,
        !isCurrentConversation && !isOwnMessage
      )
    })

    // Digitação
    window.addEventListener('chat-user-typing', event => {
      const data = event.detail
      if (data.conversation_id && data.user_id) {
        this.updateTypingIndicator(
          data.conversation_id,
          data.user_id,
          data.user_name,
          data.is_typing
        )
      }
    })

    // Conversa lida (vinda de outros lugares)
    window.addEventListener('chat-conversation-read', event => {
      const data = event.detail
      if (data.conversation_id) {
        this.markConversationAsReadGlobal(data.conversation_id)
      }
    })

    // ✅ Conversa atualizada (meta, atribuições, status etc.)
    window.addEventListener('chat-conversation-updated', event => {
      const data = event.detail
      if (data?.conversation) {
        this.updateConversationMeta(data.conversation)
      }
    })

    // ✅ Nova conversa criada
    window.addEventListener('chat-new-conversation-created', event => {
      const data = event.detail
      if (data?.conversation) {
        this.addConversation(data.conversation)
      }
    })

    // ✅ NOVO LISTENER: Para mensagens WhatsApp externas (sem notificação)
    window.addEventListener('chat-outbound-message-received', event => {
      console.log('🔍 DEBUG: Evento chat-outbound-message-received recebido:', event.detail)
      
      const data = event.detail
      const message = data?.message || data
      const conversationId = data?.conversation_id || message?.conversation_id

      console.log('🔍 DEBUG: Dados extraídos:', {
        data,
        message,
        conversationId,
        messageId: message?.id,
        content: message?.content
      })

      if (!conversationId || !message) {
        console.warn('❌ ChatStore: Dados inválidos para mensagem externa:', {
          conversationId,
          message,
          hasConversationId: !!conversationId,
          hasMessage: !!message
        })
        return
      }

      console.log('✅ ChatStore: Chamando updateConversationList para mensagem externa:', {
        conversationId,
        messageId: message.id,
        content: message.content?.substring(0, 50)
      })

      // Atualizar a lista de conversas silenciosamente (sem notificação)
      this.updateConversationList(
        conversationId,
        message,
        true, // isOwnMessage = true
        false // allowNotificationCheck = false (SEM NOTIFICAÇÃO)
      )

      console.log('✅ ChatStore: Mensagem WhatsApp externa processada silenciosamente:', message.id)
    })

    // ✅ NOVOS LISTENERS: Para eventos de finalização/reabertura
    window.addEventListener('chat-conversation-finalized', event => {
      const data = event.detail
      if (data?.conversation_id) {
        this.handleConversationFinalized(data)
      }
    })

    window.addEventListener('chat-conversation-reopened', event => {
      const data = event.detail
      if (data?.conversation_id) {
        this.handleConversationReopened(data)
      }
    })

    window.addEventListener('chat-conversation-auto-reopened', event => {
      const data = event.detail
      if (data?.conversation_id) {
        this.handleConversationReopened({
          ...data,
          reason: 'Reabertura automática por nova mensagem'
        })
      }
    })

    window.addEventListener('chat-conversation-status-updated', event => {
      const data = event.detail
      if (data?.conversation_id) {
        this.updateConversationStatus(data.conversation_id, data.new_status, data)
      }
    })

    // Conexão WebSocket
    window.addEventListener('websocket-connected', () => {
      console.log('ChatStore: WebSocket conectado')
      if (this.loadInitialConversations) {
        setTimeout(() => this.loadInitialConversations(), 1000)
      }
    })

    window.addEventListener('websocket-disconnected', () => {
      console.log('ChatStore: WebSocket desconectado')
      this.usuariosDigitando.value = markRaw(new Map())
    })
  }

  // ========= NOTIFICAÇÃO (SOMENTE DECISÃO / EVENTOS) =========

  shouldEmitNotification(data) {
    if (!this.notificationsEnabled.value) return false

    const currentUserId = window.websocketService?.getCurrentUser()?.id
    const message = data.message || data

    // não notificar mensagens do próprio usuário
    if (message?.sender_id && currentUserId && message.sender_id === currentUserId) {
      return false
    }

    const conversationId = data.conversation_id || message?.conversation_id
    const conversation = this.conversas.value.find(
      c => c.id === parseInt(conversationId)
    )

    if (!conversation) return false

    // conversa silenciada
    if (conversation.is_muted) return false

    // se o documento está oculto (aba em segundo plano) → pode notificar
    if (document.hidden) return true

    // se é a conversa atual, não notificar
    const isCurrentConversation =
      this.conversaSelecionada.value?.id === parseInt(conversationId)

    return !isCurrentConversation
  }

  // ========= OPERAÇÕES DE CONVERSA =========

  updateConversationList(
    conversationId,
    message,
    isOwnMessage = false,
    allowNotificationCheck = true
  ) {
    try {
      console.log('🔍 DEBUG updateConversationList chamado:', {
        conversationId,
        messageId: message?.id,
        isOwnMessage,
        allowNotificationCheck,
        messageContent: message?.content?.substring(0, 50)
      })

      const convId = parseInt(conversationId)
      const index = this.conversas.value.findIndex(c => c.id === convId)

      console.log('🔍 DEBUG: Procurando conversa:', {
        convId,
        index,
        totalConversas: this.conversas.value.length,
        conversaEncontrada: index !== -1
      })

      if (index === -1) {
        console.warn('❌ ChatStore: Conversa não encontrada:', convId, 'Total conversas:', this.conversas.value.length)
        if (this.loadInitialConversations) {
          setTimeout(() => this.loadInitialConversations(), 1000)
        }
        return
      }

      const array = markRaw([...this.conversas.value])
      const conversa = { ...array[index] }

      const messageContent = message.content || message.message || ''
      const messageSender = message.sender_name || 'Usuário'
      const messageTime =
        message.sent_at || message.created_at || new Date().toISOString()

      conversa.last_message = messageContent
      conversa.last_message_at = messageTime
      conversa.last_message_sender = messageSender
      conversa.last_message_type = isOwnMessage ? 'outgoing' : 'incoming'
      conversa.updated_at = new Date().toISOString()

      if (!isOwnMessage && this.conversaSelecionada.value?.id !== convId) {
        conversa.unread_count = (conversa.unread_count || 0) + 1
      }

      array.splice(index, 1)
      array.unshift(conversa)
      this.conversas.value = array

      this.invalidateCache('sort')
      this.invalidateCache('filter')
      this.debouncedSort()

      this.metrics.updateCount++

      // disparar evento para o global-chat cuidar de UI/notificações
      if (!isOwnMessage && allowNotificationCheck && this.shouldEmitNotification({ conversation_id: convId, message })) {
        this.emitNewUnreadMessage(conversa, message)
      }

      this.emitStoreUpdated()

      // console.log('✅ ChatStore: Conversa atualizada com sucesso:', convId)
    } catch (error) {
      console.error('❌ ChatStore: Erro ao atualizar conversa:', error)
    }
  }

  // ✅ MÉTODO PARA MARCAR COMO LIDA (Agora apenas local, sem automático)
  markConversationAsReadGlobal(conversationId) {
    try {
      const convId = parseInt(conversationId)
      const index = this.conversas.value.findIndex(c => c.id === convId)

      if (index === -1) return

      const array = markRaw([...this.conversas.value])
      const conversa = { ...array[index] }

      if (conversa.unread_count > 0) {
        conversa.unread_count = 0
        conversa.read_at = new Date().toISOString()

        array[index] = conversa
        this.conversas.value = array
        this.invalidateCache('filter')
        this.debouncedUpdateTotal()
        this.emitStoreUpdated()

        // console.log('ChatStore: Conversa marcada como lida localmente:', convId)
      }
    } catch (error) {
      console.error('ChatStore: Erro ao marcar como lida:', error)
    }
  }

  // ✅ NOVO MÉTODO: Marcar como lida E enviar para backend
  async markConversationAsRead(conversationId, sendToBackend = true) {
    try {
      // Marcar localmente primeiro
      this.markConversationAsReadGlobal(conversationId)

      // Enviar para backend se solicitado
      if (sendToBackend) {
        await this.sendReadStatusToBackend(conversationId)
      }
    } catch (error) {
      console.error('ChatStore: Erro ao marcar como lida completo:', error)
    }
  }

  // ✅ NOVO MÉTODO: Enviar status para backend
  async sendReadStatusToBackend(conversationId) {
    try {
      const response = await fetch(`/api/chat/conversations/${conversationId}/mark-read`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content'),
          'Accept': 'application/json'
        }
      })

      if (!response.ok) {
        throw new Error(`HTTP ${response.status}`)
      }

      const data = await response.json()
      // console.log('✅ Status de leitura enviado com sucesso:', data)
      return data

    } catch (error) {
      console.error('❌ Erro ao enviar status de leitura:', error)
      throw error
    }
  }

  // ✅ NOVOS MÉTODOS: Para finalização e reabertura de conversas

  /**
   * ✅ ATUALIZAR STATUS DA CONVERSA (FINALIZAÇÃO/REABERTURA)
   */
  updateConversationStatus(conversationId, newStatus, statusData = {}) {
    try {
      const conversationIndex = this.conversas.value.findIndex(
        conv => conv.id === parseInt(conversationId)
      )

      if (conversationIndex !== -1) {
        const updatedConversations = [...this.conversas.value]
        const conversation = { ...updatedConversations[conversationIndex] }

        // ✅ ATUALIZAR STATUS E DADOS RELACIONADOS
        conversation.status = newStatus
        conversation.updated_at = new Date().toISOString()

        if (newStatus === 'closed' || newStatus === 'archived') {
          conversation.closed_at = statusData.closed_at || new Date().toISOString()
          conversation.closed_by = statusData.closed_by || null
          conversation.close_reason = statusData.close_reason || null
        } else if (newStatus === 'active') {
          conversation.closed_at = null
          conversation.closed_by = null
          conversation.close_reason = null
        }

        updatedConversations[conversationIndex] = conversation
        this.conversas.value = markRaw(updatedConversations)

        // ✅ ATUALIZAR CONVERSA ATIVA SE FOR A MESMA
        if (this.conversaSelecionada.value?.id === parseInt(conversationId)) {
          this.conversaSelecionada.value = conversation
        }

        this.invalidateCache('sort')
        this.invalidateCache('filter')
        this.debouncedSort()
        this.emitStoreUpdated()

        console.log(`✅ ChatStore: Status da conversa ${conversationId} atualizado para ${newStatus}`)
        return true
      }

      console.warn(`⚠️ ChatStore: Conversa ${conversationId} não encontrada para atualizar status`)
      return false

    } catch (error) {
      console.error('❌ ChatStore: Erro ao atualizar status da conversa:', error)
      return false
    }
  }

  /**
   * ✅ PROCESSAR EVENTO DE FINALIZAÇÃO
   */
  handleConversationFinalized(eventData) {
    try {
      const { conversation_id, reason, finalized_by, finalized_at, archived } = eventData

      console.log('🏁 ChatStore: Processando finalização de conversa:', {
        conversation_id,
        reason,
        archived
      })

      const finalStatus = archived ? 'archived' : 'closed'
      
      const statusData = {
        closed_at: finalized_at,
        closed_by: finalized_by,
        close_reason: reason
      }

      const updated = this.updateConversationStatus(conversation_id, finalStatus, statusData)

      if (updated) {
        // ✅ NOTIFICAR OUTROS COMPONENTES
        this.emitGlobalEvent('conversation-status-changed', {
          conversation_id,
          old_status: 'active',
          new_status: finalStatus,
          reason,
          finalized_by
        })

        // ✅ REMOVER DA LISTA SE ARQUIVADA (OPCIONAL)
        if (archived) {
          this.handleConversationArchived(conversation_id)
        }
      }

      return updated

    } catch (error) {
      console.error('❌ ChatStore: Erro ao processar finalização:', error)
      return false
    }
  }

  /**
   * ✅ PROCESSAR EVENTO DE REABERTURA
   */
  handleConversationReopened(eventData) {
    try {
      const { conversation_id, reopened_by, reopened_at, reason } = eventData

      console.log('🔄 ChatStore: Processando reabertura de conversa:', {
        conversation_id,
        reason
      })

      const statusData = {
        reopened_by,
        reopened_at,
        reopen_reason: reason
      }

      const updated = this.updateConversationStatus(conversation_id, 'active', statusData)

      if (updated) {
        // ✅ NOTIFICAR OUTROS COMPONENTES
        this.emitGlobalEvent('conversation-status-changed', {
          conversation_id,
          old_status: 'closed',
          new_status: 'active',
          reason,
          reopened_by
        })
      }

      return updated

    } catch (error) {
      console.error('❌ ChatStore: Erro ao processar reabertura:', error)
      return false
    }
  }

  /**
   * ✅ REMOVER CONVERSA ARQUIVADA DA LISTA (OPCIONAL)
   */
  handleConversationArchived(conversationId) {
    try {
      // Se a conversa foi arquivada, pode remover da lista principal
      const conversationIndex = this.conversas.value.findIndex(
        conv => conv.id === parseInt(conversationId)
      )

      if (conversationIndex !== -1) {
        const updatedConversations = [...this.conversas.value]
        updatedConversations.splice(conversationIndex, 1)
        this.conversas.value = markRaw(updatedConversations)

        // Se era a conversa ativa, limpar seleção
        if (this.conversaSelecionada.value?.id === parseInt(conversationId)) {
          this.conversaSelecionada.value = null
        }

        this.invalidateCache('all')
        this.debouncedUpdateTotal()
        this.emitStoreUpdated()

        console.log(`🗂️ ChatStore: Conversa ${conversationId} removida da lista (arquivada)`)
        return true
      }

      return false

    } catch (error) {
      console.error('❌ ChatStore: Erro ao arquivar conversa:', error)
      return false
    }
  }

  updateConversationMeta(data) {
    try {
      const convId = parseInt(data.id)
      const index = this.conversas.value.findIndex(c => c.id === convId)

      if (index === -1) return

      const array = markRaw([...this.conversas.value])
      const oldConversation = array[index]
      const newConversation = { ...oldConversation, ...data }

      const hasChanges = Object.keys(data).some(
        key => oldConversation[key] !== newConversation[key]
      )

      if (hasChanges) {
        array[index] = newConversation
        this.conversas.value = array

        this.invalidateCache('sort')
        this.invalidateCache('filter')
        this.debouncedSort()
        this.emitStoreUpdated()

        // console.log('ChatStore: Meta atualizada:', convId)
      }
    } catch (error) {
      console.error('ChatStore: Erro ao atualizar meta:', error)
    }
  }

  addConversation(conversa) {
    try {
      const convId = parseInt(conversa.id)
      const existe = this.conversas.value.find(c => c.id === convId)
      if (existe) return

      this.conversas.value = markRaw([conversa, ...this.conversas.value])
      this.invalidateCache('all')
      this.debouncedSort()
      this.emitStoreUpdated()

      // console.log('ChatStore: Conversa adicionada:', convId)
    } catch (error) {
      console.error('ChatStore: Erro ao adicionar conversa:', error)
    }
  }

  removeConversation(conversationId) {
    try {
      const convId = parseInt(conversationId)
      this.conversas.value = markRaw(
        this.conversas.value.filter(c => c.id !== convId)
      )
      this.invalidateCache('all')
      this.debouncedSort()
      this.debouncedUpdateTotal()
      this.emitStoreUpdated()
      // console.log('ChatStore: Conversa removida:', convId)
    } catch (error) {
      console.error('ChatStore: Erro ao remover conversa:', error)
    }
  }

  // ========= TYPING =========

  updateTypingIndicator(conversationId, userId, userName, isTyping) {
    try {
      const convId = parseInt(conversationId)
      const userIdNum = parseInt(userId)

      if (isNaN(convId) || isNaN(userIdNum)) return

      const map = new Map(this.usuariosDigitando.value)

      if (!map.has(convId)) {
        map.set(convId, { users: [], timestamp: Date.now() })
      }

      const typing = map.get(convId)

      if (isTyping) {
        const exists = typing.users.some(u => parseInt(u.id) === userIdNum)
        if (!exists) {
          typing.users.push({
            id: userIdNum,
            name: userName || 'Usuário',
            startedAt: Date.now(),
          })
          typing.timestamp = Date.now()
        }
      } else {
        const oldLength = typing.users.length
        typing.users = typing.users.filter(u => parseInt(u.id) !== userIdNum)

        if (typing.users.length === 0) {
          map.delete(convId)
        } else if (oldLength !== typing.users.length) {
          typing.timestamp = Date.now()
        }
      }

      this.usuariosDigitando.value = markRaw(map)
      this.invalidateCache('sort')
      this.debouncedSort()
      this.emitStoreUpdated()
    } catch (error) {
      console.error('ChatStore: Erro ao atualizar digitação:', error)
    }
  }

  clearTypingIndicator(conversationId, userId = null) {
    try {
      const convId = parseInt(conversationId)
      const map = new Map(this.usuariosDigitando.value)

      if (userId) {
        const userIdNum = parseInt(userId)
        const typing = map.get(convId)
        if (typing) {
          typing.users = typing.users.filter(u => parseInt(u.id) !== userIdNum)
          if (typing.users.length === 0) {
            map.delete(convId)
          } else {
            typing.timestamp = Date.now()
          }
        }
      } else {
        map.delete(convId)
      }

      this.usuariosDigitando.value = markRaw(map)
      this.invalidateCache('sort')
      this.debouncedSort()
      this.emitStoreUpdated()
    } catch (error) {
      console.error('ChatStore: Erro ao limpar digitação:', error)
    }
  }

  startTypingCleanup() {
    this.typingCleanupTimer = setInterval(() => {
      try {
        const now = Date.now()
        const map = new Map(this.usuariosDigitando.value)
        let changed = false

        for (const [convId, typing] of map.entries()) {
          const valid = typing.users.filter(
            user => now - (user.startedAt || typing.timestamp) < 5000
          )

          if (valid.length !== typing.users.length) {
            if (valid.length === 0) {
              map.delete(convId)
            } else {
              typing.users = valid
              typing.timestamp = now
            }
            changed = true
          }
        }

        if (changed) {
          this.usuariosDigitando.value = markRaw(map)
          this.invalidateCache('sort')
          this.emitStoreUpdated()
        }
      } catch (error) {
        console.error('ChatStore: Erro na limpeza de digitação:', error)
      }
    }, this.config.typingCleanupInterval)
  }

  // ========= ORDENAÇÃO / FILTROS =========

  sortConversations() {
    try {
      const current = this.conversas.value
      if (!current || current.length === 0) return

      const currentHash = this.generateSortHash(current)
      if (this.cache.lastSortHash === currentHash && this.cache.sortedConversations) {
        this.metrics.cacheHits++
        return
      }

      this.metrics.cacheMisses++
      this.metrics.sortCount++

      const sorted = [...current].sort((a, b) => {
        const aTyping = this.usuariosDigitando.value.has(a.id)
        const bTyping = this.usuariosDigitando.value.has(b.id)

        if (aTyping !== bTyping) return aTyping ? -1 : 1

        if (a.is_pinned !== b.is_pinned) return a.is_pinned ? -1 : 1

        const aUnread = (a.unread_count || 0) > 0
        const bUnread = (b.unread_count || 0) > 0
        if (aUnread !== bUnread) return aUnread ? -1 : 1

        const dateA = new Date(
          a.last_message_at || a.updated_at || a.created_at
        )
        const dateB = new Date(
          b.last_message_at || b.updated_at || b.created_at
        )
        return dateB.getTime() - dateA.getTime()
      })

      this.cache.sortedConversations = markRaw(sorted)
      this.cache.lastSortHash = currentHash

      const currentIds = current.map(c => c.id).join(',')
      const sortedIds = sorted.map(c => c.id).join(',')

      if (currentIds !== sortedIds) {
        this.conversas.value = markRaw(sorted)
        this.emitStoreUpdated()
      }
    } catch (error) {
      console.error('ChatStore: Erro ao ordenar:', error)
    }
  }

  getFilteredConversations(filter = 'todas') {
    try {
      const conversations = this.conversas.value
      if (!conversations || conversations.length === 0) return []

      const cacheKey = this.generateFilterHash(filter, conversations)
      const cached = this.cache.filteredConversations.get(cacheKey)

      if (cached) {
        this.metrics.cacheHits++
        return cached
      }

      this.metrics.cacheMisses++

      let filtered = [...conversations]

      switch (filter) {
        case 'nao-lidas':
          filtered = filtered.filter(c => (c.unread_count || 0) > 0)
          break
        case 'internas':
          filtered = filtered.filter(c => c.type === 'internal')
          break
        case 'externas':
          filtered = filtered.filter(c => c.type !== 'internal')
          break
        case 'digitando':
          filtered = filtered.filter(c => this.usuariosDigitando.value.has(c.id))
          break
        case 'fixadas':
          filtered = filtered.filter(c => c.is_pinned)
          break
        case 'silenciadas':
          filtered = filtered.filter(c => c.is_muted)
          break
        case 'fechadas':
          filtered = filtered.filter(c => c.status === 'closed')
          break
        case 'arquivadas':
          filtered = filtered.filter(c => c.status === 'archived')
          break
      }

      const result = markRaw(filtered)
      this.cache.filteredConversations.set(cacheKey, result)

      return result
    } catch (error) {
      console.error('ChatStore: Erro ao filtrar conversas:', error)
      return this.conversas.value || []
    }
  }

  // ========= UNREAD TOTAL =========

  updateUnreadTotal() {
    try {
      const total = this.conversas.value.reduce(
        (sum, conv) => sum + (conv.unread_count || 0),
        0
      )

      if (this.unreadTotal.value !== total) {
        this.unreadTotal.value = total
        this.emitStoreUpdated()
      }
    } catch (error) {
      console.error('ChatStore: Erro ao atualizar total:', error)
    }
  }

  // ========= BUSCA =========

  selectConversation(conversationId) {
    try {
      const convId = parseInt(conversationId)
      const conversa = this.conversas.value.find(c => c.id === convId)
      if (conversa) {
        this.conversaSelecionada.value = conversa
        this.emitStoreUpdated()
        // console.log('ChatStore: Conversa selecionada:', convId)
      }
    } catch (error) {
      console.error('ChatStore: Erro ao selecionar conversa:', error)
    }
  }

  searchConversations(query, options = {}) {
    try {
      if (!query || query.length < 2) return this.conversas.value

      const {
        searchFields = ['contact_name', 'subject', 'last_message', 'phone'],
        caseSensitive = false,
        maxResults = 50,
      } = options

      const searchTerm = caseSensitive ? query : query.toLowerCase()

      const results = this.conversas.value
        .filter(conversa =>
          searchFields.some(field => {
            const value = conversa[field]
            if (!value) return false
            const searchValue = caseSensitive ? value : value.toLowerCase()
            return searchValue.includes(searchTerm)
          })
        )
        .slice(0, maxResults)

      return markRaw(results)
    } catch (error) {
      console.error('ChatStore: Erro na busca:', error)
      return []
    }
  }

  // ========= DEBOUNCE =========

  debouncedSort() {
    if (this.sortTimer) clearTimeout(this.sortTimer)
    this.sortTimer = setTimeout(() => {
      this.sortConversations()
      this.sortTimer = null
    }, this.config.debounceSort)
  }

  debouncedUpdateTotal() {
    if (this.updateTotalTimer) clearTimeout(this.updateTotalTimer)
    this.updateTotalTimer = setTimeout(() => {
      this.updateUnreadTotal()
      this.updateTotalTimer = null
    }, this.config.debounceUpdateTotal)
  }

  // ========= STATS / CONFIG =========

  getStats() {
    const conversations = this.conversas.value
    const cacheHitRate =
      (this.metrics.cacheHits /
        (this.metrics.cacheHits + this.metrics.cacheMisses || 1)) *
      100

    return {
      totalConversas: conversations.length,
      naoLidas: conversations.filter(c => (c.unread_count || 0) > 0).length,
      unreadTotal: this.unreadTotal.value,
      abertas: conversations.filter(c =>
        ['open', 'active', 'in_progress'].includes(c.status)
      ).length,
      pendentes: conversations.filter(c => c.status === 'pending').length,
      fechadas: conversations.filter(c => c.status === 'closed').length,
      arquivadas: conversations.filter(c => c.status === 'archived').length,
      internas: conversations.filter(c => c.type === 'internal').length,
      externas: conversations.filter(c => c.type !== 'internal').length,
      fixadas: conversations.filter(c => c.is_pinned).length,
      silenciadas: conversations.filter(c => c.is_muted).length,
      conversaSelecionada: this.conversaSelecionada.value?.id || null,
      digitando: this.usuariosDigitando.value.size,
      conversasComDigitacao: Array.from(this.usuariosDigitando.value.keys()),
      lastUpdate: this.lastUpdate.value,
      isLoading: this.isLoading.value,
      notificationsEnabled: this.notificationsEnabled.value,
      soundEnabled: this.soundEnabled.value,
      performance: {
        sortCount: this.metrics.sortCount,
        updateCount: this.metrics.updateCount,
        cacheHits: this.metrics.cacheHits,
        cacheMisses: this.metrics.cacheMisses,
        cacheHitRate: isNaN(cacheHitRate)
          ? '0%'
          : cacheHitRate.toFixed(2) + '%',
        totalOperations: this.metrics.cacheHits + this.metrics.cacheMisses,
        uptime:
          Math.round((Date.now() - this.metrics.lastResetTime) / 1000) + 's',
      },
      cache: {
        sortCached: !!this.cache.sortedConversations,
        filterCacheSize: this.cache.filteredConversations.size,
      },
    }
  }

  updateConfig(newConfig) {
    Object.assign(this.config, newConfig)
    // console.log('ChatStore: Configuração atualizada:', this.config)
  }

  // ✅ MÉTODOS PÚBLICOS PARA USO EXTERNO
  
  // Marcar como lida (apenas local)
  markAsReadLocal(conversationId) {
    this.markConversationAsReadGlobal(conversationId)
  }

  // Marcar como lida (local + backend)
  async markAsRead(conversationId) {
    return await this.markConversationAsRead(conversationId, true)
  }

  // Obter conversa por ID
  getConversationById(conversationId) {
    return this.conversas.value.find(c => c.id === parseInt(conversationId))
  }

  // Verificar se conversa está selecionada
  isConversationSelected(conversationId) {
    return this.conversaSelecionada.value?.id === parseInt(conversationId)
  }

  // Obter total de não lidas
  getTotalUnread() {
    return this.unreadTotal.value
  }

  // Verificar se tem conversas não lidas
  hasUnreadConversations() {
    return this.unreadTotal.value > 0
  }
}

const chatStore = new ChatStore()

window.chatStore = chatStore
window.cleanupChatStore = () => chatStore.destroy()
window.debugChatStore = () => {
  // console.log('=== CHAT STORE DEBUG ===')
  // console.log('Stats:', chatStore.getStats())
}

export default chatStore