// modules/Chat/resources/js/servicos/websocket.js

import { io } from 'socket.io-client'

const config = {
  url: null,
  debug: false,
  reconnectInterval: 5000,
  maxReconnectAttempts: 5,
  heartbeatInterval: 25000,
  connectionTimeout: 15000,
}

class WebSocketState {
  constructor() {
    this.reset()
  }

  reset() {
    this.socket = null
    this.isConnected = false
    this.isInitialized = false
    this.isDestroyed = false
    this.connectionState = 'disconnected'
    this.currentUser = null
    this.reconnectAttempts = 0
    this.lastError = null
    this.messageQueue = []
    this.heartbeatTimer = null
    this.reconnectTimer = null
    this.eventListeners = []
    this.eventCount = 0
    this.socketId = null
  }
}

const state = new WebSocketState()

function log(level, ...args) {
  if (!config.debug && level !== 'error') return
  const prefix = `[WebSocket] - ${level.toUpperCase()}:`
  console[level](prefix, ...args)
}

function emitGlobalEvent(eventName, data) {
  try {
    window.dispatchEvent(new CustomEvent(eventName, { detail: data }))
  } catch (e) {
    log('error', `Erro ao emitir evento ${eventName}:`, e)
  }
}

function getCurrentUser() {
  const chatConfig = window.Innoclapps?.scriptConfig('chat')

  if (chatConfig?.user_id) {
    return {
      id: parseInt(chatConfig.user_id),
      name: chatConfig.user_name || 'Usuário',
      email: chatConfig.user_email || null,
      super_admin: chatConfig.is_super_admin || false,
    }
  }

  return null
}

function getWebSocketUrl() {
  const chatConfig = window.Innoclapps?.scriptConfig('chat')

  return (
    chatConfig?.websocket_url?.replace(/\/socket\.io\/?$/, '') ||
    `${window.location.protocol}//${window.location.host}`
  )
}

// ✅ Normaliza payload vindo do servidor:
// - Objeto puro
// - Array [eventName, payload]
// - Array com algum objeto dentro
function normalizeWsPayload(raw) {
  if (!raw) return {}

  // Já é objeto -> usa direto
  if (typeof raw === 'object' && !Array.isArray(raw)) {
    return raw
  }

  if (Array.isArray(raw)) {
    // Caso típico: ["new_whatsapp_message", { ... }]
    if (raw.length === 2 && typeof raw[1] === 'object') {
      return raw[1] || {}
    }

    // Caso: [ { ... } ]
    if (raw.length === 1 && typeof raw[0] === 'object') {
      return raw[0] || {}
    }

    // Fallback: pega o primeiro objeto do array
    const found = raw.find(item => item && typeof item === 'object')
    if (found) return found
  }

  return {}
}

function connect() {
  return new Promise((resolve, reject) => {
    if (state.isConnected) return resolve(true)
    if (state.isDestroyed) return reject(new Error('Serviço destruído.'))

    state.connectionState = 'connecting'
    config.url = config.url || getWebSocketUrl()
    log('info', 'Conectando a:', config.url)

    const socket = io(config.url, {
      transports: ['websocket'],
      timeout: config.connectionTimeout,
      forceNew: true,
      path: '/socket.io',
      reconnection: false,
      autoConnect: false,
    })

    state.socket = socket

    const connectionTimeout = setTimeout(() => {
      socket.disconnect()
      reject(new Error('Timeout na conexão'))
    }, config.connectionTimeout)

    socket.on('connect', () => {
      clearTimeout(connectionTimeout)
      log('info', '✅ Conectado com ID:', socket.id)

      state.isConnected = true
      state.connectionState = 'connected'
      state.reconnectAttempts = 0
      state.socketId = socket.id

      authenticateUser()
      setupEventListeners()
      startHeartbeat()
      processMessageQueue()

      emitGlobalEvent('websocket-connected', { socketId: socket.id })
      resolve(true)
    })

    socket.on('connect_error', error => {
      clearTimeout(connectionTimeout)
      log('error', 'Erro de conexão:', error.message)
      state.lastError = error.message
      state.connectionState = 'error'
      reject(error)
    })

    socket.on('disconnect', reason => {
      log('warn', '🚪 Desconectado:', reason)

      state.isConnected = false
      state.connectionState = 'disconnected'
      state.socketId = null

      if (state.heartbeatTimer) {
        clearInterval(state.heartbeatTimer)
        state.heartbeatTimer = null
      }

      emitGlobalEvent('websocket-disconnected', { reason })

      if (!state.isDestroyed && reason !== 'io client disconnect') {
        scheduleReconnect()
      }
    })

    socket.connect()
  })
}

function setupEventListeners() {
  if (!state.socket || state.isDestroyed) return

  // Limpa listeners antigos
  if (state.eventListeners.length > 0) {
    state.eventListeners.forEach(({ event, handler }) =>
      state.socket.off(event, handler)
    )
    state.eventListeners = []
  }

  const addListener = (event, handler) => {
    state.socket.on(event, (...args) => {
      state.eventCount++
      // Se vier um único arg, passa direto, se vier mais, manda o array
      handler(args.length === 1 ? args[0] : args)
    })
    state.eventListeners.push({ event, handler })
  }

  // ✅ EVENTOS PRINCIPAIS - MENSAGENS
  addListener('new_whatsapp_message', data => {
    const payload = normalizeWsPayload(data)
    log(
      'debug',
      '📨 Nova mensagem WhatsApp recebida:',
      payload?.message?.id || 'unknown'
    )
    emitGlobalEvent('chat-message-received', payload)
  })

  addListener('new_message', data => {
    const payload = normalizeWsPayload(data)
    log(
      'debug',
      '📨 Nova mensagem interna recebida:',
      payload?.message?.id || 'unknown'
    )
    emitGlobalEvent('chat-message-received', payload)
  })

  // ✅ NOVO EVENTO: mensagens WhatsApp externas (sem notificação)
  addListener('outbound_whatsapp_message', data => {
    const payload = normalizeWsPayload(data)
    log('debug', '📤 Mensagem WhatsApp externa recebida (sem notificação):', payload?.message?.id || 'unknown')
    
    // Emitir evento específico para atualização silenciosa
    emitGlobalEvent('chat-outbound-message-received', payload)
  })

  addListener('chat_message_received', data => {
    const payload = normalizeWsPayload(data)
    log(
      'debug',
      '📨 Mensagem de chat recebida:',
      payload?.message?.id || 'unknown'
    )
    emitGlobalEvent('chat-message-received', payload)
  })

  // ✅ EVENTOS DE STATUS
  addListener('whatsapp_status_update', data => {
    const payload = normalizeWsPayload(data)
    log('debug', '📊 Status update recebido:', {
      message_id: payload?.message_id,
      status_change: `${payload?.old_status} → ${payload?.new_status}`,
    })
    emitGlobalEvent('chat-message-status-updated', payload)
  })

  // ✅ EVENTOS DE DIGITAÇÃO
  addListener('user.typing_started', data => {
    const payload = normalizeWsPayload(data)
    log('debug', '⌨️ Usuário começou a digitar:', payload?.user_name || 'unknown')
    emitGlobalEvent('chat-user-typing', { ...payload, is_typing: true })
  })

  addListener('user.typing_stopped', data => {
    const payload = normalizeWsPayload(data)
    log('debug', '⌨️ Usuário parou de digitar:', payload?.user_name || 'unknown')
    emitGlobalEvent('chat-user-typing', { ...payload, is_typing: false })
  })

  // ✅ EVENTOS DE CONVERSA
  addListener('chat.conversation.read', data => {
    const payload = normalizeWsPayload(data)
    log('debug', '👁️ Conversa marcada como lida:', payload?.conversation_id)
    emitGlobalEvent('chat-conversation-read', payload)
  })

  addListener('chat.conversation.updated', data => {
    const payload = normalizeWsPayload(data)
    log('debug', '🔄 Conversa atualizada:', payload?.conversation_id)
    emitGlobalEvent('chat-conversation-updated', payload)
  })

  addListener('new_conversation_created', data => {
    const payload = normalizeWsPayload(data)
    log('debug', '🆕 Nova conversa criada:', payload?.conversation_id)
    emitGlobalEvent('chat-new-conversation-created', payload)
  })

  // ✅ NOVOS EVENTOS: FINALIZAÇÃO E REABERTURA
  addListener('conversation_finalized', data => {
    const payload = normalizeWsPayload(data)
    log('debug', '🏁 Conversa finalizada recebida:', payload?.conversation_id)
    emitGlobalEvent('chat-conversation-finalized', payload)
  })

  addListener('conversation_reopened', data => {
    const payload = normalizeWsPayload(data)
    log('debug', '🔄 Conversa reaberta recebida:', payload?.conversation_id)
    emitGlobalEvent('chat-conversation-reopened', payload)
  })

  addListener('conversation_auto_reopened', data => {
    const payload = normalizeWsPayload(data)
    log('debug', '🔄 Conversa reaberta automaticamente:', payload?.conversation_id)
    emitGlobalEvent('chat-conversation-auto-reopened', payload)
  })

  addListener('conversation_status_updated', data => {
    const payload = normalizeWsPayload(data)
    log('debug', '📊 Status de conversa atualizado:', {
      conversation_id: payload?.conversation_id,
      status_change: `${payload?.old_status} → ${payload?.new_status}`
    })
    emitGlobalEvent('chat-conversation-status-updated', payload)
  })

  // ✅ EVENTOS INTERNOS
  addListener('conversation_access_denied', data => {
    const payload = normalizeWsPayload(data)
    log('warn', '🚫 Acesso à conversa negado:', payload?.conversation_id)
    emitGlobalEvent('chat-access-denied', payload)
  })

  addListener('auth_success', data => {
    log('info', '🔐 Autenticado com sucesso.', data)
  })

  addListener('auth_error', data => {
    log('error', '❌ Falha na autenticação:', data)
  })

  addListener('error', data => {
    log('error', '❌ Erro do servidor WebSocket:', data)
  })

  addListener('pong', () => {
    log('debug', '🏓 Pong recebido.')
  })

  log('info', `✅ ${state.eventListeners.length} event listeners configurados`)
}

function authenticateUser() {
  if (!state.socket?.connected) return

  state.currentUser = getCurrentUser()

  if (state.currentUser) {
    log('info', 'Autenticando usuário:', state.currentUser.name)

    state.socket.emit('auth', {
      userId: state.currentUser.id,
      userName: state.currentUser.name,
      userEmail: state.currentUser.email,
    })
  }
}

function startHeartbeat() {
  if (state.heartbeatTimer) {
    clearInterval(state.heartbeatTimer)
    state.heartbeatTimer = null
  }

  state.heartbeatTimer = setInterval(() => {
    if (state.isConnected && state.socket?.connected) {
      state.socket.emit('ping', { timestamp: new Date().toISOString() })
    }
  }, config.heartbeatInterval)
}

function processMessageQueue() {
  if (!state.isConnected || state.messageQueue.length === 0) return

  log('info', `📤 Processando fila de ${state.messageQueue.length} eventos.`)

  while (state.messageQueue.length > 0) {
    const { event, data } = state.messageQueue.shift()
    sendEvent(event, data)
  }
}

function scheduleReconnect() {
  if (state.reconnectAttempts >= config.maxReconnectAttempts || state.isDestroyed) {
    log('error', 'Máximo de tentativas de reconexão atingido. Desistindo.')
    state.connectionState = 'error'
    return
  }

  state.reconnectAttempts++
  const delay = Math.min(config.reconnectInterval * state.reconnectAttempts, 30000)

  log('warn', `🔄 Tentando reconectar em ${delay / 1000}s (tentativa ${state.reconnectAttempts})`)

  if (state.reconnectTimer) {
    clearTimeout(state.reconnectTimer)
  }

  state.reconnectTimer = setTimeout(() => {
    if (!state.isDestroyed) {
      connect().catch(() => {
        // erro já logado no connect()
      })
    }
  }, delay)
}

function sendEvent(eventName, data = {}) {
  if (state.isDestroyed || !state.isInitialized) {
    log('warn', `Serviço não inicializado/destruído. Evento '${eventName}' não enviado.`)
    return false
  }

  if (!state.isConnected || !state.socket?.connected) {
    log('warn', `Socket desconectado. Evento '${eventName}' enfileirado.`)
    state.messageQueue.push({ event: eventName, data })
    return false
  }

  state.socket.emit(eventName, data)
  log('debug', `⬆️ Evento '${eventName}' enviado.`, data)
  return true
}

const websocketService = {
  async init(options = {}) {
    if (state.isInitialized) return state.isConnected

    Object.assign(config, options)
    state.isInitialized = true
    state.isDestroyed = false

    return await connect()
  },

  destroy() {
    if (state.isDestroyed) return

    log('info', '💀 Destruindo serviço WebSocket...')

    state.isDestroyed = true

    if (state.reconnectTimer) {
      clearTimeout(state.reconnectTimer)
      state.reconnectTimer = null
    }

    if (state.heartbeatTimer) {
      clearInterval(state.heartbeatTimer)
      state.heartbeatTimer = null
    }

    if (state.socket) {
      state.socket.disconnect()
    }

    state.reset()
  },

  get isConnected() {
    return state.isConnected
  },

  get isInitialized() {
    return state.isInitialized
  },

  isValid() {
    return state.isInitialized && state.isConnected && !state.isDestroyed
  },

  getStatus() {
    return {
      isConnected: state.isConnected,
      isInitialized: state.isInitialized,
      connectionState: state.connectionState,
      currentUser: state.currentUser,
      reconnectAttempts: state.reconnectAttempts,
      maxReconnectAttempts: config.maxReconnectAttempts,
      lastError: state.lastError,
      eventCount: state.eventCount,
      socketId: state.socketId,
      queueSize: state.messageQueue.length,
      listenersCount: state.eventListeners.length,
    }
  },

  getCurrentUser: () => state.currentUser,

  sendEvent,

  joinConversation: conversationId =>
    sendEvent('join_conversation', { conversation_id: conversationId }),

  leaveConversation: conversationId =>
    sendEvent('leave_conversation', { conversation_id: conversationId }),

  sendTypingIndicator: (conversationId, isTyping) =>
    sendEvent(isTyping ? 'typing_start' : 'typing_stop', {
      conversation_id: conversationId,
    }),

  // ✅ NOVOS MÉTODOS: Para finalização e reabertura
  
  /**
   * ✅ ENVIAR EVENTO DE FINALIZAÇÃO
   */
  sendConversationFinalized(conversationId, data = {}) {
    if (!websocketService.isConnected) {
      log('warn', '⚠️ WebSocket: Não conectado, não é possível enviar evento de finalização')
      return false
    }

    try {
      const eventData = {
        conversation_id: conversationId,
        finalized_by: state.currentUser?.id,
        finalized_at: new Date().toISOString(),
        ...data
      }

      const sent = sendEvent('conversation_finalized', eventData)
      if (sent) {
        log('debug', '📤 WebSocket: Evento de finalização enviado:', eventData)
      }
      return sent

    } catch (error) {
      log('error', '❌ WebSocket: Erro ao enviar finalização:', error)
      return false
    }
  },

  /**
   * ✅ ENVIAR EVENTO DE REABERTURA
   */
  sendConversationReopened(conversationId, reason = null) {
    if (!websocketService.isConnected) {
      log('warn', '⚠️ WebSocket: Não conectado, não é possível enviar evento de reabertura')
      return false
    }

    try {
      const eventData = {
        conversation_id: conversationId,
        reopened_by: state.currentUser?.id,
        reopened_at: new Date().toISOString(),
        reason: reason || 'Reabertura solicitada'
      }

      const sent = sendEvent('conversation_reopened', eventData)
      if (sent) {
        log('debug', '📤 WebSocket: Evento de reabertura enviado:', eventData)
      }
      return sent

    } catch (error) {
      log('error', '❌ WebSocket: Erro ao enviar reabertura:', error)
      return false
    }
  },

  /**
   * ✅ ENVIAR EVENTO DE STATUS ATUALIZADO
   */
  sendConversationStatusUpdated(conversationId, oldStatus, newStatus, data = {}) {
    if (!websocketService.isConnected) {
      log('warn', '⚠️ WebSocket: Não conectado, não é possível enviar evento de status')
      return false
    }

    try {
      const eventData = {
        conversation_id: conversationId,
        old_status: oldStatus,
        new_status: newStatus,
        updated_by: state.currentUser?.id,
        updated_at: new Date().toISOString(),
        ...data
      }

      const sent = sendEvent('conversation_status_updated', eventData)
      if (sent) {
        log('debug', '📤 WebSocket: Evento de status enviado:', eventData)
      }
      return sent

    } catch (error) {
      log('error', '❌ WebSocket: Erro ao enviar status:', error)
      return false
    }
  },

  async restart() {
    log('info', '🔁 Reiniciando serviço WebSocket...')
    websocketService.destroy()
    return await websocketService.init(config)
  },

  enableDebug() {
    config.debug = true
    log('info', '🐛 Debug habilitado')
  },

  disableDebug() {
    config.debug = false
    console.info('[WebSocket] - INFO: 🔇 Debug desabilitado')
  },

  getEventListeners() {
    return state.eventListeners.map(({ event }) => event)
  },

  clearMessageQueue() {
    const queueSize = state.messageQueue.length
    state.messageQueue = []
    log('info', `🗑️ Fila de mensagens limpa (${queueSize} eventos removidos)`)
  },
}

export default websocketService