// src/utils/urlHelpers.js

/**
 * Garante que uma URL seja absoluta. Se o path for relativo,
 * ele será resolvido contra o origin da janela.
 * @param {string|null|undefined} path O caminho da URL (pode ser relativo ou absoluto).
 * @returns {string|null} A URL absoluta ou null se o path for inválido.
 */
export function ensureAbsoluteUrl(path) {
  if (!path) return null;
  if (path.startsWith('http://') || path.startsWith('https://')) return path; // Já é absoluto

  // Constrói uma URL absoluta usando o origin atual do navegador
  const origin = window.location.origin.replace(/\/$/, ''); // Remove barra final do origin

  // Garante que o path tenha uma barra inicial para ser resolvido corretamente como um path absoluto
  const cleanPath = path.startsWith('/') ? path : `/${path}`;
  
  try {
    const url = new URL(cleanPath, origin);
    return url.toString();
  } catch (e) {
    console.warn('⚠️ urlHelpers: Erro ao construir URL absoluta:', path, e);
    // Em caso de erro na construção, retorna o path original ou null
    return path; 
  }
}

/**
 * Verifica se uma string é uma URL de imagem válida.
 * Rejeita URLs que claramente apontam para páginas do app (para evitar avatares quebrados
 * que são, na verdade, rotas internas).
 * @param {string|null|undefined} url A URL a ser validada.
 * @returns {boolean} True se for uma URL de imagem válida, false caso contrário.
 */
export function isValidImageUrl(rawUrl) {
  if (!rawUrl) return false

  try {
    const url = new URL(rawUrl, window.location.origin)

    // 1) Aceitar sempre Gravatar (identicon)
    if (url.hostname.includes('gravatar.com')) {
      return true
    }

    // 2) Aceitar http/https em geral
    if (!['http:', 'https:'].includes(url.protocol)) {
      return false
    }

    // 3) Extensões comuns de imagem – se tiver extensão, valida
    const path = url.pathname.toLowerCase()
    const hasKnownExt = ['.png', '.jpg', '.jpeg', '.webp', '.gif'].some(ext =>
      path.endsWith(ext)
    )

    if (hasKnownExt) {
      return true
    }

    // 4) Se não tiver extensão (caso de alguns CDNs), mas for http(s),
    // deixa passar para avatar (não loga como inválido)
    return true
  } catch (e) {
    return false
  }
}
