<?php

namespace Modules\Chat\Http\Controllers\Api;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Modules\Core\Http\Controllers\ApiController;
use Modules\Chat\Models\Conversation;
use Modules\Chat\Models\Message;
use Modules\Chat\Services\ChatRedisService;
use Modules\Users\Models\User;
use Modules\Chat\Http\Traits\FormatsConversations;

class ConversationAssignmentController extends ApiController
{
    use FormatsConversations;

    public function __construct(
        protected ChatRedisService $chatRedis
    ) {}

    /**
     * ✅ ATRIBUIÇÃO SIMPLES (COMPATIBILIDADE COM FRONTEND)
     */
    public function assignConversation(Request $request, Conversation $conversation): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'nullable|integer|exists:users,id',
            'reason'  => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return $this->response(['errors' => $validator->errors()], 422);
        }

        try {
            $currentUserId = auth()->id();

            // Trata "" como null (sem usuário)
            $rawUserId = $request->input('user_id');
            $newUserId = $rawUserId === '' ? null : $rawUserId;
            $oldUserId = $conversation->assigned_user_id;

            if (!$this->canManageAssignments($conversation, $currentUserId)) {
                return $this->response([
                    'error'        => 'Sem permissão para gerenciar atribuições',
                    'notification' => [
                        'type'    => 'error',
                        'title'   => 'Erro',
                        'message' => 'Você não tem permissão para reatribuir esta conversa.',
                    ],
                ], 403);
            }

            DB::beginTransaction();

            // Atualiza assigned_user_id (pode ser null)
            $conversation->update(['assigned_user_id' => $newUserId]);

            if ($newUserId) {
                // Sincroniza pivot com apenas o novo usuário
                $conversation->assignedUsers()->sync([
                    $newUserId => [
                        'role'        => 'primary',
                        'assigned_at' => now(),
                        'assigned_by' => $currentUserId,
                        'is_primary'  => true,
                        'created_at'  => now(),
                        'updated_at'  => now(),
                    ],
                ]);
            } else {
                // Sem usuário => remove todas as atribuições
                $conversation->assignedUsers()->detach();
            }

            $this->createSimpleAssignmentMessage(
                $conversation,
                $oldUserId,
                $newUserId,
                $currentUserId,
                $request->input('reason')
            );

            DB::commit();

            $this->chatRedis->publishConversationUpdate(
                $conversation->fresh(['assignedUsers', 'assignedUser']),
                'assignment_updated'
            );

            $assignedUser = $newUserId ? User::find($newUserId) : null;
            $message      = $this->getAssignmentMessage($oldUserId, $newUserId, $assignedUser);

            return $this->response([
                'success'      => true,
                'conversation' => $this->formatConversation($conversation->fresh(), auth()->id()),
                'assigned_user' => $assignedUser
                    ? ['id' => $assignedUser->id, 'name' => $assignedUser->name]
                    : null,
                'notification' => [
                    'type'    => 'success',
                    'title'   => 'Sucesso',
                    'message' => $message,
                ],
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('❌ Erro ao atribuir conversa: ' . $e->getMessage());
            return $this->response([
                'error'        => 'Erro interno do servidor',
                'notification' => [
                    'type'    => 'error',
                    'title'   => 'Erro',
                    'message' => 'Falha ao atribuir conversa.',
                ],
            ], 500);
        }
    }

    /**
     * ✅ ATRIBUIR MÚLTIPLOS USUÁRIOS
     */
    public function assignMultipleUsers(Request $request, Conversation $conversation): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'user_ids'     => 'required|array|min:0|max:10',
            'user_ids.*'   => 'integer|exists:users,id',
            'roles'        => 'nullable|array',
            'roles.*'      => 'string|in:primary,secondary,observer',
            'notify_users' => 'boolean',
            'reason'       => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return $this->response(['errors' => $validator->errors()], 422);
        }

        try {
            $currentUserId = auth()->id();
            $userIds       = array_values(array_unique($request->input('user_ids', [])));
            $roles         = $request->input('roles', []);
            $notifyUsers   = $request->boolean('notify_users', true);

            if (!$this->canManageAssignments($conversation, $currentUserId)) {
                return $this->response([
                    'error'        => 'Sem permissão para gerenciar atribuições',
                    'notification' => [
                        'type'    => 'error',
                        'title'   => 'Erro',
                        'message' => 'Você não tem permissão para gerenciar esta conversa.',
                    ],
                ], 403);
            }

            DB::beginTransaction();

            if (empty($userIds)) {
                // Remover todas as atribuições
                $conversation->assignedUsers()->detach();
                $conversation->update(['assigned_user_id' => null]);

                $this->createRemovalMessage($conversation, 'todas as atribuições', $currentUserId);

                DB::commit();

                return $this->response([
                    'success'       => true,
                    'assigned_users'=> [],
                    'conversation'  => $this->formatConversation($conversation->fresh(), auth()->id()),
                    'notification'  => [
                        'type'    => 'success',
                        'title'   => 'Sucesso',
                        'message' => 'Todas as atribuições foram removidas.',
                    ],
                ]);
            }

            $assignmentData = [];
            $primaryUserId  = null;
            $hasPrimary     = false;

            // Map roles -> user_ids
            $rolesMap = [];
            foreach ($roles as $index => $role) {
                if (isset($userIds[$index])) {
                    $rolesMap[$userIds[$index]] = $role;
                }
            }

            foreach ($userIds as $userId) {
                $role = $rolesMap[$userId] ?? 'secondary';

                if ($role === 'primary') {
                    if ($hasPrimary) {
                        $role = 'secondary';
                    } else {
                        $hasPrimary    = true;
                        $primaryUserId = $userId;
                    }
                }

                $assignmentData[$userId] = [
                    'role'        => $role,
                    'assigned_at' => now(),
                    'assigned_by' => $currentUserId,
                    'is_primary'  => $role === 'primary',
                    'created_at'  => now(),
                    'updated_at'  => now(),
                ];
            }

            if (!$hasPrimary && !empty($userIds)) {
                $firstUserId = $userIds[0];
                $assignmentData[$firstUserId]['role']       = 'primary';
                $assignmentData[$firstUserId]['is_primary'] = true;
                $primaryUserId                               = $firstUserId;
            }

            $conversation->assignedUsers()->sync($assignmentData);
            $conversation->update(['assigned_user_id' => $primaryUserId]);

            $this->createMultipleAssignmentMessage(
                $conversation,
                $userIds,
                $currentUserId,
                $request->input('reason')
            );

            if ($notifyUsers && !empty($userIds)) {
                $this->notifyAssignedUsers($conversation, $userIds, $currentUserId);
            }

            DB::commit();

            $this->chatRedis->publishConversationUpdate(
                $conversation->fresh(['assignedUsers', 'assignedUser']),
                'multiple_assignment'
            );

            $assignedUsers = User::whereIn('id', $userIds)->get(['id', 'name', 'email', 'avatar']);

            return $this->response([
                'success'        => true,
                'assigned_users' => $assignedUsers->map(function ($user) use ($assignmentData) {
                    return [
                        'user' => [
                            'id'    => $user->id,
                            'name'  => $user->name,
                            'email' => $user->email,
                            'avatar'=> $user->avatar,
                        ],
                        'role'        => $assignmentData[$user->id]['role'],
                        'is_primary'  => $assignmentData[$user->id]['is_primary'],
                        'assigned_at' => $assignmentData[$user->id]['assigned_at'],
                    ];
                }),
                'conversation'   => $this->formatConversation($conversation->fresh(), auth()->id()),
                'notification'   => [
                    'type'    => 'success',
                    'title'   => 'Sucesso',
                    'message' => 'Atribuições salvas com sucesso!',
                ],
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('❌ Erro ao atribuir múltiplos usuários: ' . $e->getMessage());
            return $this->response([
                'error'        => 'Erro ao atribuir usuários',
                'notification' => [
                    'type'    => 'error',
                    'title'   => 'Erro',
                    'message' => 'Falha na atribuição múltipla.',
                ],
            ], 500);
        }
    }


/**
 * ✅ LISTAR ATRIBUIÇÕES - CORRIGIDO
 */
public function getAssignments(Conversation $conversation): JsonResponse
{
    try {
        // ✅ CORRIGIR: Especificar explicitamente as colunas com prefixo da tabela
        $assignments = $conversation->assignedUsers()
            ->withPivot(['role', 'assigned_at', 'assigned_by', 'is_primary'])
            ->select([
                'users.id',           // ✅ Especificar tabela
                'users.name', 
                'users.email', 
                'users.avatar'
            ])
            ->get();

        return $this->response([
            'conversation_id'    => $conversation->id,
            'assignments'        => $assignments->map(function ($user) {
                return [
                    'user' => [
                        'id'    => $user->id,
                        'name'  => $user->name,
                        'email' => $user->email,
                        'avatar'=> $user->avatar,
                    ],
                    'role'        => $user->pivot->role,
                    'is_primary'  => $user->pivot->is_primary,
                    'assigned_at' => $user->pivot->assigned_at,
                    'assigned_by' => $user->pivot->assigned_by,
                ];
            }),
            'total_assignments' => $assignments->count(),
            'primary_user'      => $conversation->assignedUser ? [
                'id'   => $conversation->assignedUser->id,
                'name' => $conversation->assignedUser->name,
            ] : null,
        ]);
        
    } catch (\Exception $e) {
        Log::error('❌ Erro ao carregar atribuições: ' . $e->getMessage());
        return $this->response([
            'error'        => 'Erro ao carregar atribuições',
            'notification' => [
                'type'    => 'error',
                'title'   => 'Erro',
                'message' => 'Falha ao carregar atribuições.',
            ],
        ], 500);
    }
}

    /**
     * ✅ REMOVER ATRIBUIÇÃO DE USUÁRIO ESPECÍFICO
     */
    public function removeUserAssignment(Request $request, Conversation $conversation, int $userId): JsonResponse
    {
        try {
            $currentUserId = auth()->id();

            if (!$this->canManageAssignments($conversation, $currentUserId)) {
                return $this->response([
                    'error' => 'Sem permissão',
                    'notification' => ['type' => 'error', 'title' => 'Erro', 'message' => 'Sem permissão']
                ], 403);
            }

            DB::beginTransaction();

            // ✅ VERIFICAR SE USUÁRIO ESTÁ ATRIBUÍDO
            $assignment = $conversation->assignedUsers()->where('user_id', $userId)->first();
            if (!$assignment) {
                return $this->response([
                    'error' => 'Usuário não está atribuído a esta conversa',
                    'notification' => ['type' => 'warning', 'title' => 'Aviso', 'message' => 'Usuário não atribuído']
                ], 404);
            }

            $wasPrimary = $assignment->pivot->is_primary;
            $removedUser = User::find($userId);

            // ✅ REMOVER ATRIBUIÇÃO
            $conversation->assignedUsers()->detach($userId);

            // ✅ SE ERA PRIMÁRIO, PROMOVER OUTRO USUÁRIO
            if ($wasPrimary) {
                // Tenta encontrar outro primário primeiro, senão um secundário
                $newPrimaryCandidate = $conversation->assignedUsers()
                    ->wherePivot('is_primary', true)
                    ->first();
                
                if (!$newPrimaryCandidate) {
                    $newPrimaryCandidate = $conversation->assignedUsers()
                        ->wherePivot('role', 'secondary')
                        ->first();
                }

                if ($newPrimaryCandidate) {
                    $conversation->assignedUsers()->updateExistingPivot($newPrimaryCandidate->id, [
                        'role' => 'primary',
                        'is_primary' => true
                    ]);
                    $conversation->update(['assigned_user_id' => $newPrimaryCandidate->id]);
                } else {
                    $conversation->update(['assigned_user_id' => null]);
                }
            }

            // ✅ MENSAGEM DE SISTEMA
            $this->createRemovalMessage($conversation, $removedUser->name, $currentUserId);

            DB::commit();

            // ✅ BROADCAST
            $this->chatRedis->publishConversationUpdate(
                $conversation->fresh(['assignedUsers', 'assignedUser']), 
                'assignment_removed'
            );

            return $this->response([
                'success' => true,
                'removed_user_id' => $userId,
                'conversation' => $this->formatConversation($conversation->fresh(), auth()->id()), // ✅ Chamada correta ao método do trait
                'notification' => [
                    'type' => 'success',
                    'title' => 'Sucesso',
                    'message' => "Atribuição de {$removedUser->name} removida com sucesso!"
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('❌ Erro ao remover atribuição: ' . $e->getMessage());
            return $this->response([
                'error' => 'Erro ao remover atribuição',
                'notification' => ['type' => 'error', 'title' => 'Erro', 'message' => 'Falha ao remover atribuição']
            ], 500);
        }
    }

    // ===== MÉTODOS AUXILIARES =====

    private function canManageAssignments(Conversation $conversation, int $userId): bool
    {
        $user = auth()->user();

        // 1) Super admin pode tudo
        if ($user->super_admin) {
            return true;
        }

        // 2) Se ninguém está atribuído ainda (nem assigned_user_id, nem pivot) → qualquer usuário pode atribuir
        $hasPivotAssignments = $conversation->assignedUsers()->exists();
        if (!$conversation->assigned_user_id && !$hasPivotAssignments) {
            return true;
        }

        // 3) Se o usuário é o assigned_user_id atual
        if ((int) $conversation->assigned_user_id === (int) $userId) {
            return true;
        }

        // 4) Se é usuário primário na pivot
        $assignment = $conversation->assignedUsers()
            ->where('user_id', $userId)
            ->wherePivot('is_primary', true)
            ->first();

        return $assignment !== null;
    }

    private function createSimpleAssignmentMessage(Conversation $conversation, ?int $oldUserId, ?int $newUserId, int $assignedBy, ?string $reason): void
    {
        try {
            $assignedByUser = User::find($assignedBy);
            $oldUser = $oldUserId ? User::find($oldUserId) : null;
            $newUser = $newUserId ? User::find($newUserId) : null;

            if (!$newUserId) {
                $content = "{$assignedByUser->name} removeu a atribuição desta conversa.";
            } elseif (!$oldUserId) {
                $content = "{$assignedByUser->name} atribuiu esta conversa para {$newUser->name}.";
            } else {
                $content = "{$assignedByUser->name} reatribuiu esta conversa de {$oldUser->name} para {$newUser->name}.";
            }

            if ($reason) {
                $content .= " Motivo: {$reason}";
            }

            Message::create([
                'conversation_id' => $conversation->id,
                'sender_type' => 'system',
                'content' => $content,
                'message_type' => 'system',
                'is_from_contact' => false,
                'status' => 'sent',
                'sent_at' => now(),
                'metadata' => [
                    'is_system_message' => true,
                    'system_type' => 'assignment_updated',
                    'old_user_id' => $oldUserId,
                    'new_user_id' => $newUserId,
                    'assigned_by' => $assignedBy
                ]
            ]);

            $conversation->update(['last_message_at' => now()]);

        } catch (\Exception $e) {
            Log::error('Erro ao criar mensagem de atribuição simples: ' . $e->getMessage());
        }
    }

    private function createMultipleAssignmentMessage(Conversation $conversation, array $userIds, int $assignedBy, ?string $reason): void
    {
        try {
            $assignedByUser = User::find($assignedBy);
            $assignedUsers = User::whereIn('id', $userIds)->pluck('name')->toArray();

            $content = "{$assignedByUser->name} atribuiu esta conversa para: " . implode(', ', $assignedUsers);
            if ($reason) {
                $content .= ". Motivo: {$reason}";
            }

            Message::create([
                'conversation_id' => $conversation->id,
                'sender_type' => 'system',
                'content' => $content,
                'message_type' => 'system',
                'is_from_contact' => false,
                'status' => 'sent',
                'sent_at' => now(),
                'metadata' => [
                    'is_system_message' => true,
                    'system_type' => 'multiple_assignment',
                    'assigned_user_ids' => $userIds,
                    'assigned_by' => $assignedBy
                ]
            ]);

            $conversation->update(['last_message_at' => now()]);

        } catch (\Exception $e) {
            Log::error('Erro ao criar mensagem de atribuição múltipla: ' . $e->getMessage());
        }
    }

    private function createRemovalMessage(Conversation $conversation, string $removedUserName, int $removedBy): void
    {
        try {
            $removedByUser = User::find($removedBy);
            $content = "{$removedByUser->name} removeu a atribuição de {$removedUserName} desta conversa.";

            Message::create([
                'conversation_id' => $conversation->id,
                'sender_type' => 'system',
                'content' => $content,
                'message_type' => 'system',
                'is_from_contact' => false,
                'status' => 'sent',
                'sent_at' => now(),
                'metadata' => [
                    'is_system_message' => true,
                    'system_type' => 'assignment_removed',
                    'removed_by' => $removedBy
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Erro ao criar mensagem de remoção: ' . $e->getMessage());
        }
    }

    private function notifyAssignedUsers(Conversation $conversation, array $userIds, int $assignedBy): void
    {
        // ✅ BROADCAST PARA TODOS OS USUÁRIOS ATRIBUÍDOS
        foreach ($userIds as $userId) {
            $this->chatRedis->publishToStream('user_assigned_to_conversation', [
                'conversation_id' => $conversation->id,
                'user_id' => $userId,
                'assigned_by' => $assignedBy,
                'timestamp' => now()->toISOString()
            ]);
        }
    }

    private function getAssignmentMessage(?int $oldUserId, ?int $newUserId, ?User $assignedUser): string
    {
        if (!$newUserId) {
            return 'Atribuição removida com sucesso.';
        } elseif (!$oldUserId) {
            return "Conversa atribuída para {$assignedUser->name} com sucesso.";
        } else {
            return "Conversa reatribuída para {$assignedUser->name} com sucesso.";
        }
    }
}