<?php

namespace Modules\Chat\Http\Controllers\Api;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Modules\Core\Http\Controllers\ApiController;
use Modules\Chat\Models\Conversation;
use Modules\Chat\Models\Message;
use Modules\Chat\Models\ChatChannel;
use Modules\Contacts\Models\Contact;
use Modules\Users\Models\User;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Modules\Chat\Services\ChatRedisService;
use Modules\Chat\Services\ConversationService;
use Modules\Chat\Http\Traits\FormatsConversations;

class ConversationController extends ApiController
{
    use FormatsConversations;

    private const DEFAULT_PER_PAGE = 20;
    private const MAX_PER_PAGE = 100;
    private const VALID_STATUSES = ['active', 'closed', 'waiting', 'archived', 'pending'];
    private const VALID_TYPES = ['internal', 'external'];
    private const VALID_PRIORITIES = ['low', 'normal', 'high', 'urgent'];

    protected ConversationService $conversationService;
    protected ChatRedisService $chatRedisService;

    public function __construct(
        ConversationService $conversationService,
        ChatRedisService $chatRedisService
    ) {
        $this->conversationService = $conversationService;
        $this->chatRedisService = $chatRedisService;
    }

    public function index(Request $request): JsonResponse
    {
        try {
            $userId = auth()->id();
            $user = auth()->user();
            $isAdmin = $user->super_admin ?? false;

            $filters = $this->extractFilters($request);
            $pagination = $this->extractPagination($request);
            
            // ✅ CORRIGIDO: Obter parâmetros diretamente do request
            $showMyOnly = $request->boolean('show_my_only', false);
            $showAllConversations = $request->boolean('show_all_conversations', false);
            $showMineAndUnassigned = $request->boolean('show_mine_and_unassigned', false);

            $result = $this->conversationService->getConversationsForUser(
                $userId, 
                $filters, 
                $pagination,
                $showMyOnly,
                $showAllConversations,
                $showMineAndUnassigned, // ✅ NOVO PARÂMETRO
                $isAdmin
            );

            return $this->response([
                'data' => $result['conversations']->map(fn($conv) => $this->formatConversation($conv, $userId))->toArray(),
                'meta' => $this->buildPaginationMeta(
                    $result['conversations'], 
                    $filters, 
                    $isAdmin, 
                    $userId, 
                    $showMyOnly, 
                    $showAllConversations,
                    $showMineAndUnassigned // ✅ ADICIONAR AO META
                )
            ]);

        } catch (\Exception $e) {
            Log::error('Erro ao carregar conversas', [
                'user_id' => auth()->id(),
                'error' => $e->getMessage()
            ]);

            return $this->response([
                'error' => 'Erro ao carregar conversas',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao carregar conversas.',
                ],
            ], 500);
        }
    }

    public function store(Request $request): JsonResponse
    {
        $rules = $this->getValidationRules($request->input('type'));
        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return $this->response([
                'errors' => $validator->errors(),
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro de Validação',
                    'message' => 'Dados inválidos para criar conversa.'
                ]
            ], 422);
        }

        try {
            $result = $this->conversationService->createConversation($request->all());
            return $this->response($result, $result['success'] ? 201 : 422);

        } catch (\Exception $e) {
            Log::error('Erro ao criar conversa', [
                'type' => $request->input('type'),
                'error' => $e->getMessage(),
                'user_id' => auth()->id()
            ]);

            return $this->response([
                'error' => 'Erro interno do servidor',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao criar conversa.'
                ]
            ], 500);
        }
    }

    public function show($conversationId): JsonResponse
    {
        try {
            $conversation = Conversation::with([
                'contact',
                'assignedUser',
                'lastMessage',
                'participants'
            ])->findOrFail($conversationId);

            // Verificar permissão
            if (!$conversation->canUserView(auth()->id())) {
                return $this->response([
                    'error' => 'Acesso negado',
                    'notification' => ['type' => 'error', 'title' => 'Erro', 'message' => 'Você não tem permissão para ver esta conversa.']
                ], 403);
            }

            return $this->response($this->formatConversation($conversation, auth()->id()));
        } catch (\Exception $e) {
            Log::error('Erro ao carregar conversa', ['error' => $e->getMessage()]);
            return $this->response([
                'error' => 'Conversa não encontrada',
                'notification' => ['type' => 'error', 'title' => 'Erro', 'message' => 'Conversa não encontrada.']
            ], 404);
        }
    }

    public function finalize(Request $request, $conversationId): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'reason' => 'required|string|max:255',
            'resolution' => 'nullable|string|max:1000',
            'send_farewell_message' => 'boolean',
            'farewell_message' => 'nullable|string|max:500',
            'archive_conversation' => 'boolean',
            'block_contact' => 'boolean',
            'notify_team' => 'boolean',
            'request_feedback' => 'boolean',
            'closed_at' => 'nullable|date',
            'finalized_by' => 'nullable|integer|exists:users,id'
        ]);

        if ($validator->fails()) {
            return $this->response([
                'errors' => $validator->errors(),
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro de Validação',
                    'message' => 'Dados inválidos para finalizar conversa.'
                ]
            ], 422);
        }

        try {
            $conversation = Conversation::findOrFail($conversationId);
            
            // Verificar permissão
            if (!$conversation->canUserView(auth()->id())) {
                return $this->response([
                    'error' => 'Acesso negado',
                    'notification' => [
                        'type' => 'error', 
                        'title' => 'Erro', 
                        'message' => 'Você não tem permissão para finalizar esta conversa.'
                    ]
                ], 403);
            }

            $result = $this->conversationService->finalizeConversation(
                $conversation, 
                $request->all(),
                auth()->id()
            );

            return $this->response($result);

        } catch (\Exception $e) {
            Log::error('Erro ao finalizar conversa', [
                'conversation_id' => $conversationId,
                'error' => $e->getMessage(),
                'user_id' => auth()->id()
            ]);

            return $this->response([
                'error' => 'Erro interno do servidor',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao finalizar conversa.'
                ]
            ], 500);
        }
    }

    public function update(Request $request, Conversation $conversation): JsonResponse
    {
        $rules = [
            'is_pinned' => 'nullable|boolean',
            'is_muted' => 'nullable|boolean',
            'subject' => 'nullable|string|max:255',
            'description' => 'nullable|string|max:500',
            'priority' => 'nullable|string|in:' . implode(',', self::VALID_PRIORITIES),
            'assigned_user_id' => 'nullable|integer|exists:users,id',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return $this->response([
                'errors' => $validator->errors(),
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro de Validação',
                    'message' => 'Dados inválidos para atualizar conversa.'
                ]
            ], 422);
        }

        try {
            $result = $this->conversationService->updateConversation($conversation, $request->all());
            return $this->response($result);

        } catch (\Exception $e) {
            Log::error('Erro ao atualizar conversa', [
                'conversation_id' => $conversation->id,
                'error' => $e->getMessage()
            ]);
            return $this->response([
                'error' => 'Erro interno do servidor',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao atualizar conversa.'
                ]
            ], 500);
        }
    }

    public function updateStatus(Request $request, $conversationId): JsonResponse
    {
        // ✅ DETECTAR TIPO DE OPERAÇÃO PELA ROTA
        $currentRoute = $request->route()->getName() ?? $request->getPathInfo();
        $isFinalize = str_contains($currentRoute, 'finalize');
        $isReopen = str_contains($currentRoute, 'reopen');
        
        // ✅ VALIDAÇÃO DINÂMICA BASEADA NA OPERAÇÃO
        $rules = [
            'reason' => 'nullable|string|max:500'
        ];
        
        // ✅ REGRAS ADICIONAIS PARA FINALIZAÇÃO
        if ($isFinalize) {
            $rules = array_merge($rules, [
                'resolution' => 'nullable|string|max:1000',
                'send_farewell_message' => 'boolean',
                'farewell_message' => 'nullable|string|max:500',
                'archive_conversation' => 'boolean',
                'block_contact' => 'boolean',
                'notify_team' => 'boolean',
                'request_feedback' => 'boolean',
                'closed_at' => 'nullable|date',
                'finalized_by' => 'nullable|integer|exists:users,id'
            ]);
            
            // ✅ FORÇAR STATUS CLOSED PARA FINALIZAÇÃO
            $request->merge(['status' => 'closed']);
        }
        
        // ✅ REGRAS ADICIONAIS PARA REABERTURA
        if ($isReopen) {
            $rules = array_merge($rules, [
                'reopen_reason' => 'nullable|string|max:500',
                'reopened_at' => 'nullable|date',
                'reopened_by' => 'nullable|integer|exists:users,id'
            ]);
            
            // ✅ FORÇAR STATUS ACTIVE PARA REABERTURA
            $request->merge(['status' => 'active']);
        }
        
        // ✅ SE NÃO FOR FINALIZE OU REOPEN, VALIDAR STATUS NORMAL
        if (!$isFinalize && !$isReopen) {
            $rules['status'] = 'required|in:' . implode(',', self::VALID_STATUSES);
        }

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return $this->response([
                'errors' => $validator->errors(),
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro de Validação',
                    'message' => 'Dados inválidos para atualizar status da conversa.'
                ]
            ], 422);
        }

        try {
            $conversation = Conversation::findOrFail($conversationId);
            
            // ✅ VERIFICAR PERMISSÃO
            if (!$conversation->canUserView(auth()->id())) {
                return $this->response([
                    'error' => 'Acesso negado',
                    'notification' => [
                        'type' => 'error', 
                        'title' => 'Erro', 
                        'message' => 'Você não tem permissão para alterar esta conversa.'
                    ]
                ], 403);
            }
            
            // ✅ EXTRAIR DADOS EXTRAS PARA FINALIZAÇÃO/REABERTURA
            $extraData = [];
            if ($isFinalize) {
                $extraData = $request->only([
                    'resolution', 'send_farewell_message', 'farewell_message',
                    'archive_conversation', 'block_contact', 'notify_team', 
                    'request_feedback', 'closed_at', 'finalized_by'
                ]);
            } elseif ($isReopen) {
                $extraData = $request->only([
                    'reopen_reason', 'reopened_at', 'reopened_by'
                ]);
            }
            
            // ✅ USAR O MÉTODO EXISTENTE COM DADOS EXTRAS
            $result = $this->conversationService->updateConversationStatus(
                $conversation, 
                $request->input('status'), 
                $request->input('reason'),
                $extraData
            );

            return $this->response($result);

        } catch (\Exception $e) {
            Log::error('Erro ao atualizar status', [
                'conversation_id' => $conversationId,
                'operation' => $isFinalize ? 'finalize' : ($isReopen ? 'reopen' : 'update_status'),
                'error' => $e->getMessage()
            ]);
            
            return $this->response([
                'error' => 'Erro interno do servidor',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao atualizar status da conversa.'
                ]
            ], 500);
        }
    }

    public function assign(Request $request, $conversationId): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'nullable|integer|exists:users,id'
        ]);

        if ($validator->fails()) {
            return $this->response(['errors' => $validator->errors()], 422);
        }

        try {
            $conversation = Conversation::findOrFail($conversationId);
            $result = $this->conversationService->assignConversation(
                $conversation, 
                $request->input('user_id')
            );

            return $this->response($result);

        } catch (\Exception $e) {
            Log::error('Erro ao atribuir conversa', ['error' => $e->getMessage()]);
            return $this->response([
                'error' => 'Erro interno do servidor',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao atribuir conversa.'
                ]
            ], 500);
        }
    }

    public function markAsRead(Request $request, $conversationId): JsonResponse
    {
        try {
            $result = $this->conversationService->markAsRead($conversationId, auth()->id());
            return $this->response($result);

        } catch (\Exception $e) {
            Log::error('Erro ao marcar como lida', ['error' => $e->getMessage()]);
            return $this->response([
                'error' => 'Erro interno do servidor',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao marcar conversa como lida.'
                ]
            ], 500);
        }
    }

    public function markAsUnread(Request $request, $conversationId): JsonResponse
    {
        try {
            $result = $this->conversationService->markAsUnread($conversationId, auth()->id());
            return $this->response($result);

        } catch (\Exception $e) {
            Log::error('Erro ao marcar como não lida', ['error' => $e->getMessage()]);
            return $this->response([
                'error' => 'Erro interno do servidor',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao marcar conversa como não lida.'
                ]
            ], 500);
        }
    }

    public function destroy(Conversation $conversation): JsonResponse
    {
        try {
            $result = $this->conversationService->deleteConversation($conversation);
            return $this->response($result);

        } catch (\Exception $e) {
            Log::error('Erro ao excluir conversa', [
                'conversation_id' => $conversation->id,
                'error' => $e->getMessage(),
                'error_type' => get_class($e)
            ]);

            if (str_contains($e->getMessage(), 'Lock wait timeout')) {
                return $this->response([
                    'error' => 'A conversa está sendo processada. Tente novamente em alguns segundos.',
                    'notification' => [
                        'type' => 'warning',
                        'title' => 'Aguarde',
                        'message' => 'A conversa está sendo processada. Tente novamente em alguns segundos.'
                    ]
                ], 423);
            }

            return $this->response([
                'error' => 'Erro interno do servidor',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao excluir conversa.'
                ]
            ], 500);
        }
    }

    // ===== MÉTODOS DE ESTATÍSTICAS =====

    public function getStats(): JsonResponse
    {
        try {
            $stats = $this->conversationService->getStats(auth()->id());
            return $this->response($stats);

        } catch (\Exception $e) {
            Log::error('Erro ao carregar estatísticas', ['error' => $e->getMessage()]);
            return $this->response([
                'error' => 'Erro interno do servidor',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao carregar estatísticas.'
                ]
            ], 500);
        }
    }

    public function summary(Request $request): JsonResponse
    {
        try {
            $filters = $this->extractFilters($request);
            $summary = $this->conversationService->getSummary(auth()->id(), $filters);
            return $this->response($summary);

        } catch (\Exception $e) {
            Log::error('Erro ao obter resumo', ['error' => $e->getMessage()]);
            return $this->response([
                'error' => 'Erro ao obter resumo',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao obter resumo das conversas.'
                ]
            ], 500);
        }
    }

    public function getUsers(): JsonResponse
    {
        try {
            $users = User::select(['id', 'name', 'email', 'avatar'])
                ->orderBy('name')
                ->get()
                ->map(function ($user) {
                    return [
                        'id' => $user->id,
                        'name' => $user->name,
                        'email' => $user->email,
                        'avatar' => $user->avatar ?? null,
                        'is_super_admin' => $user->super_admin ?? false,
                        'has_api_access' => $user->access_api ?? false
                    ];
                });

            return $this->response($users);
        } catch (\Exception $e) {
            Log::error('Erro ao carregar usuários', ['error' => $e->getMessage()]);
            return $this->response([
                'error' => 'Erro interno do servidor',
                                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Falha ao carregar usuários.'
                ]
            ], 500);
        }
    }

    // ===== MÉTODOS AUXILIARES =====

        private function extractFilters(Request $request): array
    {
        return [
            'status' => $request->input('status'),
            'type' => $request->input('type'),
            'assigned' => $request->input('assigned'),
            'search' => trim((string) $request->input('search', '')),
            'only_unread' => $request->boolean('only_unread', false),
            'include_archived' => $request->boolean('include_archived', false),
            // ✅ CORRIGIDO: Adicionar o filtro que estava faltando
            'show_mine_and_unassigned' => $request->boolean('show_mine_and_unassigned', false),
        ];
    }

    private function extractPagination(Request $request): array
    {
        return [
            'per_page' => min((int) $request->input('per_page', self::DEFAULT_PER_PAGE), self::MAX_PER_PAGE),
            'page' => max((int) $request->input('page', 1), 1),
        ];
    }

    private function getValidationRules(string $type): array
    {
        $baseRules = [
            'type' => 'required|in:' . implode(',', self::VALID_TYPES),
            'priority' => 'nullable|in:' . implode(',', self::VALID_PRIORITIES),
            'tags' => 'nullable|array',
            'initial_message' => 'nullable|string|max:4000',
            'notifications_enabled' => 'boolean',
            'auto_assign' => 'boolean',
        ];

        if ($type === 'external') {
            return array_merge($baseRules, [
                'channel_id' => 'required|exists:chat_channels,id',
                'contact_id' => 'nullable|exists:contacts,id',
                'contact_name' => 'required_without:contact_id|string|max:255',
                'phone' => 'required_without:contact_id|string|max:20',
                'contact_email' => 'nullable|email|max:255',
            ]);
        }

        if ($type === 'internal') {
            return array_merge($baseRules, [
                'subject' => 'required|string|max:255',
                'participants' => 'required|array|min:1',
                'participants.*' => 'exists:users,id',
                'internal_type' => 'nullable|in:discussion,project,support,meeting,announcement,other',
                'description' => 'nullable|string|max:500',
                'channel_id' => 'nullable|exists:chat_channels,id',
            ]);
        }

        return $baseRules;
    }

    private function buildPaginationMeta(
        $conversations, 
        array $filters, 
        bool $isAdmin, 
        int $userId, 
        bool $showMyOnly = false, 
        bool $showAllConversations = false,
        bool $showMineAndUnassigned = false // ✅ NOVO PARÂMETRO
    ): array {
        return [
            'current_page' => $conversations->currentPage(),
            'last_page' => $conversations->lastPage(),
            'per_page' => $conversations->perPage(),
            'total' => $conversations->total(),
            'from' => $conversations->firstItem(),
            'to' => $conversations->lastItem(),
            'has_more' => $conversations->hasMorePages(),
            'is_admin' => $isAdmin,
            'user_id' => $userId,
            'show_my_only' => $showMyOnly,
            'show_all_conversations' => $showAllConversations,
            'show_mine_and_unassigned' => $showMineAndUnassigned, // ✅ ADICIONAR AO META
            'filters' => $filters,
        ];
    }
}