<?php
// modules/Chat/Http/Controllers/Api/EvolutionController.php - VALIDADO E COMPLETO

namespace Modules\Chat\Http\Controllers\Api;

use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Modules\Core\Http\Controllers\ApiController;
use Modules\Chat\Models\EvolutionInstance;
use Modules\Chat\Services\EvolutionApiService;
use Modules\Chat\Services\EvolutionSyncService;
use Modules\Chat\Services\ChatRedisService;
use Exception;
use Carbon\Carbon;

class EvolutionController extends ApiController
{
    protected EvolutionApiService $evolutionApi;
    protected EvolutionSyncService $evolutionSync;
    protected ChatRedisService $chatRedis;

    public function __construct(
        EvolutionApiService $evolutionApi,
        EvolutionSyncService $evolutionSync,
        ChatRedisService $chatRedis
    ) {
        $this->middleware('auth:sanctum');
        $this->evolutionApi = $evolutionApi;
        $this->evolutionSync = $evolutionSync;
        $this->chatRedis = $chatRedis;
    }

    /**
     * ✅ LISTAR TODAS AS INSTÂNCIAS EVOLUTION
     * @param Request $request
     * @return JsonResponse
     */
public function index(Request $request): JsonResponse
    {
        try {
            Log::info('📋 Listando instâncias Evolution', ['user_id' => auth()->id()]);

            $query = EvolutionInstance::query();

            // Incluir relacionamentos, como 'createdBy'
            if ($request->has('with')) {
                $query->with(explode(',', $request->input('with')));
            }

            $instances = $query->get();

            $instances->each(function ($instance) {
                // Força o acesso aos accessors para que os atributos calculados apareçam no JSON
                $instance->messages_sent_today;
                $instance->messages_received_today;
                $instance->is_online;
                $instance->qr_code_expired;
                // Esconder API Key na resposta por segurança
                $instance->makeHidden('api_key');
            });

            Log::info('✅ Instâncias Evolution carregadas', ['count' => $instances->count()]);
            return $this->response($instances);

        } catch (Exception $e) {
            Log::error('❌ Erro ao listar instâncias Evolution: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
            return $this->response(['error' => 'Erro ao listar instâncias: ' . $e->getMessage()], 500);
        }
    }

        public function instances(Request $request): JsonResponse
    {
        return $this->index($request);
    }

/**
 * ➕ CRIAR NOVA INSTÂNCIA EVOLUTION - VERSÃO CORRIGIDA
 */
public function store(Request $request): JsonResponse
{
    $validator = Validator::make($request->all(), [
        'instance_name' => 'required|string|max:255|unique:evolution_instances,instance_name',
        'display_name' => 'required|string|max:255',
        'description' => 'nullable|string|max:1000',
        'api_url' => 'nullable|url|max:255',
        'api_key' => 'nullable|string|max:255',
        'api_version' => 'nullable|string|max:10',
        'is_default' => 'sometimes|boolean',
        'priority' => 'nullable|integer|min:1',
    ]);

    if ($validator->fails()) {
        return $this->response(['errors' => $validator->errors()], 422);
    }

    DB::beginTransaction();
    try {
        // ✅ OBTER CONFIGURAÇÕES DO ENV E PROCESSAR
        $envApiUrl = env('EVOLUTION_API_URL');
        $envApiKey = env('EVOLUTION_API_KEY');
        $envApiVersion = env('EVOLUTION_API_VERSION', 'v1');

        // ✅ PROCESSAR URL DO ENV (adicionar https:// se necessário)
        $defaultApiUrl = null;
        if ($envApiUrl) {
            if (!str_starts_with($envApiUrl, 'http://') && !str_starts_with($envApiUrl, 'https://')) {
                $defaultApiUrl = 'https://' . $envApiUrl;
            } else {
                $defaultApiUrl = $envApiUrl;
            }
        }

        // ✅ USAR VALORES DO REQUEST OU PADRÃO DO ENV
        $apiUrl = $request->api_url ?: $defaultApiUrl;
        $apiKey = $request->api_key ?: $envApiKey;
        $apiVersion = $request->api_version ?: $envApiVersion;

        // ✅ VALIDAR SE TEMOS AS CONFIGURAÇÕES NECESSÁRIAS
        if (!$apiUrl) {
            return $this->response([
                'error' => 'API URL não configurada. Configure EVOLUTION_API_URL no .env ou forneça uma URL.',
                'env_value' => $envApiUrl,
                'processed_value' => $defaultApiUrl
            ], 422);
        }

        if (!$apiKey) {
            return $this->response([
                'error' => 'API Key não configurada. Configure EVOLUTION_API_KEY no .env ou forneça uma chave.',
                'env_value' => $envApiKey ? 'CONFIGURADA' : 'NÃO CONFIGURADA'
            ], 422);
        }

        Log::info('✅ Configurações Evolution obtidas', [
            'api_url' => $apiUrl,
            'api_key_configured' => !empty($apiKey),
            'api_version' => $apiVersion,
            'source' => $request->api_url ? 'request' : 'env'
        ]);

        // Se for definida como padrão, garantir que outras não sejam
        if ($request->boolean('is_default')) {
            EvolutionInstance::where('is_default', true)->update(['is_default' => false]);
        }

        $instance = EvolutionInstance::create([
            'instance_name' => $request->instance_name,
            'display_name' => $request->display_name,
            'description' => $request->description,
            'api_url' => $apiUrl,
            'api_key' => $apiKey,
            'api_version' => $apiVersion,
            'is_default' => $request->boolean('is_default'),
            'priority' => $request->priority ?? 1,
            'created_by' => auth()->id(),
            'status' => 'disconnected',
            'connection_state' => 'close',
        ]);

        // ✅ CONFIGURAR WEBHOOK
        try {
            $webhookUrl = route('evolution.webhook', ['instanceName' => $instance->instance_name]);
            
            $instance->update([
                'webhook_url' => $webhookUrl,
                'webhook_events' => $request->webhook_events ?? [
                    'messages', 'message_status', 'connection_status', 'qrcode_updated'
                ]
            ]);

            Log::info('✅ Webhook URL configurada', [
                'instance_name' => $instance->instance_name,
                'webhook_url' => $webhookUrl
            ]);

            // ✅ TENTAR CONFIGURAR O WEBHOOK NA API EVOLUTION
            $this->evolutionApi->setInstance($instance);
            try {
                $webhookResponse = $this->evolutionApi->setupWebhook();
                $instance->update(['webhook_active' => true]);
                Log::info('✅ Webhook configurado na Evolution API', [
                    'instance_name' => $instance->instance_name,
                    'response' => $webhookResponse
                ]);
            } catch (Exception $webhookSetupException) {
                Log::warning('⚠️ Erro ao configurar webhook na Evolution API (instância criada mesmo assim)', [
                    'instance_name' => $instance->instance_name,
                    'error' => $webhookSetupException->getMessage()
                ]);
                $instance->recordError('Webhook setup failed: ' . $webhookSetupException->getMessage());
            }

        } catch (Exception $webhookException) {
            Log::warning('⚠️ Erro ao gerar URL do webhook', [
                'instance_name' => $instance->instance_name,
                'error' => $webhookException->getMessage()
            ]);
        }

        DB::commit();
        Log::info('✅ Instância Evolution criada', [
            'instance_name' => $instance->instance_name, 
            'user_id' => auth()->id(),
            'api_url' => $apiUrl
        ]);
        
        return $this->response($instance->makeHidden('api_key'), 201);

    } catch (Exception $e) {
        DB::rollBack();
        Log::error('❌ Erro ao criar instância Evolution: ' . $e->getMessage(), [
            'request_data' => $request->all(),
            'trace' => $e->getTraceAsString()
        ]);
        return $this->response(['error' => 'Erro ao criar instância: ' . $e->getMessage()], 500);
    }
}

/**
 * 👁️ MOSTRAR DETALHES DE UMA INSTÂNCIA EVOLUTION - CORRIGIDO
 */
public function show(string $instanceName): JsonResponse
{
    try {
        // ✅ USAR first() EM VEZ DE firstOrFail()
        $instance = EvolutionInstance::where('instance_name', $instanceName)->first();
        
        if (!$instance) {
            return $this->response(['error' => 'Instância não encontrada'], 404);
        }
        
        $instance->messages_sent_today;
        $instance->messages_received_today;
        $instance->is_online;
        $instance->qr_code_expired;
        
        Log::info('📋 Detalhes da instância Evolution', ['instance_name' => $instanceName]);
        return $this->response($instance->makeHidden('api_key'));

    } catch (Exception $e) {
        Log::error('❌ Erro ao obter detalhes da instância Evolution: ' . $e->getMessage(), ['instance_name' => $instanceName]);
        return $this->response(['error' => 'Instância não encontrada ou erro: ' . $e->getMessage()], 404);
    }
}

    /**
     * 🔄 ATUALIZAR INSTÂNCIA EVOLUTION
     * @param Request $request
     * @param string $instanceName
     * @return JsonResponse
     */
    public function update(Request $request, string $instanceName): JsonResponse
    {
        $instance = EvolutionInstance::where('instance_name', $instanceName)->first();

        if (!$instance) {
        return $this->response(['error' => 'Instância não encontrada'], 404);
        }

        $validator = Validator::make($request->all(), [
            'display_name' => 'sometimes|required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'api_url' => 'sometimes|required|url|max:255',
            'api_key' => 'nullable|string|max:255', // Permite null para não alterar
            'api_version' => 'nullable|string|max:10',
            'is_default' => 'sometimes|boolean',
            'priority' => 'nullable|integer|min:1',
        ]);

        if ($validator->fails()) {
            return $this->response(['errors' => $validator->errors()], 422);
        }

        DB::beginTransaction();
        try {
            $updateData = $request->only(['display_name', 'description', 'api_url', 'api_version', 'is_default', 'priority']);

            if ($request->has('api_key') && !empty($request->api_key)) {
                $updateData['api_key'] = $request->api_key; // Mutator irá criptografar
            }

            if ($request->boolean('is_default')) {
                EvolutionInstance::where('is_default', true)->where('id', '!=', $instance->id)->update(['is_default' => false]);
            }

            $instance->update($updateData);

            // Reconfigurar webhook se API URL, KEY ou eventos mudaram, e o webhook estava ativo
            if (($request->has('api_url') || $request->has('api_key')) && $instance->webhook_active) {
                $this->evolutionApi->setInstance($instance);
                try {
                    $this->evolutionApi->setupWebhook();
                } catch (Exception $webhookUpdateException) {
                     Log::warning('⚠️ Erro ao reconfigurar webhook após atualização da instância.', [
                        'instance_name' => $instance->instance_name,
                        'error' => $webhookUpdateException->getMessage()
                    ]);
                    $instance->recordError('Webhook re-setup failed: ' . $webhookUpdateException->getMessage());
                }
            }

            DB::commit();
            Log::info('✅ Instância Evolution atualizada', ['instance_name' => $instance->instance_name, 'user_id' => auth()->id()]);
            return $this->response($instance->makeHidden('api_key'));

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('❌ Erro ao atualizar instância Evolution: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
            return $this->response(['error' => 'Erro ao atualizar instância: ' . $e->getMessage()], 500);
        }
    }

    /**
     * 🗑️ EXCLUIR INSTÂNCIA EVOLUTION
     * @param string $instanceName
     * @return JsonResponse
     */
    public function destroy(string $instanceName): JsonResponse
    {
        $instance = EvolutionInstance::where('instance_name', $instanceName)->first();
            
            if (!$instance) {
                return $this->response(['error' => 'Instância não encontrada'], 404);
            }

        DB::beginTransaction();
        try {
            // Ao usar SoftDeletes, ele será marcado como deleted_at
            $instance->delete();

            // Opcional: Desvincular conversas ou excluir (se não for feito via SoftDeletes na relação)
            // Exemplo para desvincular:
            // $instance->conversations()->update(['evolution_instance_name' => null]);
            // Exemplo para excluir:
            // $instance->conversations()->delete();

            DB::commit();
            Log::info('✅ Instância Evolution excluída (soft delete)', ['instance_name' => $instance->instance_name, 'user_id' => auth()->id()]);
            return $this->response(['message' => 'Instância excluída com sucesso'], 200);

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('❌ Erro ao excluir instância Evolution: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
            return $this->response(['error' => 'Erro ao excluir instância: ' . $e->getMessage()], 500);
        }
    }

/**
 * 🔗 OBTER STATUS DE CONEXÃO DA INSTÂNCIA - VERSÃO CORRIGIDA
 */
public function connectionStatus(string $instanceName): JsonResponse
{
    try {
        Log::info('🔍 Buscando instância:', ['instance_name' => $instanceName]);
        
        $instance = EvolutionInstance::where('instance_name', $instanceName)->first();
        
        if (!$instance) {
            Log::warning('❌ Instância não encontrada:', ['instance_name' => $instanceName]);
            
            $available = EvolutionInstance::pluck('instance_name')->toArray();
            Log::info('📋 Instâncias disponíveis:', $available);
            
            return $this->response([
                'error' => 'Instância não encontrada',
                'instance_name' => $instanceName,
                'available_instances' => $available
            ], 404);
        }
        
        Log::info('✅ Instância encontrada:', [
            'id' => $instance->id,
            'instance_name' => $instance->instance_name,
            'display_name' => $instance->display_name
        ]);
        
        try {
            // ✅ CHAMAR O SERVICE PARA OBTER STATUS
            $this->evolutionApi->setInstance($instance);
            $statusDataFromApi = $this->evolutionApi->getConnectionStatus();
            $fullResponseData = $statusDataFromApi['full_response'];
            $state = $statusDataFromApi['state'];
            $connected = $statusDataFromApi['connected'];

            $oldConnectionState = $instance->connection_state;
            
            // ✅ GARANTIR QUE O STATE É VÁLIDO PARA O ENUM
            $validStates = ['close', 'connecting', 'open', 'qrcode', 'disconnected', 'error', 'unknown'];
            $validState = in_array($state, $validStates) ? $state : 'unknown';
            
            // ✅ ATUALIZAR O MODELO COM OS DADOS MAIS RECENTES DA API
            $updateData = [
                'status' => $connected ? 'active' : 'disconnected',
                'connection_state' => $validState,
                'whatsapp_name' => $fullResponseData['instance']['profileName'] ?? null,
                'whatsapp_number' => $fullResponseData['instance']['number'] ?? null,
                'whatsapp_profile_picture' => $fullResponseData['instance']['profilePicture'] ?? null,
                'connected_at' => $connected ? now() : $instance->connected_at,
                'disconnected_at' => !$connected ? now() : $instance->disconnected_at,
                'last_ping' => now(),
                'instance_data' => $fullResponseData['instance'] ?? []
            ];
            
            Log::info('🔄 Atualizando instância com dados:', [
                'instance_name' => $instanceName,
                'old_state' => $oldConnectionState,
                'new_state' => $validState,
                'connected' => $connected
            ]);
            
            $instance->update($updateData);

            // ✅ PUBLICAR EVENTO REDIS SE O STATUS MUDOU
            if ($oldConnectionState !== $validState) {
                $this->chatRedis->publishEvolutionConnectionStatus(
                    $instance->instance_name,
                    $validState,
                    $fullResponseData
                );
            }

            Log::info('✅ Status de conexão obtido e atualizado', [
                'instance_name' => $instanceName, 
                'status' => $validState,
                'connected' => $connected
            ]);
            
            return $this->response([
                'connected' => $connected,
                'state' => $validState,
                'instance_data' => $fullResponseData,
                'is_online' => $instance->is_online,
                'whatsapp_name' => $instance->whatsapp_name,
                'whatsapp_number' => $instance->whatsapp_number,
                'qr_code_expired' => $instance->qr_code_expired,
                'qr_code_base64' => $instance->qr_code['base64'] ?? null,
            ]);
            
        } catch (Exception $apiException) {
            // ✅ SE DEU ERRO NA API, ATUALIZAR COMO DISCONNECTED
            Log::warning('⚠️ Erro ao comunicar com Evolution API', [
                'instance_name' => $instanceName,
                'error' => $apiException->getMessage()
            ]);
            
            $instance->update([
                'connection_state' => 'disconnected',
                'status' => 'disconnected',
                'disconnected_at' => now(),
                'last_ping' => now()
            ]);
            
            return $this->response([
                'connected' => false,
                'state' => 'disconnected',
                'instance_data' => [],
                'is_online' => false,
                'error_message' => 'Erro de comunicação com a Evolution API',
                'api_error' => $apiException->getMessage()
            ]);
        }
        
    } catch (Exception $e) {
        Log::error('❌ Erro ao obter status de conexão: ' . $e->getMessage(), [
            'instance_name' => $instanceName,
            'user_id' => auth()->id(),
            'trace' => $e->getTraceAsString()
        ]);
        
        return $this->response([
            'error' => 'Erro ao obter status de conexão',
            'message' => $e->getMessage(),
            'connected' => false,
            'state' => 'unknown'
        ], 500);
    }
}

    /**
     * 📱 OBTER QR CODE DA INSTÂNCIA
     * @param string $instanceName
     * @return JsonResponse
     */
    public function getQrCode(string $instanceName): JsonResponse
    {
        try {
            $instance = EvolutionInstance::where('instance_name', $instanceName)->firstOrFail();
            $this->evolutionApi->setInstance($instance);
            
            $qrDataFromApi = $this->evolutionApi->getQrCode();

            // if (!empty($qrDataFromApi['qrcode'])) {
            //     $instance->updateQrCode($qrDataFromApi);
            //     $this->chatRedis->publishEvolutionQrCode($instance->instance_name, $instance->qr_code);
            // }

            Log::info('✅ QR Code obtido', ['instance_name' => $instanceName]);
            return $this->response([
                'qrcode' => $qrDataFromApi['qrcode'] ?? null,
                'base64' => $qrDataFromApi['base64'] ?? null,
                'qr_code_expired' => $instance->qr_code_expired,
            ]);
        } catch (Exception $e) {
            $instance = EvolutionInstance::where('instance_name', $instanceName)->first();
            if ($instance) $instance->recordError('Falha ao obter QR Code: ' . $e->getMessage());
            Log::error('❌ Erro ao obter QR Code: ' . $e->getMessage(), ['instance_name' => $instanceName, 'trace' => $e->getTraceAsString()]);
            return $this->response(['error' => 'Erro ao obter QR Code: ' . $e->getMessage()], 500);
        }
    }

    /**
     * 🔌 CONECTAR INSTÂNCIA
     * @param string $instanceName
     * @return JsonResponse
     */
    public function connect(string $instanceName): JsonResponse
    {
        try {
            $instance = EvolutionInstance::where('instance_name', $instanceName)->firstOrFail();
            $this->evolutionApi->setInstance($instance);
            $response = $this->evolutionApi->connectInstance();

            Log::info('✅ Comando de conectar enviado', ['instance_name' => $instanceName, 'response' => $response]);
            return $this->response(['message' => 'Comando de conectar enviado', 'response' => $response]);
        } catch (Exception $e) {
            $instance = EvolutionInstance::where('instance_name', $instanceName)->first();
            if ($instance) $instance->recordError('Falha ao conectar instância: ' . $e->getMessage());
            Log::error('❌ Erro ao conectar instância: ' . $e->getMessage(), ['instance_name' => $instanceName, 'trace' => $e->getTraceAsString()]);
            return $this->response(['error' => 'Erro ao conectar instância: ' . $e->getMessage()], 500);
        }
    }

/**
 * 🔌 DESCONECTAR INSTÂNCIA - CORRIGIDO
 * @param string $instanceName
 * @return JsonResponse
 */
public function disconnect(string $instanceName): JsonResponse
{
    try {
        $instance = EvolutionInstance::where('instance_name', $instanceName)->firstOrFail();
        $this->evolutionApi->setInstance($instance);
        
        // ✅ CHAMAR O MÉTODO CORRETO
        $response = $this->evolutionApi->disconnectInstance();
        
        // ✅ ATUALIZAR STATUS NO BANCO
        $instance->markAsDisconnected('Manual disconnect');

        Log::info('✅ Comando de desconectar enviado', [
            'instance_name' => $instanceName, 
            'response' => $response
        ]);
        
        return $this->response([
            'success' => true,
            'message' => 'Comando de desconectar enviado', 
            'response' => $response
        ]);
        
    } catch (Exception $e) {
        $instance = EvolutionInstance::where('instance_name', $instanceName)->first();
        if ($instance) {
            $instance->recordError('Falha ao desconectar instância: ' . $e->getMessage());
        }
        
        Log::error('❌ Erro ao desconectar instância: ' . $e->getMessage(), [
            'instance_name' => $instanceName, 
            'trace' => $e->getTraceAsString()
        ]);
        
        return $this->response([
            'success' => false,
            'error' => 'Erro ao desconectar instância: ' . $e->getMessage()
        ], 500);
    }
}

    /**
     * 🧪 TESTAR CONEXÃO DA INSTÂNCIA
     * @param string $instanceName
     * @return JsonResponse
     */
    public function testConnection(string $instanceName): JsonResponse
    {
        try {
            $instance = EvolutionInstance::where('instance_name', $instanceName)->firstOrFail();
            $this->evolutionApi->setInstance($instance);
            $response = $this->evolutionApi->testConnection(); // Retorna sucesso/falha/tempo

            Log::info('✅ Teste de conexão Evolution realizado', ['instance_name' => $instanceName, 'response' => $response]);
            return $this->response($response); // Retorna a resposta completa do testConnection
        } catch (Exception $e) {
            $instance = EvolutionInstance::where('instance_name', $instanceName)->first();
            if ($instance) $instance->recordError('Falha no teste de conexão: ' . $e->getMessage());
            Log::error('❌ Erro no teste de conexão Evolution: ' . $e->getMessage(), ['instance_name' => $instanceName, 'trace' => $e->getTraceAsString()]);
            return $this->response(['success' => false, 'message' => 'Erro no teste de conexão: ' . $e->getMessage()], 500);
        }
    }

    /**
     * 📊 OBTER ESTATÍSTICAS DA INSTÂNCIA
     * @param string $instanceName
     * @return JsonResponse
     */
    public function stats(string $instanceName): JsonResponse
    {
        try {
            $instance = EvolutionInstance::where('instance_name', $instanceName)->firstOrFail();
            return $this->response([
                'messages_sent_today' => $instance->messages_sent_today,
                'messages_received_today' => $instance->messages_received_today,
                'uptime_percentage' => $instance->uptime_percentage,
                'status' => $instance->status,
                'connection_state' => $instance->connection_state,
                'last_ping' => $instance->last_ping?->toISOString(),
            ]);
        } catch (Exception $e) {
            Log::error('❌ Erro ao obter estatísticas da instância Evolution: ' . $e->getMessage(), ['instance_name' => $instanceName, 'trace' => $e->getTraceAsString()]);
            return $this->response(['error' => 'Erro ao obter estatísticas: ' . $e->getMessage()], 500);
        }
    }

    /**
     * 🔄 SINCRONIZAR DADOS DA INSTÂNCIA
     * @param Request $request
     * @param string $instanceName
     * @return JsonResponse
     */
    public function sync(Request $request, string $instanceName): JsonResponse
    {
        $instance = EvolutionInstance::where('instance_name', $instanceName)->firstOrFail();

        $validator = Validator::make($request->all(), [
            'type' => 'required|string|in:full,incremental,messages,contacts,profiles,chats',
            'options' => 'nullable|array',
        ]);

        if ($validator->fails()) {
            return $this->response(['errors' => $validator->errors()], 422);
        }

        try {
            $this->evolutionSync->setInstance($instance);
            $syncResult = $this->evolutionSync->sync($request->type, $request->options ?? []);

            Log::info('✅ Sincronização Evolution iniciada', ['instance_name' => $instanceName, 'type' => $request->type]);
            return $this->response(['message' => 'Sincronização iniciada com sucesso', 'result' => $syncResult]);
        } catch (Exception $e) {
            $instance->recordError('Falha na sincronização: ' . $e->getMessage());
            Log::error('❌ Erro na sincronização Evolution: ' . $e->getMessage(), ['instance_name' => $instanceName, 'type' => $request->type, 'trace' => $e->getTraceAsString()]);
            return $this->response(['error' => 'Erro na sincronização: ' . $e->getMessage()], 500);
        }
    }

    /**
     * 🕸️ CONFIGURAR WEBHOOK NA EVOLUTION API
     * @param string $instanceName
     * @return JsonResponse
     */
    public function setupWebhook(string $instanceName): JsonResponse
    {
        try {
            $instance = EvolutionInstance::where('instance_name', $instanceName)->firstOrFail();
            $this->evolutionApi->setInstance($instance);
            $response = $this->evolutionApi->setupWebhook();

            if (($response['success'] ?? false) === true) {
                $instance->update(['webhook_active' => true]);
            } else {
                // Se a API não reportar sucesso, o webhook pode não estar ativo
                Log::warning('⚠️ Webhook setup reported non-success by API.', ['instance_name' => $instanceName, 'response' => $response]);
            }

            Log::info('✅ Webhook configurado na Evolution API', ['instance_name' => $instanceName, 'response' => $response]);
            return $this->response(['success' => true, 'message' => 'Webhook configurado com sucesso', 'response' => $response]);
        } catch (Exception $e) {
            $instance = EvolutionInstance::where('instance_name', $instanceName)->first();
            if ($instance) $instance->recordError('Falha ao configurar webhook: ' . $e->getMessage());
            Log::error('❌ Erro ao configurar webhook na Evolution API: ' . $e->getMessage(), ['instance_name' => $instanceName, 'trace' => $e->getTraceAsString()]);
            return $this->response(['success' => false, 'message' => 'Erro ao configurar webhook: ' . $e->getMessage()], 500);
        }
    }

    
}