<?php

namespace Modules\Chat\Http\Controllers\Api;

use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Modules\Core\Http\Controllers\ApiController;
use Modules\Chat\Models\EvolutionWebhook;
use Modules\Chat\Jobs\ProcessEvolutionWebhook;
use Exception;

class EvolutionWebhookController extends ApiController
{
    private const HIGH_PRIORITY_EVENTS = ['messages.upsert', 'messages.update', 'connection.update'];
    private const LOW_PRIORITY_EVENTS = ['presence.update', 'chats.set', 'contacts.set'];

    public function handle(Request $request, string $instanceName): JsonResponse
    {
        try {
            $payload = $request->all();
            $eventType = $this->normalizeEventType($payload['event'] ?? 'unknown');

            $webhook = EvolutionWebhook::create([
                'instance_name' => $instanceName,
                'event_type' => $eventType,
                'payload' => $payload,
                'received_at' => now(),
                'status' => 'pending',
                'processing_attempts' => 0,
            ]);

            $queueName = $this->getQueueByPriority($eventType);
            ProcessEvolutionWebhook::dispatch($webhook)->onQueue($queueName);

            Log::info('Webhook registrado e despachado', [
                'webhook_db_id' => $webhook->id,
                'instance_name' => $instanceName,
                'event_type' => $eventType,
                'queue' => $queueName
            ]);

            return $this->response(['status' => 'received'], 200);

        } catch (Exception $e) {
            Log::error('Erro ao processar webhook Evolution', [
                'instance' => $instanceName,
                'error' => $e->getMessage(),
                'payload' => $request->all()
            ]);

            return $this->response(['error' => 'Erro interno'], 500);
        }
    }

    public function handleByEvent(Request $request, string $instanceName, string $eventName): JsonResponse
    {
        $payload = $request->all();
        $payload['event'] = str_replace('-', '.', strtolower($eventName));
        
        Log::info('Webhook Evolution BY EVENT recebido', [
            'instance' => $instanceName,
            'event_name_from_url' => $eventName,
            'source_ip' => $request->ip(),
            'payload_size' => strlen(json_encode($payload))
        ]);

        $request->merge($payload);
        return $this->handle($request, $instanceName);
    }

    public function verify(Request $request, string $instanceName): JsonResponse
    {
        return $this->response(['status' => 'ok', 'instance' => $instanceName]);
    }

    private function normalizeEventType(string $event): string
    {
        return str_replace(['-', '_'], '.', strtolower(trim($event)));
    }

    private function getQueueByPriority(string $eventType): string
    {
        if (in_array($eventType, self::HIGH_PRIORITY_EVENTS)) {
            return 'webhooks-high';
        }
        
        if (in_array($eventType, self::LOW_PRIORITY_EVENTS)) {
            return 'webhooks-low';
        }

        return 'webhooks';
    }
}