<?php

namespace Modules\Chat\Http\Controllers\Api;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Modules\Core\Http\Controllers\ApiController;
use Modules\Chat\Models\MessageAttachment;
use Exception;

class MediaProxyController extends ApiController
{
    /**
     * ✅ PROXY PARA SERVIR MÍDIAS DO WHATSAPP SEM ERRO 403
     */
    public function proxy(MessageAttachment $attachment): Response
    {
        try {
            // Se já temos o arquivo local, redirecionar
            if ($attachment->isDownloaded()) {
                return response()->json([
                    'redirect_url' => url('storage/' . $attachment->file_path)
                ]);
            }
            
            // Cache da mídia por 1 hora
            $cacheKey = "media_proxy:{$attachment->id}";
            $cachedData = Cache::get($cacheKey);
            
            if ($cachedData) {
                return response($cachedData['content'])
                    ->header('Content-Type', $cachedData['mime_type'])
                    ->header('Content-Length', strlen($cachedData['content']))
                    ->header('Cache-Control', 'public, max-age=3600')
                    ->header('Access-Control-Allow-Origin', '*');
            }
            
            // Fazer download da URL externa
            $url = $attachment->whatsapp_media_url ?? $attachment->external_url;
            
            if (!$url) {
                return response('Mídia não encontrada', 404);
            }
            
            Log::info('📥 Fazendo proxy de mídia', [
                'attachment_id' => $attachment->id,
                'url_preview' => substr($url, 0, 50) . '...'
            ]);
            
            $response = Http::timeout(30)
                ->withHeaders([
                    'User-Agent' => 'Mozilla/5.0 (compatible; WhatsApp-Media-Proxy/1.0)',
                    'Accept' => '*/*',
                    'Referer' => 'https://web.whatsapp.com/',
                ])
                ->get($url);
            
            if (!$response->successful()) {
                Log::warning('❌ Falha no proxy de mídia', [
                    'attachment_id' => $attachment->id,
                    'status' => $response->status()
                ]);
                return response('Erro ao carregar mídia', $response->status());
            }
            
            $content = $response->body();
            $mimeType = $attachment->mime_type ?? 'application/octet-stream';
            
            // Cache por 1 hora
            Cache::put($cacheKey, [
                'content' => $content,
                'mime_type' => $mimeType
            ], 3600);
            
            Log::info('✅ Mídia servida via proxy', [
                'attachment_id' => $attachment->id,
                'size_mb' => round(strlen($content) / 1024 / 1024, 2)
            ]);
            
            return response($content)
                ->header('Content-Type', $mimeType)
                ->header('Content-Length', strlen($content))
                ->header('Cache-Control', 'public, max-age=3600')
                ->header('Access-Control-Allow-Origin', '*');
                
        } catch (Exception $e) {
            Log::error('❌ Erro no proxy de mídia', [
                'attachment_id' => $attachment->id,
                'error' => $e->getMessage()
            ]);
            
            return response('Erro interno do servidor', 500);
        }
    }
}