<?php
// modules/Chat/app/Http/Resources/ConversationResource.php
namespace Modules\Chat\Http\Resources;

use Illuminate\Http\Resources\Json\JsonResource;

class ConversationResource extends JsonResource
{
    public function toArray($request)
    {
        return [
            'id' => $this->id,
            'contact_name' => $this->getDisplayName(),
            'last_message' => $this->lastMessage?->content,
            'type' => $this->type,
            'status' => $this->status,
            'unread_count' => $this->unread_count ?? 0,
            'last_message_at' => $this->last_message_at,
            'created_at' => $this->created_at,
            'assignedUser' => $this->whenLoaded('assignedUser', function () {
                return [
                    'id' => $this->assignedUser->id,
                    'name' => $this->assignedUser->name,
                    'email' => $this->assignedUser->email,
                    'avatar' => $this->assignedUser->avatar
                ];
            }),
            'assignedUsers' => $this->whenLoaded('assignedUsers', function () {
                return $this->assignedUsers->map(function ($user) {
                    return [
                        'user' => [
                            'id' => $user->id,
                            'name' => $user->name,
                            'email' => $user->email,
                            'avatar' => $user->avatar ?? null
                        ],
                        'role' => $user->pivot->role,
                        'is_primary' => $user->pivot->is_primary
                    ];
                });
            }),
            'is_pinned' => $this->is_pinned,
            'is_muted' => $this->is_muted,
            'processed_avatar' => $this->getProcessedAvatar(),
            'processed_display_name' => $this->getDisplayName()
        ];
    }
    
    private function getDisplayName(): string
    {
        if ($this->type === 'internal') {
            return data_get($this->metadata, 'subject', 'Chat Interno');
        }
        
        return $this->contact_name ?: 
               $this->whatsapp_profile_name ?: 
               $this->formatPhoneForDisplay($this->contact_phone ?: '') ?: 
               'Sem nome';
    }
    
    private function getProcessedAvatar(): ?string
    {
        if ($this->type === 'internal') {
            $firstParticipant = $this->participants?->first();
            return $firstParticipant?->avatar;
        }
        
        return $this->contact?->best_profile_photo ?? 
               $this->whatsapp_profile_picture ?? 
               null;
    }
    
    private function formatPhoneForDisplay(string $phone): string
    {
        $clean = preg_replace('/[^0-9]/', '', $phone);
        
        if (strlen($clean) === 13 && str_starts_with($clean, '55')) {
            $ddd = substr($clean, 2, 2);
            $numero = substr($clean, 4);
            if (strlen($numero) === 9) {
                return "+55 ({$ddd}) " . substr($numero, 0, 5) . "-" . substr($numero, 5);
            }
        }
        
        return $phone;
    }
}