<?php

namespace Modules\Chat\Http\Traits;

use Carbon\Carbon;
use Modules\Chat\Models\Conversation;
use Exception;
use Illuminate\Support\Facades\Log;

trait FormatsConversations
{
    /**
     * ✅ FORMATAR CONVERSA - VERSÃO ROBUSTA COM TRATAMENTO DE ERROS
     */
    protected function formatConversation(Conversation $conversation, int $userId): array
    {
        try {
            $isInternal = $conversation->type === 'internal';
            
            // ✅ DADOS BÁSICOS COMUNS
            $baseData = [
                'id' => $conversation->id,
                'type' => $conversation->type,
                'status' => $conversation->status ?? 'active',
                'last_message' => $conversation->lastMessage?->content ?? '',
                'last_message_at' => $this->formatDate($conversation->lastMessage?->sent_at ?? $conversation->updated_at),
                'unread_count' => $conversation->unread_count ?? 0,
                'created_at' => $this->formatDate($conversation->created_at),
                'assigned_user_id' => $conversation->assigned_user_id,
                'closed_at' => $this->formatDate($conversation->closed_at),
                'closed_by' => $conversation->closed_by,
                'is_pinned' => (bool) $conversation->is_pinned,
                'is_muted' => (bool) $conversation->is_muted,
                
                // ✅ DADOS DO USUÁRIO ATRIBUÍDO
                'assignedUser' => $conversation->assignedUser ? [
                    'id' => $conversation->assignedUser->id,
                    'name' => $conversation->assignedUser->name,
                    'email' => $conversation->assignedUser->email,
                    'avatar' => $conversation->assignedUser->avatar ?? null
                ] : null,
                
                // ✅ INFORMAÇÕES DE VISIBILIDADE SIMPLIFICADAS
                'visibility' => [
                    'is_assigned_to_user' => $conversation->assigned_user_id === $userId,
                    'can_user_view' => $conversation->canUserView($userId),
                    'is_unassigned' => !$conversation->assigned_user_id
                ],
            ];
            
            if ($isInternal) {
                return $this->formatInternalConversation($conversation, $baseData, $userId);
            } else {
                return $this->formatExternalConversation($conversation, $baseData);
            }
            
        } catch (Exception $e) {
            Log::error('Erro ao formatar conversa', [
                'conversation_id' => $conversation->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Retorno básico em caso de erro
            return [
                'id' => $conversation->id,
                'type' => $conversation->type,
                'status' => $conversation->status ?? 'active',
                'contact_name' => 'Erro ao carregar',
                'error' => true
            ];
        }
    }

    /**
     * ✅ FORMATAR CONVERSA INTERNA - SEM USAR PIVOT
     */
    private function formatInternalConversation(Conversation $conversation, array $baseData, int $userId): array
    {
        $subject = data_get($conversation->metadata, 'subject', 'Chat Interno');
        $participants = data_get($conversation->metadata, 'participants', []);
        
        // ✅ BUSCAR NOMES DOS PARTICIPANTES (EXCETO O USUÁRIO ATUAL)
        $otherParticipantIds = array_filter($participants, fn($id) => $id != $userId);
        $participantNames = '';
        
        if (!empty($otherParticipantIds)) {
            try {
                $users = \Modules\Users\Models\User::whereIn('id', $otherParticipantIds)
                    ->pluck('name')
                    ->toArray();
                $participantNames = implode(', ', $users);
            } catch (Exception $e) {
                Log::warning('Erro ao buscar nomes dos participantes', [
                    'conversation_id' => $conversation->id,
                    'participant_ids' => $otherParticipantIds,
                    'error' => $e->getMessage()
                ]);
                $participantNames = 'Participantes';
            }
        }
        
        return array_merge($baseData, [
            'subject' => $subject,
            'contact_name' => $participantNames ?: $subject,
            'description' => data_get($conversation->metadata, 'description'),
            'internal_type' => data_get($conversation->metadata, 'internal_type', 'discussion'),
            'priority' => data_get($conversation->metadata, 'priority', 'normal'),
            'participants' => $participants,
            'participant_names' => $participantNames,
            'last_message_type' => 'internal',
            'created_by' => data_get($conversation->metadata, 'created_by'),
            'phone' => null,
            'channel_type' => 'internal',
            'processed_display_name' => $participantNames ?: $subject,
            'contact' => null,
            'company_name' => null,
            'company_id' => null,
        ]);
    }

    /**
     * ✅ FORMATAR CONVERSA EXTERNA - COM TRATAMENTO ROBUSTO DE ERROS
     */
    private function formatExternalConversation(Conversation $conversation, array $baseData): array
    {
        $contactName = 'Contato sem nome';
        $profilePhoto = null;
        $isWhatsAppBusiness = false;
        $phoneNumber = '';
        $contactEmail = null;
        $companyName = null;
        $companyId = null;

        // ✅ PROCESSAR DADOS DO CONTATO COM TRATAMENTO DE ERRO
        if ($conversation->contact) {
            try {
                $contact = $conversation->contact;
                
                // ✅ NOME DO CONTATO - TENTAR VÁRIAS OPÇÕES
                if (!empty($contact->first_name)) {
                    $contactName = trim($contact->first_name . ' ' . ($contact->last_name ?? ''));
                } elseif (!empty($contact->name)) {
                    $contactName = $contact->name;
                } elseif (!empty($contact->email)) {
                    $contactName = $contact->email;
                }
                
                // ✅ FOTO DE PERFIL - TENTAR VÁRIAS OPÇÕES
                $profilePhoto = $contact->whatsapp_profile_photo_url ?? 
                               $contact->avatar ?? 
                               null;
                
                // ✅ DADOS ADICIONAIS
                $isWhatsAppBusiness = (bool)($contact->is_whatsapp_business ?? false);
                $contactEmail = $contact->email;
                
                // ✅ TELEFONE - TENTAR VÁRIAS OPÇÕES
                if (!empty($contact->whatsapp_number)) {
                    $phoneNumber = $contact->whatsapp_number;
                } elseif (method_exists($contact, 'phones') && $contact->phones()->exists()) {
                    $firstPhone = $contact->phones()->first();
                    if ($firstPhone && !empty($firstPhone->number)) {
                        $phoneNumber = $firstPhone->number;
                    }
                }

                // ✅ EMPRESA - MÉTODO MAIS SEGURO
                try {
                    if (method_exists($contact, 'companies')) {
                        $companies = $contact->companies;
                        if ($companies && $companies->isNotEmpty()) {
                            $primaryCompany = $companies->first();
                            if ($primaryCompany) {
                                $companyName = $primaryCompany->name;
                                $companyId = $primaryCompany->id;
                            }
                        }
                    }
                } catch (Exception $e) {
                    Log::warning('Erro ao buscar empresa do contato', [
                        'contact_id' => $contact->id,
                        'error' => $e->getMessage()
                    ]);
                }
                
            } catch (Exception $e) {
                Log::warning('Erro ao processar dados do contato', [
                    'conversation_id' => $conversation->id,
                    'contact_id' => $conversation->contact_id,
                    'error' => $e->getMessage()
                ]);
            }
        }
        
        // ✅ FALLBACKS PARA NOME DO CONTATO
        if ($contactName === 'Contato sem nome') {
            if (!empty($conversation->whatsapp_profile_name)) {
                $contactName = $conversation->whatsapp_profile_name;
            } elseif (!empty($conversation->contact_phone)) {
                $contactName = $this->formatPhoneForDisplay($conversation->contact_phone);
            } elseif (!empty($phoneNumber)) {
                $contactName = $this->formatPhoneForDisplay($phoneNumber);
            } else {
                $contactName = "Conversa Externa #{$conversation->id}";
            }
        }
        
        // ✅ USAR TELEFONE DA CONVERSA SE NÃO TEM DO CONTATO
        if (empty($phoneNumber) && !empty($conversation->contact_phone)) {
            $phoneNumber = $conversation->contact_phone;
        }
        
        return array_merge($baseData, [
            'contact_name' => $contactName,
            'phone' => $phoneNumber,
            'avatar_url' => $profilePhoto, 
            'is_whatsapp_business' => $isWhatsAppBusiness, 
            'whatsapp_profile_picture' => $conversation->whatsapp_profile_picture,
            'last_message_type' => 'external',
            'channel_type' => $conversation->chatChannel?->type ?? 'evolution_api',
            'subject' => null,
            'participants' => [],
            'company_name' => $companyName,
            'company_id' => $companyId,
            'contact' => $conversation->contact ? [
                'id' => $conversation->contact->id,
                'first_name' => $conversation->contact->first_name ?? null,
                'last_name' => $conversation->contact->last_name ?? null,
                'name' => $conversation->contact->name ?? null,
                'whatsapp_number' => $phoneNumber,
                'email' => $contactEmail,
                'whatsapp_profile_photo_url' => $profilePhoto,
                'company' => $companyName ? [
                    'id' => $companyId,
                    'name' => $companyName,
                ] : null,
            ] : null,
            'processed_display_name' => $contactName
        ]);
    }

    // ===== MÉTODOS AUXILIARES =====

    protected function formatPhoneForDisplay(string $phone): string
    {
        try {
            $clean = preg_replace('/[^0-9]/', '', $phone);
            
            if (strlen($clean) === 13 && str_starts_with($clean, '55')) {
                $ddd = substr($clean, 2, 2);
                $numero = substr($clean, 4);
                if (strlen($numero) === 9) {
                    return "+55 ({$ddd}) " . substr($numero, 0, 5) . "-" . substr($numero, 5);
                }
            } elseif (strlen($clean) === 11) {
                $ddd = substr($clean, 0, 2);
                $numero = substr($clean, 2);
                if (strlen($numero) === 9) {
                    return "({$ddd}) " . substr($numero, 0, 5) . "-" . substr($numero, 5);
                }
            } elseif (strlen($clean) === 10) {
                $ddd = substr($clean, 0, 2);
                $numero = substr($clean, 2);
                return "({$ddd}) " . substr($numero, 0, 4) . "-" . substr($numero, 4);
            }
            
            return $phone;
        } catch (Exception $e) {
            Log::warning('Erro ao formatar telefone', [
                'phone' => $phone,
                'error' => $e->getMessage()
            ]);
            return $phone;
        }
    }

    protected function formatDate($date): ?string
    {
        if (!$date) {
            return null;
        }

        try {
            if (is_string($date)) {
                $carbonDate = Carbon::parse($date);
                return $carbonDate->toISOString();
            }

            if ($date instanceof Carbon || $date instanceof \DateTime) {
                return $date->toISOString();
            }

            return Carbon::parse($date)->toISOString();
        } catch (Exception $e) {
            Log::warning('Erro ao formatar data', [
                'date' => $date,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }
}