<?php

namespace Modules\Chat\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Http;
use Modules\Chat\Models\MessageAttachment;
use Exception;

class DownloadWhatsAppMedia implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected MessageAttachment $attachment;
    public int $tries = 3;
    public int $timeout = 60;

    public function __construct(MessageAttachment $attachment)
    {
        $this->attachment = $attachment;
        $this->onQueue('media-downloads');
    }

    public function handle(): void
    {
        try {
            if ($this->attachment->isDownloaded()) {
                Log::info('Mídia já baixada', ['attachment_id' => $this->attachment->id]);
                return;
            }

            Log::info('Iniciando download de mídia WhatsApp', [
                'attachment_id' => $this->attachment->id,
                'file_name' => $this->attachment->file_name
            ]);

            $this->attachment->update(['download_status' => 'downloading']);

            $response = Http::timeout(30)
                ->withHeaders([
                    'User-Agent' => 'Mozilla/5.0 (compatible; WhatsApp-Media-Downloader/1.0)',
                    'Accept' => '*/*',
                    'Accept-Encoding' => 'identity',
                ])
                ->get($this->attachment->whatsapp_media_url);
            
            if (!$response->successful()) {
                throw new Exception("HTTP {$response->status()}: Falha ao baixar mídia");
            }

            $fileData = $response->body();
            if (empty($fileData)) {
                throw new Exception('Dados de mídia vazios');
            }

            $this->validateMediaData($fileData);

            $fileName = time() . '_' . uniqid() . '.' . $this->getFileExtension();
            $filePath = 'chat-media/' . $fileName;
            
            $saved = Storage::disk('public')->put($filePath, $fileData);
            
            if (!$saved) {
                throw new Exception('Falha ao salvar arquivo no storage');
            }
            
            $this->attachment->update([
                'file_path' => $filePath,
                'download_status' => 'completed',
                'downloaded_at' => now(),
                'download_error' => null,
                'file_size' => strlen($fileData)
            ]);

            Log::info('Mídia WhatsApp baixada com sucesso', [
                'attachment_id' => $this->attachment->id,
                'file_path' => $filePath,
                'file_size_mb' => round(strlen($fileData) / 1024 / 1024, 2)
            ]);

        } catch (Exception $e) {
            Log::error('Erro ao baixar mídia WhatsApp', [
                'attachment_id' => $this->attachment->id,
                'error' => $e->getMessage()
            ]);
            
            $this->attachment->markDownloadFailed($e->getMessage());
            throw $e;
        }
    }

    protected function validateMediaData(string $data): void
    {
        if (strlen($data) < 10) {
            throw new Exception('Dados de mídia muito pequenos');
        }
        
        $header = bin2hex(substr($data, 0, 4));
        
        $validHeaders = [
            'ffd8ff' => 'JPEG', '89504e' => 'PNG', '474946' => 'GIF', '524946' => 'WEBP'
        ];
        
        foreach ($validHeaders as $magic => $type) {
            if (str_starts_with($header, $magic)) {
                Log::debug("Mídia {$type} válida detectada", [
                    'header' => $header, 'size' => strlen($data)
                ]);
                return;
            }
        }
        
        if ($this->attachment->file_type === 'audio') {
            Log::warning('Header de áudio não reconhecido, mas prosseguindo', [
                'header' => $header, 'size' => strlen($data)
            ]);
            return;
        }
        
        Log::warning('Header de mídia não reconhecido', [
            'header' => $header, 'file_type' => $this->attachment->file_type
        ]);
    }

    protected function getFileExtension(): string
    {
        $mapping = [
            'image/jpeg' => 'jpg', 'image/png' => 'png', 'image/gif' => 'gif',
            'image/webp' => 'webp', 'audio/ogg' => 'ogg', 'audio/mpeg' => 'mp3',
            'audio/mp4' => 'm4a', 'video/mp4' => 'mp4'
        ];
        
        return $mapping[$this->attachment->mime_type] ?? 'bin';
    }

    public function failed(Exception $exception): void
    {
        Log::error('Job de download falhou permanentemente', [
            'attachment_id' => $this->attachment->id,
            'error' => $exception->getMessage()
        ]);
        
        $this->attachment->markDownloadFailed(
            'Download falhou após ' . $this->tries . ' tentativas: ' . $exception->getMessage()
        );
    }
}