<?php

namespace Modules\Chat\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Modules\Contacts\Models\Contact;
use Modules\Chat\Models\EvolutionInstance;
use Modules\Chat\Models\Conversation;
use Modules\Chat\Services\ContactProfilePhotoService;
use Exception;

class FetchContactPhotoJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $timeout = 120;
    public $tries = 2;

    protected Contact $contact;
    protected string $phoneNumber;
    protected int $conversationId;
    protected bool $forceUpdate;

    public function __construct(Contact $contact, string $phoneNumber, int $conversationId, bool $forceUpdate = false)
    {
        $this->contact = $contact;
        $this->phoneNumber = $phoneNumber;
        $this->conversationId = $conversationId;
        $this->forceUpdate = $forceUpdate;
    }

    public function handle(ContactProfilePhotoService $photoService): void
    {
        try {
            $realPhoneNumber = $this->getRealPhoneNumber();
            
            Log::info('📸 [JOB] Iniciando busca de foto', [
                'contact_id' => $this->contact->id,
                'contact_name' => $this->contact->first_name,
                'phone' => $realPhoneNumber,
                'conversation_id' => $this->conversationId,
                'attempt' => $this->attempts()
            ]);

            $conversation = Conversation::find($this->conversationId);
            if (!$conversation) {
                throw new Exception('Conversa não encontrada');
            }

            $instance = EvolutionInstance::where('instance_name', $conversation->evolution_instance_name)->first();
            if (!$instance) {
                throw new Exception('Instância Evolution não encontrada');
            }

            $success = $photoService->fetchAndSaveContactProfilePhoto(
                $this->contact, 
                $realPhoneNumber, 
                $instance
            );

            if ($success) {
                Log::info('✅ [JOB] Foto salva com sucesso', [
                    'contact_id' => $this->contact->id,
                    'phone' => $realPhoneNumber,
                    'conversation_id' => $this->conversationId
                ]);
            } else {
                Log::info('📸 [JOB] Nenhuma foto encontrada', [
                    'contact_id' => $this->contact->id,
                    'phone' => $realPhoneNumber,
                    'conversation_id' => $this->conversationId
                ]);
            }

        } catch (Exception $e) {
            Log::error('❌ [JOB] Erro ao buscar foto', [
                'contact_id' => $this->contact->id,
                'phone' => $this->phoneNumber,
                'error' => $e->getMessage()
            ]);
            throw $e;
        } finally {
            Cache::forget("photo_job_pending_{$this->contact->id}");
        }
    }

    private function getRealPhoneNumber(): string
    {
        $cleanPhone = preg_replace('/[^0-9]/', '', $this->phoneNumber);
        
        if (strlen($cleanPhone) >= 10) {
            return $cleanPhone;
        }

        try {
            $conversation = Conversation::find($this->conversationId);
            
            if ($conversation) {
                if ($conversation->whatsapp_remote_jid) {
                    $phoneFromJid = preg_replace('/[^0-9]/', '', $conversation->whatsapp_remote_jid);
                    if (strlen($phoneFromJid) >= 10) {
                        return $phoneFromJid;
                    }
                }
                
                if ($conversation->contact_phone) {
                    $phoneFromContact = preg_replace('/[^0-9]/', '', $conversation->contact_phone);
                    if (strlen($phoneFromContact) >= 10) {
                        return $phoneFromContact;
                    }
                }
            }
        } catch (Exception $e) {
            Log::warning('⚠️ Erro ao buscar número real', [
                'conversation_id' => $this->conversationId,
                'error' => $e->getMessage()
            ]);
        }

        return $cleanPhone;
    }

    public function failed(Exception $exception): void
    {
        Log::error('❌ [JOB] Job falhou', [
            'contact_id' => $this->contact->id,
            'phone' => $this->phoneNumber,
            'error' => $exception->getMessage()
        ]);

        Cache::forget("photo_job_pending_{$this->contact->id}");
    }
}