<?php

namespace Modules\Chat\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Modules\Chat\Models\EvolutionWebhook;
use Modules\Chat\Services\EvolutionWebhookProcessorService;
use Carbon\Carbon;
use Exception;

class ProcessEvolutionWebhook implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected EvolutionWebhook $webhook;
    public int $tries = 5;
    public int $timeout = 120;
    public array $backoff = [10, 30, 60, 180, 300];

    public function __construct(EvolutionWebhook $webhook)
    {
        $this->webhook = $webhook;
        $this->onQueue('webhooks');
    }

    public function handle(EvolutionWebhookProcessorService $processor): void
    {
        Log::info('Job ProcessEvolutionWebhook iniciando', [
            'webhook_db_id' => $this->webhook->id,
            'instance_name' => $this->webhook->instance_name,
            'event_type' => $this->webhook->event_type,
            'attempt' => $this->getCurrentAttemptNumber()
        ]);

        try {
            $this->markAsProcessing();
            
            DB::beginTransaction();
            $result = $processor->processWebhook($this->webhook);
            $this->markAsProcessed($result);
            DB::commit();

            Log::info('Job ProcessEvolutionWebhook concluído', [
                'webhook_db_id' => $this->webhook->id,
                'result' => $result['action'] ?? 'processed'
            ]);

        } catch (Exception $e) {
            DB::rollBack();
            
            Log::error('Erro no job ProcessEvolutionWebhook', [
                'webhook_db_id' => $this->webhook->id,
                'error' => $e->getMessage(),
                'attempt' => $this->getCurrentAttemptNumber()
            ]);

            $this->markAsFailed($e->getMessage());
            throw $e;
        }
    }

    public function failed(\Throwable $exception): void
    {
        Log::error('Job ProcessEvolutionWebhook falhou permanentemente', [
            'webhook_db_id' => $this->webhook->id,
            'error' => $exception->getMessage()
        ]);

        $this->markAsFailed('Job permanently failed: ' . $exception->getMessage());
    }

    private function markAsProcessing(): void
    {
        $this->webhook->update([
            'status' => 'processing',
            'processing_attempts' => ($this->webhook->processing_attempts ?? 0) + 1,
            'next_attempt_at' => $this->calculateNextAttempt()
        ]);
    }

    private function markAsProcessed(array $result = []): void
    {
        $updates = [
            'status' => 'processed',
            'processed_at' => Carbon::now(),
            'processing_result' => json_encode($result),
            'error_message' => null,
            'error_details' => null
        ];

        if (!empty($result['conversation_id'])) {
            $updates['conversation_id'] = $result['conversation_id'];
        }
        if (!empty($result['message_id'])) {
            $updates['message_id'] = $result['message_id'];
        }
        if (!empty($result['contact_id'])) {
            $updates['contact_id'] = $result['contact_id'];
        }

        $this->webhook->update($updates);
    }

    private function markAsFailed(string $message): void
    {
        $this->webhook->update([
            'status' => 'failed',
            'error_message' => substr($message, 0, 1000),
            'processed_at' => Carbon::now(),
            'next_attempt_at' => $this->calculateNextAttempt()
        ]);
    }

    private function calculateNextAttempt(): ?string
    {
        $attempts = (int)($this->webhook->processing_attempts ?? 0);
        $delay = $this->backoff[min($attempts, count($this->backoff) - 1)] ?? 60;
        
        return Carbon::now()->addSeconds($delay)->toDateTimeString();
    }

    private function getCurrentAttemptNumber(): int
    {
        return ((int)$this->webhook->processing_attempts) + 1;
    }
}