<?php

namespace Modules\Chat\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Modules\Chat\Models\Message;
use Modules\Chat\Models\Conversation;
use Modules\Chat\Services\EvolutionApiService;
use Modules\Users\Models\User;
use Exception;

class SendWhatsAppMessageJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    private const STATUS_MAPPING = [
        'PENDING' => 'pending',
        'SENT' => 'sent', 
        'DELIVERED' => 'delivered',
        'READ' => 'read',
        'FAILED' => 'failed'
    ];

    public int $tries = 3;
    public int $timeout = 60;
    public array $backoff = [10, 30, 60];

    public function __construct(
        public Message $message,
        public Conversation $conversation,
        public ?string $attachmentPath = null
    ) {
        $this->onQueue('whatsapp-messages');
        $this->delay = now()->addSeconds($this->conversation->id % 5);
    }

    public function handle(): void
    {
        try {
            $instance = $this->getEvolutionInstance();
            $this->validateInstance($instance);

            $targetNumber = $this->getTargetNumber();
            $this->validateTargetNumber($targetNumber);

            // ✅ VALIDAÇÃO OPCIONAL - NÃO FALHA MAIS
            $whatsappValidation = $this->validateWhatsAppNumber($targetNumber);
            
            $evolutionApi = app(EvolutionApiService::class)->setInstance($instance);
            $targetJid = $whatsappValidation['jid'] ?? $targetNumber;

            $result = $this->sendMessage($evolutionApi, $targetJid);
            $this->processResponse($result, $whatsappValidation);

            Log::info('✅ Mensagem WhatsApp enviada com sucesso', [
                'message_id' => $this->message->id,
                'conversation_id' => $this->conversation->id,
                'whatsapp_id' => $result['key']['id'] ?? null,
                'target_number' => substr($targetNumber, 0, 4) . '****'
            ]);

        } catch (Exception $e) {
            $this->failWithError($e->getMessage());
        }
    }

    private function validateInstance($instance): void
    {
        if (!$instance || !$instance->is_online) {
            throw new Exception('Instância Evolution não encontrada ou offline');
        }
    }

    private function validateTargetNumber(?string $targetNumber): void
    {
        if (!$targetNumber) {
            throw new Exception('Número de destino não encontrado');
        }
    }

    // ✅ MÉTODO CORRIGIDO - PRINCIPAL MUDANÇA
    private function getTargetNumber(): ?string
    {
        $remoteJid = $this->conversation->whatsapp_remote_jid;
        
        // ✅ SE REMOTE_JID É UM LID OU NÚMERO INVÁLIDO, USAR CONTATO
        if ($remoteJid && $this->isInvalidNumber($remoteJid)) {
            Log::info('🔄 Remote JID é LID/inválido, buscando número real do contato', [
                'conversation_id' => $this->conversation->id,
                'remote_jid' => substr($remoteJid, 0, 10) . '****',
                'contact_id' => $this->conversation->contact_id
            ]);
            
            $contactNumber = $this->getContactPhone();
            if ($contactNumber) {
                Log::info('✅ Número real encontrado no contato', [
                    'conversation_id' => $this->conversation->id,
                    'real_number' => substr($contactNumber, 0, 4) . '****'
                ]);
                return $contactNumber;
            }
            
            // Fallback para contact_phone da conversa
            $contactPhoneFromConversation = $this->safeFormatPhone($this->conversation->contact_phone);
            if ($contactPhoneFromConversation) {
                return $contactPhoneFromConversation;
            }
            
            throw new Exception('Não foi possível encontrar número válido para LID: ' . substr($remoteJid, 0, 10) . '****');
        }
        
        // ✅ USAR ORDEM SEGURA PARA JIDs VÁLIDOS
        $contactPhoneFromConversation = $this->safeFormatPhone($this->conversation->contact_phone);
        if ($contactPhoneFromConversation) {
            return $contactPhoneFromConversation;
        }
        
        $contactPhone = $this->getContactPhone();
        if ($contactPhone) {
            return $contactPhone;
        }
        
        if ($remoteJid) {
            return $this->extractPhoneFromJid($remoteJid);
        }
        
        return null;
    }

    // ✅ MÉTODO AUXILIAR PARA DETECTAR NÚMEROS INVÁLIDOS
    private function isInvalidNumber(string $jid): bool
    {
        if (str_contains($jid, '@lid')) {
            return true;
        }
        
        $number = preg_replace('/[^0-9]/', '', explode('@', $jid)[0]);
        
        // Números muito longos ou que começam com padrões estranhos
        if (strlen($number) > 15) return true;
        if (strlen($number) < 10) return true;
        
        // Padrões conhecidos de LIDs
        $lidPatterns = ['141283', '90744', '23648'];
        foreach ($lidPatterns as $pattern) {
            if (str_starts_with($number, $pattern)) return true;
        }
        
        return false;
    }

    // ✅ VERSÃO SEGURA DO formatPhone QUE ACEITA NULL
    private function safeFormatPhone(?string $phone): ?string
    {
        if (empty($phone)) {
            return null;
        }
        
        return $this->formatPhone($phone);
    }

    // ✅ VALIDAÇÃO QUE NÃO FALHA O JOB
    private function validateWhatsAppNumber(string $number): array
    {
        try {
            $evolutionApi = app(EvolutionApiService::class)->setInstance($this->getEvolutionInstance());
            $result = $evolutionApi->checkSingleWhatsAppNumber($number);
            
            if (!$result['exists']) {
                Log::warning('⚠️ Número WhatsApp não encontrado na validação, tentando enviar mesmo assim', [
                    'number' => substr($number, 0, 6) . '****',
                    'message_id' => $this->message->id,
                    'conversation_id' => $this->conversation->id,
                    'reason' => 'validation_returned_false_but_continuing'
                ]);
            }
            
            return $result;
        } catch (Exception $e) {
            Log::warning('⚠️ Erro na validação WhatsApp, continuando envio', [
                'error' => $e->getMessage(),
                'number' => substr($number, 0, 6) . '****',
                'message_id' => $this->message->id,
                'reason' => 'validation_error_but_continuing'
            ]);
            
            return ['exists' => true, 'number' => $number, 'jid' => null, 'validation_skipped' => true];
        }
    }

    private function prepareMessageContent(): string
    {
        $originalContent = $this->message->content;

        if (!$this->shouldAddUserPrefix()) {
            return $originalContent;
        }

        $userName = $this->getUserName();
        if (!$userName) {
            return $originalContent;
        }

        $prefix = "*{$userName}*";
        
        if (str_starts_with(trim($originalContent), $prefix)) {
            return $originalContent;
        }

        return $prefix . "\n" . $originalContent;
    }

    private function shouldAddUserPrefix(): bool
    {
        return $this->message->sender_type === '\Modules\Users\Models\User' &&
               $this->message->message_type !== 'system' &&
               config('chat.add_user_prefix_to_messages', true);
    }

    private function getUserName(): ?string
    {
        $user = $this->getUser();
        if (!$user) return null;

        return $user->first_name ?: 
               explode(' ', trim($user->name))[0] ?: 
               ucfirst(explode('@', $user->email)[0]);
    }

    private function getUser(): ?User
    {
        if ($this->message->sender_type !== '\Modules\Users\Models\User' || !$this->message->sender_id) {
            return null;
        }
        return User::find($this->message->sender_id);
    }

    private function getEvolutionInstance(): ?\Modules\Chat\Models\EvolutionInstance
    {
        if ($this->conversation->relationLoaded('evolutionInstance') && $this->conversation->evolutionInstance) {
            return $this->conversation->evolutionInstance;
        }

        if ($this->conversation->evolution_instance_name) {
            return \Modules\Chat\Models\EvolutionInstance::where('instance_name', $this->conversation->evolution_instance_name)->first();
        }

        if ($this->conversation->chat_channel_id) {
            $channel = $this->conversation->chatChannel ?? $this->conversation->load('chatChannel')->chatChannel;
            
            if ($channel && $channel->type === 'evolution_api') {
                $instanceName = $channel->getConfig('instance_name');
                if ($instanceName) {
                    return \Modules\Chat\Models\EvolutionInstance::where('instance_name', $instanceName)->first();
                }
            }
        }

        return null;
    }

    // ✅ MÉTODO CORRIGIDO PARA EVITAR NULL
    private function getContactPhone(): ?string
    {
        if (!$this->conversation->contact) {
            return null;
        }

        $contact = $this->conversation->contact;
        
        // ✅ PRIORIZAR whatsapp_number
        if (!empty($contact->whatsapp_number)) {
            return $this->formatPhone($contact->whatsapp_number);
        }
        
        // ✅ BUSCAR EM TELEFONES RELACIONADOS
        $firstPhone = $contact->phones?->first();
        if ($firstPhone && !empty($firstPhone->number)) {
            return $this->formatPhone($firstPhone->number);
        }
        
        return null;
    }

    private function formatPhone(string $phone): string
    {
        $phone = preg_replace('/[^0-9]/', '', $phone);

        if (strlen($phone) >= 13 && str_starts_with($phone, '55')) {
            $phone = substr($phone, 2);
        }

        if (strlen($phone) == 10 || strlen($phone) == 11) {
            $phone = '55' . $phone;
        }

        if (strlen($phone) == 12) {
            $areaCode = substr($phone, 2, 2);
            $number = substr($phone, 4);
            
            if (in_array($number[0], ['6', '7', '8', '9']) && strlen($number) == 8) {
                $phone = '55' . $areaCode . '9' . $number;
            }
        }

        return $phone;
    }

    private function extractPhoneFromJid(string $jid): string
    {
        $phone = preg_replace('/[^0-9]/', '', explode('@', $jid)[0]);
        if (strlen($phone) === 11 && !str_starts_with($phone, '55')) {
            $phone = '55' . $phone;
        }
        return $phone;
    }

    private function sendMessage(EvolutionApiService $evolutionApi, string $targetJid): array
    {
        $messageContent = $this->prepareMessageContent();

        if ($this->attachmentPath) {
            if ($this->message->message_type === 'audio') {
                if ($messageContent) {
                    try {
                        $evolutionApi->sendTextMessage($targetJid, $messageContent);
                    } catch (Exception $e) {
                        Log::warning('Erro ao enviar texto antes do áudio', ['error' => $e->getMessage()]);
                    }
                }
                return $evolutionApi->sendFile($targetJid, $this->attachmentPath, '');
            } else {
                return $evolutionApi->sendFile($targetJid, $this->attachmentPath, $messageContent ?: '');
            }
        } else {
            return $evolutionApi->sendTextMessage($targetJid, $messageContent);
        }
    }

    private function processResponse(array $result, array $validation): void
    {
        $whatsappMessageId = $result['key']['id'] ?? null;
        $remoteJid = $result['key']['remoteJid'] ?? $validation['jid'] ?? null;
        $status = $result['status'] ?? 'PENDING';

        $this->message->update([
            'whatsapp_message_id' => $whatsappMessageId,
            'delivery_status' => self::STATUS_MAPPING[strtoupper($status)] ?? 'sent',
            'status' => 'sent'
        ]);

        $updateData = ['last_message_at' => $this->message->sent_at];
        if ($remoteJid && !$this->conversation->whatsapp_remote_jid) {
            $updateData['whatsapp_remote_jid'] = $remoteJid;
        }
        
        $this->conversation->update($updateData);
    }

    private function failWithError(string $message): void
    {
        Log::error('❌ Job WhatsApp falhou', [
            'message_id' => $this->message->id,
            'conversation_id' => $this->conversation->id,
            'error' => $message
        ]);

        $this->message->update([
            'delivery_status' => 'failed',
            'status' => 'failed'
        ]);

        $this->fail(new Exception($message));
    }

    public function failed(\Throwable $exception): void
    {
        $this->message->update([
            'delivery_status' => 'failed',
            'status' => 'failed'
        ]);
    }
}