<?php
// modules/Chat/app/Models/ChatChannel.php

namespace Modules\Chat\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class ChatChannel extends Model
{
    use SoftDeletes;

    protected $table = 'chat_channels';

    protected $fillable = [
        'name',
        'type',
        'config',
        'is_active',
        'description',
        'priority',
        'webhook_url',
        'webhook_token'
    ];

    protected $casts = [
        'config' => 'array',
        'is_active' => 'boolean',
        'priority' => 'integer'
    ];

    /**
     * Conversas deste canal
     */
    public function conversations(): HasMany
    {
        return $this->hasMany(Conversation::class, 'chat_channel_id');
    }

    /**
     * Verificar se é canal externo
     */
    public function isExternal(): bool
    {
        return !in_array($this->type, ['internal_chat']);
    }

    /**
     * Verificar se é canal interno
     */
    public function isInternal(): bool
    {
        return $this->type === 'internal_chat';
    }

    /**
     * ✅ MÉTODO ORIGINAL (MANTER COMPATIBILIDADE)
     */
    public function getConfig(string $key, $default = null)
    {
        return data_get($this->config, $key, $default);
    }

    /**
     * ✅ ALIAS PARA COMPATIBILIDADE (MÉTODO QUE ESTAVA SENDO CHAMADO)
     */
    public function getConfigValue(string $key, $default = null)
    {
        return $this->getConfig($key, $default);
    }

    /**
     * ✅ DEFINIR CONFIGURAÇÃO
     */
    public function setConfig(string $key, $value): self
    {
        $config = $this->config ?? [];
        data_set($config, $key, $value);
        $this->config = $config;
        return $this;
    }

    /**
     * ✅ VERIFICAR SE CONFIGURAÇÃO EXISTE
     */
    public function hasConfig(string $key): bool
    {
        return data_get($this->config, $key) !== null;
    }

    /**
     * ✅ OBTER TODAS AS CONFIGURAÇÕES
     */
    public function getAllConfig(): array
    {
        return $this->config ?? [];
    }

    // ===== SCOPES =====

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeByType($query, string $type)
    {
        return $query->where('type', $type);
    }

    public function scopeExternal($query)
    {
        return $query->whereNotIn('type', ['internal_chat']);
    }

    public function scopeInternal($query)
    {
        return $query->where('type', 'internal_chat');
    }

    public function scopeInternalChat($query)
    {
        return $query->where('type', 'internal_chat');
    }

    public function scopeEvolutionApi($query)
    {
        return $query->where('type', 'evolution_api');
    }

    public function scopeWhatsapp($query)
    {
        return $query->where('type', 'whatsapp');
    }

    public function scopeTelegram($query)
    {
        return $query->where('type', 'telegram');
    }

    // ===== MÉTODOS AUXILIARES =====

    /**
     * ✅ VERIFICAR SE CANAL ESTÁ CONFIGURADO CORRETAMENTE
     */
    public function isConfigured(): bool
    {
        if (!$this->is_active) {
            return false;
        }

        // Verificações específicas por tipo
        switch ($this->type) {
            case 'evolution_api':
                return $this->hasConfig('api_url') && 
                       $this->hasConfig('api_key') && 
                       $this->hasConfig('instance_name');
                       
            case 'whatsapp':
                return $this->hasConfig('phone_number') && 
                       $this->hasConfig('api_token');
                       
            case 'telegram':
                return $this->hasConfig('bot_token') && 
                       $this->hasConfig('chat_id');
                       
            case 'internal_chat':
                return true; // Sempre configurado
                
            default:
                return !empty($this->config);
        }
    }

    /**
     * ✅ OBTER URL DO WEBHOOK
     */
    public function getWebhookUrl(): ?string
    {
        return $this->webhook_url ?: $this->getConfig('webhook_url');
    }

    /**
     * ✅ OBTER TOKEN DO WEBHOOK
     */
    public function getWebhookToken(): ?string
    {
        return $this->webhook_token ?: $this->getConfig('webhook_token');
    }

    /**
     * ✅ VERIFICAR SE WEBHOOK ESTÁ CONFIGURADO
     */
    public function hasWebhook(): bool
    {
        return !empty($this->getWebhookUrl());
    }

    /**
     * ✅ OBTER CONFIGURAÇÕES DE CONEXÃO PARA EVOLUTION API
     */
    public function getEvolutionConfig(): array
    {
        if ($this->type !== 'evolution_api') {
            return [];
        }

        return [
            'api_url' => $this->getConfig('api_url'),
            'api_key' => $this->getConfig('api_key'),
            'instance_name' => $this->getConfig('instance_name'),
            'webhook_url' => $this->getWebhookUrl(),
            'webhook_token' => $this->getWebhookToken(),
        ];
    }

    /**
     * ✅ VALIDAR CONFIGURAÇÕES DO CANAL
     */
    public function validateConfig(): array
    {
        $errors = [];

        if (!$this->name) {
            $errors[] = 'Nome do canal é obrigatório';
        }

        if (!$this->type) {
            $errors[] = 'Tipo do canal é obrigatório';
        }

        // Validações específicas por tipo
        switch ($this->type) {
            case 'evolution_api':
                if (!$this->getConfig('api_url')) {
                    $errors[] = 'URL da API Evolution é obrigatória';
                }
                if (!$this->getConfig('api_key')) {
                    $errors[] = 'Chave da API Evolution é obrigatória';
                }
                if (!$this->getConfig('instance_name')) {
                    $errors[] = 'Nome da instância Evolution é obrigatório';
                }
                break;

            case 'whatsapp':
                if (!$this->getConfig('phone_number')) {
                    $errors[] = 'Número do WhatsApp é obrigatório';
                }
                break;

            case 'telegram':
                if (!$this->getConfig('bot_token')) {
                    $errors[] = 'Token do bot Telegram é obrigatório';
                }
                break;
        }

        return $errors;
    }

    /**
     * ✅ OBTER STATUS DO CANAL
     */
    public function getStatus(): string
    {
        if (!$this->is_active) {
            return 'inactive';
        }

        if (!$this->isConfigured()) {
            return 'misconfigured';
        }

        return 'active';
    }

    /**
     * ✅ OBTER ESTATÍSTICAS DO CANAL
     */
    public function getStats(): array
    {
        return [
            'total_conversations' => $this->conversations()->count(),
            'active_conversations' => $this->conversations()->where('status', 'active')->count(),
            'total_messages' => $this->conversations()->withCount('messages')->get()->sum('messages_count'),
            'unread_messages' => $this->conversations()->sum('unread_count'),
        ];
    }

    // ===== MÉTODOS ESTÁTICOS =====

    /**
     * ✅ OBTER TIPOS DE CANAL DISPONÍVEIS
     */
    public static function getAvailableTypes(): array
    {
        return [
            'internal_chat' => 'Chat Interno',
            'evolution_api' => 'Evolution API',
            'whatsapp' => 'WhatsApp Business',
            'telegram' => 'Telegram Bot',
            'webhook' => 'Webhook Genérico',
        ];
    }

    /**
     * ✅ CRIAR CANAL PADRÃO
     */
    public static function createDefault(string $name, string $type, array $config = []): self
    {
        return self::create([
            'name' => $name,
            'type' => $type,
            'config' => $config,
            'is_active' => true,
            'priority' => 1,
            'description' => "Canal {$name} criado automaticamente"
        ]);
    }

    /**
     * ✅ OBTER CANAL INTERNO PADRÃO
     */
    public static function getInternalChannel(): self
    {
        return self::firstOrCreate(
            ['type' => 'internal_chat'],
            [
                'name' => 'Chat Interno',
                'type' => 'internal_chat',
                'config' => [],
                'is_active' => true,
                'priority' => 0,
                'description' => 'Canal de chat interno entre usuários'
            ]
        );
    }
}