<?php

namespace Modules\Chat\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\DB;
use Modules\Contacts\Models\Contact;
use Modules\Users\Models\User;

class Conversation extends Model
{
    use SoftDeletes;

    // ✅ CORREÇÃO: Definir nome da tabela explicitamente
    protected $table = 'conversations';

    const TYPE_INTERNAL = 'internal';
    const TYPE_EXTERNAL = 'external';
    const STATUS_ACTIVE = 'active';
    const STATUS_CLOSED = 'closed';
    const STATUS_WAITING = 'waiting';
    const STATUS_ARCHIVED = 'archived';

    protected $fillable = [
        'contact_id', 'chat_channel_id', 'external_id', 'contact_phone',
        'contact_identifier', 'status', 'type', 'assigned_user_id',
        'last_message_at', 'metadata', 'unread_count', 'is_pinned', 
        'is_muted', 'closed_by', 'closed_at', 'close_reason',
        'whatsapp_chat_id', 'whatsapp_remote_jid', 'evolution_instance_name',
        'whatsapp_status', 'last_whatsapp_activity', 'whatsapp_profile_name',
        'whatsapp_profile_picture', 'is_whatsapp_business', 'whatsapp_labels'
    ];

    protected $casts = [
        'last_message_at' => 'datetime',
        'metadata' => 'array',
        'unread_count' => 'integer',
        'is_pinned' => 'boolean',
        'is_muted' => 'boolean',
        'closed_at' => 'datetime',
        'last_whatsapp_activity' => 'datetime',
        'is_whatsapp_business' => 'boolean',
        'whatsapp_labels' => 'array'
    ];

    // ✅ RELACIONAMENTOS CORRIGIDOS COM NOMES DE TABELA EXPLÍCITOS
    public function contact(): BelongsTo
    {
        return $this->belongsTo(Contact::class);
    }

    public function chatChannel(): BelongsTo
    {
        return $this->belongsTo(ChatChannel::class);
    }

    public function assignedUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'assigned_user_id');
    }

    public function messages(): HasMany
    {
        return $this->hasMany(Message::class)->orderBy('sent_at');
    }

    public function lastMessage(): HasOne
    {
        return $this->hasOne(Message::class)->latestOfMany('sent_at');
    }

    public function evolutionInstance(): BelongsTo
    {
        return $this->belongsTo(EvolutionInstance::class, 'evolution_instance_name', 'instance_name');
    }


    public function participants(): BelongsToMany
    {
        return $this->belongsToMany(
            User::class, 
            'conversation_participants', // ✅ Nome explícito da tabela
            'conversation_id', 
            'user_id'
        )->withPivot(['role', 'is_active', 'joined_at', 'left_at', 'last_read_at'])
        ->withTimestamps()
        ->wherePivot('is_active', true);
    }

     public function allParticipants(): BelongsToMany
    {
        return $this->belongsToMany(
            User::class, 
            'conversation_participants', // ✅ Nome explícito da tabela
            'conversation_id', 
            'user_id'
        )->withPivot(['role', 'is_active', 'joined_at', 'left_at', 'last_read_at'])
        ->withTimestamps();
    }

    public function assignedUsers(): BelongsToMany
    {
        return $this->belongsToMany(
            User::class, 
            'conversation_assignments', // ✅ Nome explícito da tabela
            'conversation_id', 
            'user_id'
        )->withPivot(['role', 'assigned_at', 'assigned_by', 'is_primary'])
        ->withTimestamps()
        ->orderByPivot('is_primary', 'desc')
        ->orderByPivot('assigned_at', 'asc');
    }

    public function scopeForUser($query, int $userId, bool $showMyOnly = false, bool $isAdmin = false)
    {
        if ($showMyOnly) {
            return $query->where(function ($q) use ($userId) {
                $q->where('assigned_user_id', $userId)
                  ->orWhereExists(function($exists) use ($userId) {
                      $exists->select(DB::raw(1))
                             ->from('conversation_assignments') // ✅ Nome correto
                             ->whereColumn('conversations.id', 'conversation_assignments.conversation_id')
                             ->where('conversation_assignments.user_id', $userId);
                  })
                  ->orWhereExists(function($exists) use ($userId) {
                      $exists->select(DB::raw(1))
                             ->from('conversation_participants') // ✅ Nome correto
                             ->whereColumn('conversations.id', 'conversation_participants.conversation_id')
                             ->where('conversation_participants.user_id', $userId)
                             ->where('conversation_participants.is_active', true);
                  });
            });
        }

        return $query->where(function ($q) use ($userId, $isAdmin) {
            // Suas conversas
            $q->where('assigned_user_id', $userId)
              ->orWhereExists(function($exists) use ($userId) {
                  $exists->select(DB::raw(1))
                         ->from('conversation_assignments') // ✅ Nome correto
                         ->whereColumn('conversations.id', 'conversation_assignments.conversation_id')
                         ->where('conversation_assignments.user_id', $userId);
              })
              ->orWhereExists(function($exists) use ($userId) {
                  $exists->select(DB::raw(1))
                         ->from('conversation_participants') // ✅ Nome correto
                         ->whereColumn('conversations.id', 'conversation_participants.conversation_id')
                         ->where('conversation_participants.user_id', $userId)
                         ->where('conversation_participants.is_active', true);
              })
              // ✅ CONVERSAS NÃO ATRIBUÍDAS
              ->orWhere(function($unassignedQ) {
                  $unassignedQ->where('type', 'external')
                              ->whereNull('assigned_user_id')
                              ->whereNotExists(function($notExists) {
                                  $notExists->select(DB::raw(1))
                                           ->from('conversation_assignments') // ✅ Nome correto
                                           ->whereColumn('conversations.id', 'conversation_assignments.conversation_id');
                              });
              });
        });
    }
/**
 * ✅ NOVO SCOPE PARA ADMIN VER TODAS AS CONVERSAS
 */
public function scopeAllConversations($query)
{
    return $query; // Sem filtros - retorna todas as conversas
}

    /**
     * ✅ SCOPE ESPECÍFICO PARA "APENAS MINHAS CONVERSAS"
     */
   public function scopeMyConversationsOnly($query, int $userId)
    {
        return $query->where(function ($q) use ($userId) {
            $q->where('assigned_user_id', $userId)
              ->orWhereExists(function($exists) use ($userId) {
                  $exists->select(DB::raw(1))
                         ->from('conversation_assignments') // ✅ Nome correto
                         ->whereColumn('conversations.id', 'conversation_assignments.conversation_id')
                         ->where('conversation_assignments.user_id', $userId);
              })
              ->orWhereExists(function($exists) use ($userId) {
                  $exists->select(DB::raw(1))
                         ->from('conversation_participants') // ✅ Nome correto
                         ->whereColumn('conversations.id', 'conversation_participants.conversation_id')
                         ->where('conversation_participants.user_id', $userId)
                         ->where('conversation_participants.is_active', true);
              });
        });
    }
    /**
     * ✅ MANTER COMPATIBILIDADE
     */
    public function scopeVisibleToUser($query, int $userId)
    {
        return $this->scopeForUser($query, $userId);
    }

    public function scopeAccessibleByUser($query, int $userId)
    {
        return $this->scopeForUser($query, $userId);
    }

    /**
     * ✅ VERIFICAÇÃO OTIMIZADA DE PERMISSÃO
     */
       public function canUserView(int $userId): bool
    {
        static $userCache = [];
        
        if (!isset($userCache[$userId])) {
            $userCache[$userId] = User::find($userId);
        }
        
        $user = $userCache[$userId];
        
        if ($user && $user->super_admin) {
            return true;
        }

        if ($this->assigned_user_id === $userId) {
            return true;
        }

        if ($this->relationLoaded('assignedUsers')) {
            $assigned = $this->assignedUsers->contains('id', $userId);
        } else {
            $assigned = $this->assignedUsers()->where('user_id', $userId)->exists();
        }

        if ($assigned) {
            return true;
        }

        if ($this->type === 'external' && 
            !$this->assigned_user_id && 
            (!$this->relationLoaded('assignedUsers') || $this->assignedUsers->isEmpty())) {
            return true;
        }

        if ($this->type === 'internal') {
            if ($this->relationLoaded('participants')) {
                return $this->participants->contains('id', $userId);
            }
            return $this->participants()->where('user_id', $userId)->exists();
        }

        return false;
    }

    // Métodos auxiliares mantidos
    public function isInternal(): bool
    {
        return $this->type === self::TYPE_INTERNAL;
    }

    public function isExternal(): bool
    {
        return $this->type === self::TYPE_EXTERNAL;
    }

    public function isWhatsApp(): bool
    {
        return $this->isExternal() && 
               ($this->chatChannel && $this->chatChannel->type === 'evolution_api');
    }

    public function getDisplayName(): string
    {
        if ($this->isInternal()) {
            return data_get($this->metadata, 'subject', 'Chat Interno');
        }

        return $this->contact?->display_name ?? 
               $this->whatsapp_profile_name ?? 
               $this->contact_phone ?? 
               'Conversa Externa';
    }

    public function getAssignedUserIds(): array
    {
        if ($this->relationLoaded('assignedUsers')) {
            return $this->assignedUsers->pluck('id')->toArray();
        }
        
        return $this->assignedUsers()->pluck('user_id')->toArray();
    }

    public function isAssignedToUser(int $userId): bool
    {
        return in_array($userId, $this->getAssignedUserIds()) || 
               $this->assigned_user_id === $userId;
    }

    // Scopes adicionais
    public function scopeByStatus($query, string $status)
    {
        return $query->where('status', $status);
    }

    public function scopeByType($query, string $type)
    {
        return $query->where('type', $type);
    }

    public function scopeActive($query)
    {
        return $query->where('status', self::STATUS_ACTIVE);
    }

    public function scopeUnassigned($query)
    {
        return $query->whereNull('assigned_user_id')
                     ->whereDoesntHave('assignedUsers');
    }

    public function scopeWithBasicRelations($query)
    {
        return $query->with([
            'contact',
            'assignedUser',
            'lastMessage'
        ]);
    }
}