<?php
// modules/Chat/Models/EvolutionInstance.php

namespace Modules\Chat\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Crypt;
use Carbon\Carbon;

class EvolutionInstance extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'instance_name',
        'display_name',
        'description',
        'api_url',
        'api_key',
        'api_version',
        'status',
        'connection_state',
        'last_ping',
        'connected_at',
        'disconnected_at',
        'qr_code',
        'qr_code_generated_at',
        'qr_code_expires_at',
        'instance_data',
        'whatsapp_number',
        'whatsapp_name',
        'whatsapp_profile_picture',
        'is_business',
        'settings',
        'webhook_url',
        'webhook_events',
        'webhook_active',
        'messages_sent_today',
        'messages_received_today',
        'last_message_at',
        'error_log',
        'error_count',
        'last_error_at',
        'created_by',
        'managed_by',
        'allowed_users',
        'is_default',
        'priority'
    ];

    protected $casts = [
        'qr_code' => 'array',
        'qr_code_generated_at' => 'datetime',
        'qr_code_expires_at' => 'datetime',
        'instance_data' => 'array',
        'settings' => 'array',
        'webhook_events' => 'array',
        'webhook_active' => 'boolean',
        'is_business' => 'boolean',
        'last_ping' => 'datetime',
        'connected_at' => 'datetime',
        'disconnected_at' => 'datetime',
        'last_message_at' => 'datetime',
        'error_log' => 'array',
        'last_error_at' => 'datetime',
        'allowed_users' => 'array',
        'is_default' => 'boolean',
        'messages_sent_today' => 'integer',
        'messages_received_today' => 'integer',
        'error_count' => 'integer',
        'priority' => 'integer'
    ];

    // ===== RELATIONSHIPS =====

    public function getRouteKeyName(): string
    {
        return 'instance_name';
    }

    public function conversations(): HasMany
    {
        return $this->hasMany(Conversation::class, 'evolution_instance_name', 'instance_name');
    }

    public function webhooks(): HasMany
    {
        return $this->hasMany(EvolutionWebhook::class, 'instance_name', 'instance_name');
    }

    public function statistics(): HasMany
    {
        return $this->hasMany(EvolutionStatistic::class, 'instance_name', 'instance_name');
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(\Modules\Users\Models\User::class, 'created_by');
    }

    public function managedBy(): BelongsTo
    {
        return $this->belongsTo(\Modules\Users\Models\User::class, 'managed_by');
    }

    // ===== SCOPES =====

    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function scopeConnected($query)
    {
        return $query->where('connection_state', 'open');
    }

    public function scopeDefault($query)
    {
        return $query->where('is_default', true);
    }

    public function scopeForUser($query, $userId)
    {
        return $query->where(function ($q) use ($userId) {
            $q->where('created_by', $userId)
                ->orWhere('managed_by', $userId)
                ->orWhereJsonContains('allowed_users', $userId);
        });
    }

    // ===== ACCESSORS & MUTATORS =====

    public function getApiKeyAttribute($value)
    {
        try {
            return $value ? Crypt::decryptString($value) : null;
        } catch (\Exception $e) {
            return $value; // Fallback para chaves não criptografadas
        }
    }

    public function setApiKeyAttribute($value)
    {
        $this->attributes['api_key'] = $value ? Crypt::encryptString($value) : null;
    }

    public function getIsOnlineAttribute(): bool
    {
        return $this->status === 'active' &&
            $this->connection_state === 'open';
    }
    public function getQrCodeExpiredAttribute(): bool
    {
        return $this->qr_code_expires_at && $this->qr_code_expires_at->isPast();
    }

    public function getUptimePercentageAttribute(): float
    {
        // Calcular uptime baseado nos últimos 30 dias
        $totalMinutes = 30 * 24 * 60; // 30 dias em minutos
        $downMinutes = $this->calculateDowntime();

        return round((($totalMinutes - $downMinutes) / $totalMinutes) * 100, 2);
    }

    // ===== MÉTODOS PÚBLICOS =====

    public function isUserAllowed(int $userId): bool
    {
        return $this->created_by === $userId ||
            $this->managed_by === $userId ||
            in_array($userId, $this->allowed_users ?? []);
    }

    public function updatePing(): bool
    {
        return $this->update([
            'last_ping' => now(),
            'error_count' => 0
        ]);
    }

    public function recordError(string $message, array $details = []): void
    {
        $errorLog = $this->error_log ?? [];

        // Manter apenas os últimos 10 erros
        if (count($errorLog) >= 10) {
            array_shift($errorLog);
        }

        $errorLog[] = [
            'message' => $message,
            'details' => $details,
            'timestamp' => now()->toISOString()
        ];

        $this->update([
            'error_log' => $errorLog,
            'error_count' => $this->error_count + 1,
            'last_error_at' => now()
        ]);
    }

    public function updateQrCode(array $qrData): void
    {
        $this->update([
            'qr_code' => $qrData,
            'qr_code_generated_at' => now(),
            'qr_code_expires_at' => now()->addMinutes(2), // QR expira em 2 minutos
            'status' => 'qr_pending'
        ]);
    }

    public function markAsConnected(array $instanceData = []): void
    {
        $this->update([
            'status' => 'active',
            'connection_state' => 'open',
            'connected_at' => now(),
            'disconnected_at' => null,
            'instance_data' => $instanceData,
            'whatsapp_number' => $instanceData['number'] ?? null,
            'whatsapp_name' => $instanceData['name'] ?? null,
            'whatsapp_profile_picture' => $instanceData['profilePicture'] ?? null,
            'is_business' => $instanceData['business'] ?? false,
            'error_count' => 0,
            'qr_code' => null,
            'qr_code_generated_at' => null,
            'qr_code_expires_at' => null
        ]);
    }

    public function markAsDisconnected(string $reason = null): void
    {
        $this->update([
            'status' => 'disconnected',
            'connection_state' => 'close',
            'disconnected_at' => now()
        ]);

        if ($reason) {
            $this->recordError("Disconnected: {$reason}");
        }
    }

    public function incrementMessageCount(bool $sent = true): void
    {
        $field = $sent ? 'messages_sent_today' : 'messages_received_today';

        $this->increment($field);
        $this->update(['last_message_at' => now()]);
    }

    public function resetDailyCounters(): void
    {
        $this->update([
            'messages_sent_today' => 0,
            'messages_received_today' => 0
        ]);
    }

    public function getApiEndpoint(string $path = ''): string
    {
        $baseUrl = rtrim($this->api_url, '/');
        $version = $this->api_version ?: 'v1';
        $instance = $this->instance_name;

        return "{$baseUrl}/{$version}/instance/{$instance}" . ($path ? "/{$path}" : '');
    }

    public function getWebhookEndpoint(): string
    {
        $baseUrl = config('app.url') ?: request()->getSchemeAndHttpHost();
        return rtrim($baseUrl, '/') . '/api/evolution/webhook/' . $this->instance_name;
    }
    // ===== MÉTODOS PRIVADOS =====

    private function calculateDowntime(): int
    {
        // Implementar cálculo de downtime baseado em logs
        // Por enquanto, retorna 0
        return 0;
    }

    // ===== BOOT =====

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($instance) {
            // Definir webhook_url automático se não fornecido
            if (!$instance->webhook_url) {
                $instance->webhook_url = $instance->getWebhookEndpoint();
            }

            // Definir eventos padrão do webhook
            if (!$instance->webhook_events) {
                $instance->webhook_events = [
                    'messages',
                    'message_status',
                    'connection_status',
                    'qrcode_updated'
                ];
            }
        });

        static::updating(function ($instance) {
            // Reset contadores diários se mudou o dia
            if ($instance->isDirty('updated_at')) {
                $lastUpdate = $instance->getOriginal('updated_at');
                if ($lastUpdate && !Carbon::parse($lastUpdate)->isToday()) {
                    $instance->resetDailyCounters();
                }
            }
        });
    }
}