<?php
// modules/Chat/Models/EvolutionStatistic.php

namespace Modules\Chat\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Carbon\Carbon;

class EvolutionStatistic extends Model
{
    protected $fillable = [
        'instance_name',
        'date',
        'period_type',
        'messages_sent',
        'messages_received',
        'messages_failed',
        'messages_delivered',
        'messages_read',
        'conversations_started',
        'conversations_active',
        'conversations_closed',
        'new_contacts',
        'active_contacts',
        'avg_response_time',
        'uptime_percentage',
        'connection_drops',
        'reconnections',
        'webhook_errors',
        'api_errors',
        'timeout_errors',
        'text_messages',
        'image_messages',
        'audio_messages',
        'video_messages',
        'document_messages',
        'other_messages'
    ];

    protected $casts = [
        'date' => 'date',
        'avg_response_time' => 'decimal:2',
        'uptime_percentage' => 'decimal:2'
    ];

    // ===== RELATIONSHIPS =====

    public function instance(): BelongsTo
    {
        return $this->belongsTo(EvolutionInstance::class, 'instance_name', 'instance_name');
    }

    // ===== SCOPES =====

    public function scopeDaily($query)
    {
        return $query->where('period_type', 'daily');
    }

    public function scopeWeekly($query)
    {
        return $query->where('period_type', 'weekly');
    }

    public function scopeMonthly($query)
    {
        return $query->where('period_type', 'monthly');
    }

    public function scopeForPeriod($query, Carbon $start, Carbon $end)
    {
        return $query->whereBetween('date', [$start->format('Y-m-d'), $end->format('Y-m-d')]);
    }

    public function scopeForInstance($query, string $instanceName)
    {
        return $query->where('instance_name', $instanceName);
    }

    // ===== MÉTODOS ESTÁTICOS =====

    public static function createOrUpdateDaily(string $instanceName, array $data): self
    {
        return static::updateOrCreate([
            'instance_name' => $instanceName,
            'date' => now()->format('Y-m-d'),
            'period_type' => 'daily'
        ], $data);
    }

    public static function getTodayStats(string $instanceName): ?self
    {
        return static::where('instance_name', $instanceName)
                    ->where('date', now()->format('Y-m-d'))
                    ->where('period_type', 'daily')
                    ->first();
    }

    // ===== ACCESSORS =====

    public function getTotalMessagesAttribute(): int
    {
        return $this->messages_sent + $this->messages_received;
    }

    public function getSuccessRateAttribute(): float
    {
        $total = $this->messages_sent;
        if ($total === 0) return 100.0;
        
        $successful = $total - $this->messages_failed;
        return round(($successful / $total) * 100, 2);
    }

    public function getDeliveryRateAttribute(): float
    {
        $sent = $this->messages_sent;
        if ($sent === 0) return 0.0;
        
        return round(($this->messages_delivered / $sent) * 100, 2);
    }

    public function getReadRateAttribute(): float
    {
        $delivered = $this->messages_delivered;
        if ($delivered === 0) return 0.0;
        
        return round(($this->messages_read / $delivered) * 100, 2);
    }
}