<?php

namespace Modules\Chat\Models;

use Modules\Users\Models\User;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Database\Eloquent\SoftDeletes;
use Modules\Contacts\Models\Contact;

class Message extends Model
{
    use SoftDeletes;
    
    protected $table = 'messages';
    
    const SENDER_TYPE_USER = 'user'; 
    const SENDER_TYPE_CONTACT = 'contact';
    const SENDER_TYPE_SYSTEM = 'system';

    protected $fillable = [
        'conversation_id', 
        'external_id', 
        'whatsapp_message_id', 
        'whatsapp_key_id', 
        'whatsapp_key_remote_jid', 
        'whatsapp_key_from_me', 
        'whatsapp_message_type', 
        'delivery_status', 
        'whatsapp_timestamp', 
        'content', 
        'message_type', 
        'is_from_contact', 
        'sender_type', 
        'sender_id', 
        'sent_at', 
        'delivered_at', 
        'read_at', 
        'whatsapp_participant', 
        'whatsapp_quoted_message', 
        'whatsapp_reactions', 
        'is_forwarded', 
        'forward_score', 
        'is_ephemeral', 
        'ephemeral_expires_at', 
        'reply_to_id', 
        'status', 
        'metadata'
    ];
    
    protected $casts = [
        'sent_at' => 'datetime',
        'delivered_at' => 'datetime', 
        'read_at' => 'datetime',
        'whatsapp_timestamp' => 'datetime',
        'ephemeral_expires_at' => 'datetime',
        'metadata' => 'array',
        'whatsapp_quoted_message' => 'array',
        'whatsapp_reactions' => 'array',
        'is_from_contact' => 'boolean',
        'whatsapp_key_from_me' => 'boolean',
        'is_forwarded' => 'boolean',
        'is_ephemeral' => 'boolean'
    ];
    
    public function conversation(): BelongsTo
    {
        return $this->belongsTo(Conversation::class);
    }
    
    public function sender(): MorphTo
    {
        return $this->morphTo();
    }

    public function attachments(): HasMany
    {
        return $this->hasMany(MessageAttachment::class, 'message_id');
    }
    
    public function replyTo(): BelongsTo
    {
        return $this->belongsTo(self::class, 'reply_to_id');
    }
    
    public function replies(): HasMany
    {
        return $this->hasMany(self::class, 'reply_to_id');
    }
    
    public function isFromMe(): bool
    {
        return $this->whatsapp_key_from_me || !$this->is_from_contact;
    }
    
    public function isGroupMessage(): bool
    {
        return !empty($this->whatsapp_participant);
    }
    
    public function hasAttachments(): bool
    {
        return $this->attachments()->count() > 0;
    }
    
    public function isReply(): bool
    {
        return !empty($this->reply_to_id);
    }
    
    public function getSenderName(): string
    {
        if (!empty($this->metadata['sender_name'])) {
            return $this->metadata['sender_name'];
        }
        
        if (!empty($this->metadata['push_name'])) {
            return $this->metadata['push_name'];
        }
        
        if ($this->sender_type === self::SENDER_TYPE_SYSTEM) {
            return 'Sistema'; 
        }

        $this->loadMissing('sender'); 

        if ($this->relationLoaded('sender') && $this->sender) {
            if ($this->sender instanceof User) {
                return $this->sender->name ?? 'Usuário';
            }
            if ($this->sender instanceof Contact) {
                return $this->sender->display_name ?? $this->sender->first_name ?? 'Contato';
            }
        }
        
        return $this->getSenderNameFallback();
    }
    
    public function getSenderAvatar(): ?string
    {
        $this->loadMissing('sender');
        
        if ($this->sender instanceof User) {
            return $this->sender->avatar_url ?? null;
        }
        
        if ($this->sender instanceof Contact) {
            return $this->sender->avatar_url ?? $this->sender->whatsapp_profile_photo_url ?? null;
        }
        
        return null;
    }
    
    public function toFrontendArray(): array
    {
        $this->loadMissing(['sender', 'attachments', 'replyTo']);
        
        return [
            'id' => $this->id,
            'conversation_id' => $this->conversation_id,
            'content' => $this->content,
            'message_type' => $this->message_type,
            'whatsapp_message_type' => $this->whatsapp_message_type,
            'delivery_status' => $this->delivery_status,
            'status' => $this->status,
            'sent_at' => $this->sent_at?->toISOString(),
            'delivered_at' => $this->delivered_at?->toISOString(),
            'read_at' => $this->read_at?->toISOString(),
            'whatsapp_timestamp' => $this->whatsapp_timestamp?->toISOString(),
            
            'sender_id' => $this->sender_id,
            'sender_type' => $this->sender_type,
            'sender_name' => $this->getSenderName(),
            'sender_avatar' => $this->getSenderAvatar(),
            'is_from_me' => $this->isFromMe(),
            'is_from_contact' => $this->is_from_contact,
            
            'whatsapp_message_id' => $this->whatsapp_message_id,
            'whatsapp_key_from_me' => $this->whatsapp_key_from_me,
            'whatsapp_participant' => $this->whatsapp_participant,
            'is_group_message' => $this->isGroupMessage(),
            'is_forwarded' => $this->is_forwarded,
            'forward_score' => $this->forward_score,
            
            'attachments' => $this->attachments->map(function($attachment) {
                return $attachment->toFrontendArray();
            })->toArray(),
            'has_attachments' => $this->hasAttachments(),
            
            'reply_to_id' => $this->reply_to_id,
            'is_reply' => $this->isReply(),
            'reply_to' => $this->replyTo ? [
                'id' => $this->replyTo->id,
                'content' => $this->replyTo->content,
                'sender_name' => $this->replyTo->getSenderName(),
                'message_type' => $this->replyTo->message_type
            ] : null,
            
            'whatsapp_reactions' => $this->whatsapp_reactions,
            'whatsapp_quoted_message' => $this->whatsapp_quoted_message,
            
            'is_ephemeral' => $this->is_ephemeral,
            'ephemeral_expires_at' => $this->ephemeral_expires_at?->toISOString(),
            
            'metadata' => $this->metadata,
            'created_at' => $this->created_at?->toISOString(),
            'updated_at' => $this->updated_at?->toISOString()
        ];
    }
    
    private function getSenderNameFallback(): string
    {
        return match($this->sender_type) {
            '\Modules\Users\Models\User' => 'Usuário CRM', 
            '\Modules\Contacts\Models\Contact' => 'Contato WhatsApp',
            self::SENDER_TYPE_SYSTEM => 'Sistema',
            default => 'Desconhecido'
        };
    }
    
    public static function normalizeSenderType(?string $senderType): ?string
    {
        return match($senderType) {
            'user', self::SENDER_TYPE_USER, User::class => 
                '\Modules\Users\Models\User',
                
            'contact', self::SENDER_TYPE_CONTACT, Contact::class => 
                '\Modules\Contacts\Models\Contact',
                
            'system', self::SENDER_TYPE_SYSTEM, null, '' => 
                self::SENDER_TYPE_SYSTEM,
                
            '\Modules\Users\Models\User',
            '\Modules\Contacts\Models\Contact' => $senderType,
                
            default => $senderType
        };
    }
    
    // Scopes úteis
    public function scopeFromMe($query)
    {
        return $query->where('whatsapp_key_from_me', true);
    }
    
    public function scopeFromContact($query)
    {
        return $query->where('is_from_contact', true);
    }
    
    public function scopeWithAttachments($query)
    {
        return $query->whereHas('attachments');
    }
    
    public function scopeGroupMessages($query)
    {
        return $query->whereNotNull('whatsapp_participant');
    }
    
    public function scopeByType($query, string $type)
    {
        return $query->where('message_type', $type);
    }
    
    public function scopeByStatus($query, string $status)
    {
        return $query->where('delivery_status', $status);
    }
    
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($message) {
            $message->sent_at ??= now();
            
            if ($message->sender_type && $message->sender_type !== self::SENDER_TYPE_SYSTEM) {
                $message->sender_type = static::normalizeSenderType($message->sender_type);
            }
        });

        static::created(function ($message) {
            $message->conversation?->update([
                'last_message_at' => $message->sent_at
            ]);
        });
    }
}