<?php

namespace Modules\Chat\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class MessageAttachment extends Model
{
    use SoftDeletes;
    
    protected $table = 'message_attachments'; 
    
    protected $fillable = [
        'message_id', 'file_name', 'file_path', 'file_type', 'file_size', 'mime_type',
        'external_url', 'whatsapp_media_key', 'whatsapp_direct_path', 'whatsapp_media_url',
        'whatsapp_metadata', 'download_status', 'downloaded_at', 'download_attempts',
        'download_error', 'width', 'height', 'duration', 'thumbnail_path', 'is_animated'
    ];
    
    protected $casts = [
        'file_size' => 'integer', 'deleted_at' => 'datetime', 'whatsapp_metadata' => 'array',
        'downloaded_at' => 'datetime', 'download_attempts' => 'integer', 'width' => 'integer',
        'height' => 'integer', 'duration' => 'integer', 'is_animated' => 'boolean'
    ];
    
    public function message(): BelongsTo
    {
        return $this->belongsTo(Message::class, 'message_id');
    }
    
    /**
     * ✅ URL SEMPRE SEGURA PARA FRONTEND
     */
    public function getFileUrlAttribute(): string
    {
        // 1. PRIORIDADE: Arquivo local baixado
        if ($this->file_path && $this->download_status === 'completed') {
            return url('storage/' . $this->file_path); // ✅ URL LOCAL
        }
        
        // 2. FALLBACK: Proxy para evitar erro 403
        if ($this->whatsapp_media_url || $this->external_url) {
            return route('chat.media.proxy', ['attachment' => $this->id]); // ✅ URL DO PROXY
        }
        
        return '';
    }
    
    /**
     * ✅ ARRAY OTIMIZADO PARA FRONTEND
     */
    public function toFrontendArray(): array
    {
        return [
            'id' => $this->id,
            'file_name' => $this->file_name,
            'file_type' => $this->file_type,
            'file_url' => $this->file_url, // ✅ USA getFileUrlAttribute()
            'mime_type' => $this->mime_type,
            'file_size' => $this->file_size,
            'formatted_file_size' => $this->formatted_file_size,
            'width' => $this->width,
            'height' => $this->height,
            'duration' => $this->duration,
            'is_image' => $this->isImage(),
            'is_video' => $this->isVideo(),
            'is_audio' => $this->isAudio(),
            'is_voice_note' => $this->isVoiceNote(),
            'is_document' => $this->isDocument(),
            'download_status' => $this->download_status,
            'is_downloaded' => $this->isDownloaded(),
            'thumbnail_url' => $this->thumbnail_url
        ];
    }
    
    // ... resto dos métodos existentes ...
    
    public function getFormattedFileSizeAttribute(): string
    {
        if (!$this->file_size) return '0 B';
        
        $bytes = $this->file_size;
        $k = 1024;
        $sizes = ['B', 'KB', 'MB', 'GB'];
        $i = floor(log($bytes) / log($k));
        
        return round($bytes / pow($k, $i), 1) . ' ' . $sizes[$i];
    }
    
    public function getThumbnailUrlAttribute(): ?string
    {
        if ($this->thumbnail_path) {
            return asset('storage/' . $this->thumbnail_path);
        }
        
        return null;
    }
    
    public function isImage(): bool
    {
        return $this->file_type === 'image' || str_starts_with($this->mime_type ?? '', 'image/');
    }
    
    public function isVideo(): bool
    {
        return $this->file_type === 'video' || str_starts_with($this->mime_type ?? '', 'video/');
    }
    
    public function isAudio(): bool
    {
        return $this->file_type === 'audio' || str_starts_with($this->mime_type ?? '', 'audio/');
    }
    
    public function isDocument(): bool
    {
        return !$this->isImage() && !$this->isVideo() && !$this->isAudio();
    }
    
    public function isVoiceNote(): bool
    {
        return $this->isAudio() && ($this->whatsapp_metadata['is_voice_note'] ?? false);
    }
    
    public function isDownloaded(): bool
    {
        return $this->download_status === 'completed' && !empty($this->file_path);
    }
    
    public function markAsDownloaded(string $filePath): bool
    {
        return $this->update([
            'file_path' => $filePath,
            'download_status' => 'completed',
            'downloaded_at' => now(),
            'download_error' => null
        ]);
    }
    
    public function markDownloadFailed(string $error): bool
    {
        return $this->update([
            'download_status' => 'failed',
            'download_attempts' => $this->download_attempts + 1,
            'download_error' => $error
        ]);
    }
    
    public function scopeDownloaded($query)
    {
        return $query->where('download_status', 'completed');
    }
    
    public function scopeFailedDownload($query)
    {
        return $query->where('download_status', 'failed');
    }
}