<?php
// modules/Chat/app/Policies/ConversationPolicy.php

namespace Modules\Chat\Policies;

use Illuminate\Auth\Access\HandlesAuthorization;
use Modules\Users\Models\User;
use Modules\Chat\Models\Conversation;

class ConversationPolicy
{
    use HandlesAuthorization;

    /**
     * ✅ VERIFICAR SE PODE VER A CONVERSA
     */
    public function view(User $user, Conversation $conversation): bool
    {
        // Super admin pode ver tudo
        if ($user->super_admin) {
            return true;
        }

        // Pode ver todas as conversas
        if ($user->can('view all conversations')) {
            return true;
        }

        // É atribuída ao usuário
        if ((int) $conversation->assigned_user_id === (int) $user->id) {
            return true;
        }

        // Está nas atribuições múltiplas
        if ($conversation->assignedUsers()->where('user_id', $user->id)->exists()) {
            return true;
        }

        // É participante (conversas internas)
        if ($conversation->type === 'internal' && 
            $conversation->participants()->where('user_id', $user->id)->exists()) {
            return true;
        }

        // Pode ver conversas da equipe
        if ($conversation->assigned_user_id && $user->can('view team conversations')) {
            return $user->managesAnyTeamsOf($conversation->assigned_user_id);
        }

        // Pode ver conversas não atribuídas
        if (!$conversation->assigned_user_id && 
            !$conversation->assignedUsers()->exists() && 
            $user->can('view unassigned conversations')) {
            return true;
        }

        return false;
    }

    /**
     * ✅ VERIFICAR SE PODE VER TODAS AS CONVERSAS
     */
    public function viewAll(User $user): bool
    {
        return $user->super_admin || $user->can('view all conversations');
    }

    /**
     * ✅ VERIFICAR SE PODE CRIAR CONVERSA
     */
    public function create(User $user): bool
    {
        return $user->can('create conversations');
    }

    /**
     * ✅ VERIFICAR SE PODE EDITAR CONVERSA
     */
    public function update(User $user, Conversation $conversation): bool
    {
        if ($user->super_admin || $user->can('edit all conversations')) {
            return true;
        }

        if ((int) $conversation->assigned_user_id === (int) $user->id && 
            $user->can('edit own conversations')) {
            return true;
        }

        if ($conversation->assigned_user_id && $user->can('edit team conversations')) {
            return $user->managesAnyTeamsOf($conversation->assigned_user_id);
        }

        return false;
    }

    /**
     * ✅ VERIFICAR SE PODE ATRIBUIR CONVERSA
     */
    public function assign(User $user, Conversation $conversation): bool
    {
        if ($user->super_admin || $user->can('assign any conversation')) {
            return true;
        }

        if ((int) $conversation->assigned_user_id === (int) $user->id && 
            $user->can('assign own conversations')) {
            return true;
        }

        if ($conversation->assigned_user_id && $user->can('assign team conversations')) {
            return $user->managesAnyTeamsOf($conversation->assigned_user_id);
        }

        return false;
    }

    /**
     * ✅ VERIFICAR SE PODE ARQUIVAR CONVERSA
     */
    public function archive(User $user, Conversation $conversation): bool
    {
        if ($user->super_admin || $user->can('archive any conversation')) {
            return true;
        }

        if ((int) $conversation->assigned_user_id === (int) $user->id && 
            $user->can('archive own conversations')) {
            return true;
        }

        return false;
    }

    /**
     * ✅ VERIFICAR SE PODE FINALIZAR CONVERSA
     */
    public function finalize(User $user, Conversation $conversation): bool
    {
        if ($user->super_admin || $user->can('finalize any conversation')) {
            return true;
        }

        if ((int) $conversation->assigned_user_id === (int) $user->id && 
            $user->can('finalize own conversations')) {
            return true;
        }

        return false;
    }

    /**
     * ✅ VERIFICAR SE PODE DELETAR CONVERSA
     */
    public function delete(User $user, Conversation $conversation): bool
    {
        if ($user->super_admin || $user->can('delete any conversation')) {
            return true;
        }

        if ((int) $conversation->assigned_user_id === (int) $user->id && 
            $user->can('delete own conversations')) {
            return true;
        }

        if ($conversation->assigned_user_id && $user->can('delete team conversations')) {
            return $user->managesAnyTeamsOf($conversation->assigned_user_id);
        }

        return false;
    }

    /**
     * ✅ VERIFICAR SE PODE ENVIAR MENSAGENS
     */
    public function sendMessage(User $user, Conversation $conversation): bool
    {
        // Primeiro verificar se pode ver a conversa
        if (!$this->view($user, $conversation)) {
            return false;
        }

        // Verificar permissões de envio
        if ($conversation->type === 'internal') {
            return $user->can('send internal messages');
        } else {
            return $user->can('send external messages');
        }
    }
}